/**
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Fabien GIQUEL (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.java.actions;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.logging.AtlLogHandler;
import org.eclipse.gmt.modisco.common.core.utils.UriUtils;
import org.eclipse.gmt.modisco.java.JavaActivator;
import org.eclipse.m2m.atl.common.ATLLaunchConstants;
import org.eclipse.m2m.atl.common.ATLLogger;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.core.IExtractor;
import org.eclipse.m2m.atl.core.IInjector;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.ModelFactory;
import org.eclipse.m2m.atl.core.emf.EMFModel;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.core.service.CoreService;
import org.eclipse.m2m.atl.core.ui.vm.asm.ASMModelWrapper;
import org.eclipse.m2m.atl.drivers.emf4atl.ASMEMFModel;
import org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel;

public class TranslateJavaModelToKdm {

	// resources location
	private static final String KDM_MM_URI = org.eclipse.gmt.modisco.omg.kdm.action.ActionPackage.eNS_URI;
	private static final String JAVA_MM_URI = org.eclipse.gmt.modisco.java.emf.JavaPackage.eNS_URI;

	protected static final Logger logger = Logger
			.getLogger(ATLLogger.LOGGER_ID);

	public Resource getKDMModelFromJavaModelWithCustomTransformation(
			final URI javaSourceModelUri, final URI kdmTargetModelUri)
			throws IOException,
			ATLCoreException {
		final URL transformation = TranslateJavaModelToKdm.class
		.getResource("resources/transformations/javaToKdm.asm"); //$NON-NLS-1$
		return this.getKDMModelFromJavaModelWithCustomTransformation(javaSourceModelUri, transformation, kdmTargetModelUri);
	}
		
	public Resource getKDMModelFromJavaModelWithCustomTransformation(
			final URI javaSourceModelUri, final URL transformation, final URI kdmTargetModelUri)
			throws IOException,
			ATLCoreException {
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String javaMetaModelName = "java"; //$NON-NLS-1$
		modelHandlers.put(javaMetaModelName, "EMF"); //$NON-NLS-1$
		String kdmMetaModelName = "kdm"; //$NON-NLS-1$
		modelHandlers.put(kdmMetaModelName, "EMF"); //$NON-NLS-1$

		Map<String, Object> launcherOptions = new HashMap<String, Object>();
		launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER,
				modelHandlers);

		String launcherName = ATLLaunchConstants.EMF_VM_NAME; // REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);

		ModelFactory factory = CoreService.createModelFactory(launcher
				.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory
				.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory
				.getDefaultExtractorName());

		// Loading Java metamodel
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put(
				"modelHandlerName", modelHandlers.get(javaMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", javaMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", JAVA_MM_URI); //$NON-NLS-1$
		IReferenceModel javaMM = factory
				.newReferenceModel(referenceModelOptions);
		injector.inject(javaMM, JAVA_MM_URI);

		// Loading Java model
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(javaSourceModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(javaMM, modelOptions);
		injector.inject(input, UriUtils.toString(javaSourceModelUri));
		launcher.addInModel(input, inModelName, javaMetaModelName);

		// Loading KDM metamodel
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put(
				"modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", KDM_MM_URI); //$NON-NLS-1$
		IReferenceModel kdmMM = factory
				.newReferenceModel(referenceModelOptions);
		injector.inject(kdmMM, KDM_MM_URI);
		// Loading KDM model
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		if (kdmTargetModelUri == null) {
			// default value
			modelOptions.put("path", UriUtils.toString(javaSourceModelUri //$NON-NLS-1$
					.trimFileExtension().appendFileExtension("kdm"))); //$NON-NLS-1$
		}
		modelOptions.put("path", UriUtils.toString(kdmTargetModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(kdmMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, kdmMetaModelName);

		// New Thread for a new empty stack
		final String targetLogFile = kdmTargetModelUri.toFileString() + ".log"; //$NON-NLS-1$
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true"); //$NON-NLS-1$ //$NON-NLS-2$
		options.put("printExecutionTime", "true"); //$NON-NLS-1$ //$NON-NLS-2$

		Job transformationThread = new Job(Messages.TranslateJavaModelToKdm_ConvertJavaToKDM) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				AtlLogHandler logHandler = new AtlLogHandler(targetLogFile);
				TranslateJavaModelToKdm.logger.addHandler(logHandler);

				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					IStatus status = new Status(IStatus.ERROR,
							JavaActivator.PLUGIN_ID, "Unkown error.", e); //$NON-NLS-1$
					JavaActivator.getDefault().getLog().log(status);
				} finally {
					TranslateJavaModelToKdm.logger.removeHandler(logHandler);
					logHandler.close();
				}

				return result;
			}
		};


		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			IStatus status = new Status(IStatus.ERROR, JavaActivator.PLUGIN_ID,
					"Unkown error.", e); //$NON-NLS-1$
			JavaActivator.getDefault().getLog().log(status);
		}

		extractor.extract(outputInstance, UriUtils
					.toString(kdmTargetModelUri));

		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		} else if (outputInstance instanceof EMFModel) {
			output = ((EMFModel) outputInstance).getResource();
		}
		return output;
	}

}
