/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.java.actions;

import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.discoverersmanager.AbstractDiscovererImpl;
import org.eclipse.gmt.modisco.discoverersmanager.Discoverer;
import org.eclipse.gmt.modisco.discoverersmanager.DiscoveryParameter;
import org.eclipse.gmt.modisco.java.ASTNode;
import org.eclipse.gmt.modisco.java.AbstractMethodDeclaration;
import org.eclipse.gmt.modisco.java.CompilationUnit;
import org.eclipse.gmt.modisco.java.FieldDeclaration;
import org.eclipse.gmt.modisco.java.JavaActivator;
import org.eclipse.gmt.modisco.java.Model;
import org.eclipse.gmt.modisco.java.VariableDeclaration;
import org.eclipse.gmt.modisco.java.core.JavaJdtBridge;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.internal.ui.javaeditor.JavaEditor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

/**
 * @author Gabriel Barbier
 *
 */
@SuppressWarnings("restriction")
public class BrowseJavaCode extends AbstractDiscovererImpl implements
		Discoverer {

	/* 
	 * Main idea is to be able to browse corresponding java code 
	 * from an element of java model.
	 * First: retrieve corresponding java project in workspace
	 * Second: get the jdt model
	 * Third: find corresponding jdt element in jdt model
	 * Forth: open editor on selected element
	 * 
	 */
	public void discoverElement(Object source,
			Map<DiscoveryParameter, Object> parameters) {
		ASTNode node = (ASTNode) source;
		this.browseJavaCode(node);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.discoverersmanager.Discoverer#isApplicableTo(java.lang.Object)
	 */
	public boolean isApplicableTo(Object source) {
		boolean result = false;
		/*
		 * A filter of instances from Java model
		 * Here is the four types that don't inherit from ASTNode
		 * 1. Model
		 * 2. Manifest
		 * 3. ManifestAttribute
		 * 4. ManifestEntry
		 */
		if (source instanceof ASTNode) {
			result = true;
			/*
			 * elements that cannot being linked to a java file:
			 * 
			 * Package = ? what if it exists a <Package name>.java which would be here only for javadoc
			 */
		}
		return result;
	}

	public void browseJavaCode(ASTNode node) {
		CompilationUnit compilationUnit = node.getOriginalCompilationUnit();
		if (node instanceof CompilationUnit) {
			compilationUnit = (CompilationUnit) node;
		}
		
		if (compilationUnit != null) {
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			IPath filePath = new Path(compilationUnit.getOriginalFilePath());
			final IFile ifile = root.getFileForLocation(filePath);
			/*
			 * in case of fields or methods, or statements,
			 * we have to find corresponding jdt element (IJavaElement) 
			 */
			final IJavaElement jdtElement = this.findCorrespondingJdtElement(node, compilationUnit);
			if ((ifile != null) && ifile.exists()) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						try {
							IWorkbenchPage page = PlatformUI.getWorkbench()
							.getActiveWorkbenchWindow().getActivePage();
							// programmatically opens the j2se5 model using the
							// default editor if one is present
							IEditorPart result = IDE.openEditor(page, ifile);
							if (jdtElement != null) {
								// TODO discouraged access !!!
								// perhaps do we have to fill a bug ?
								((JavaEditor) result).setSelection(jdtElement);
							}

						} catch (PartInitException e) {
							MoDiscoLogger.logError(e, JavaActivator.getDefault());
						}
					}
				});
			} else {
				MoDiscoLogger.logInfo(Messages.BrowseJavaCode_noFileFound + filePath, JavaActivator.getDefault());
			}
		} else {
			MoDiscoLogger.logInfo(Messages.BrowseJavaCode_notHandledJavaElement, JavaActivator.getDefault());
		}
	}

	/**
	 * @param node
	 * @return
	 */
	private IJavaElement findCorrespondingJdtElement(ASTNode node, CompilationUnit compilationUnit) {
		IJavaElement result = null;
		
		EObject eobject = compilationUnit.eContainer();
		if (eobject instanceof Model) {
			Model model = (Model) eobject;
			/*
			 * first. Get the java project
			 */
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			String projectName = model.getName();
			IProject project = root.getProject(projectName);
			if (project != null) {
				// retrieval of Java project
				IJavaProject tempProject = null;
				if (project instanceof IJavaProject) {
					tempProject = (IJavaProject) project;
				} else {
					try {
						if (project.hasNature(JavaCore.NATURE_ID)) {
							tempProject = JavaCore.create(project);
						}
					} catch (CoreException e) {
						MoDiscoLogger.logError(e, JavaActivator.getDefault());
					}
				}
				final IJavaProject javaProject = tempProject;
				if (node instanceof AbstractMethodDeclaration) {
					AbstractMethodDeclaration operation = (AbstractMethodDeclaration) node;
					JavaJdtBridge bridge = new JavaJdtBridge();
					result = bridge.getJdtOperation(javaProject, operation);
				} else if (node instanceof VariableDeclaration) {
					VariableDeclaration field = (VariableDeclaration) node;
					JavaJdtBridge bridge = new JavaJdtBridge();
					result = bridge.getJdtField(javaProject, field);
				} else if (node instanceof FieldDeclaration) {
					FieldDeclaration fieldDeclaration = (FieldDeclaration) node;
					// retireve the first fragment definition
					VariableDeclaration field = fieldDeclaration.getFragments().get(0);
					JavaJdtBridge bridge = new JavaJdtBridge();
					result = bridge.getJdtField(javaProject, field);
				}
			} else {
				MoDiscoLogger.logInfo(Messages.BrowseJavaCode_projectNotFound + projectName, JavaActivator.getDefault());
			}
		}

		return result;
	}
}
