/**
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Gregoire Dupe (Mia-Software) - initial API and implementation
 *     Fabien Giquel (Mia-Software) - initial API and implementation
 */
package org.eclipse.gmt.modisco.java.discoverer.benchmark.cdo.client;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.cdo.CDOObject;
import org.eclipse.emf.cdo.common.revision.cache.CDORevisionCache;
import org.eclipse.emf.cdo.common.revision.cache.CDORevisionCacheUtil;
import org.eclipse.emf.cdo.eresource.CDOResource;
import org.eclipse.emf.cdo.net4j.CDONet4jUtil;
import org.eclipse.emf.cdo.net4j.CDOSessionConfiguration;
import org.eclipse.emf.cdo.session.CDOSession;
import org.eclipse.emf.cdo.transaction.CDOTransaction;
import org.eclipse.emf.cdo.view.CDOView;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.java.discoverer.benchmark.Activator;
import org.eclipse.gmt.modisco.java.discoverer.tests.cdo.server.AbstractCDOServer;
import org.eclipse.gmt.modisco.java.discoverer.tests.emfstat.DiscoverJavaModelFromJavaProjectEMFStat;
import org.eclipse.gmt.modisco.java.discoverer.tests.emfstat.JavaReaderEMFStat;
import org.eclipse.gmt.modisco.java.emf.JavaFactory;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.net4j.Net4jUtil;
import org.eclipse.net4j.tcp.ITCPConnector;
import org.eclipse.net4j.tcp.TCPUtil;
import org.eclipse.net4j.util.container.ContainerUtil;
import org.eclipse.net4j.util.container.IManagedContainer;
import org.eclipse.net4j.util.om.OMPlatform;
import org.eclipse.net4j.util.om.log.PrintLogHandler;
import org.eclipse.net4j.util.om.trace.PrintTraceHandler;
import org.eclipse.net4j.util.transaction.TransactionException;

public class JavaDiscovererCDO extends DiscoverJavaModelFromJavaProjectEMFStat {

	private String cacheType = LRU_CACHE;
	private Object serverDescription = Messages.JavaDiscovererCDO_0;
	private int port = JavaDiscovererCDO.DEFAULT_PORT_VALUE;
	public static final int DEFAULT_LRU_CAPACITY = 1000;
	public static final int DEFAULT_PORT_VALUE = 2036;
	public static final String LRU_CACHE = Messages.JavaDiscovererCDO_1;
	public static final String DEFAULT_CACHE = Messages.JavaDiscovererCDO_2;
	public static final String MEM_CACHE = Messages.JavaDiscovererCDO_3;
	public static final String TWO_LEVEL_CACHE = Messages.JavaDiscovererCDO_4;
	public static final String CDO_CACHE_TYPE = Messages.JavaDiscovererCDO_5;
	public static final String CURRENT_LRU_CAPACITY = Messages.JavaDiscovererCDO_6;
	public static final String REVISED_LRU_CAPACITY = Messages.JavaDiscovererCDO_7;
	public static final String CDO_VERSION = Messages.JavaDiscovererCDO_8;
	public static final String CDO_SERVER_DESCRIPTION = Messages.JavaDiscovererCDO_9;
	public static final int LRU_CAPACITY_CURRENT = 100000;
	public static final int LRU_CAPACITY_REVISED = 100;

	private static Map<CDOView, ITCPConnector> connectors = new HashMap<CDOView, ITCPConnector>();

	@Override
	protected void terminate(final Resource r) {
		close(r, true);
	}

	private static void close(final Resource r, final boolean commit) {
		if (r instanceof CDOResource) {
			CDOResource cdoRessource = (CDOResource) r;
			CDOView cdoView = cdoRessource.cdoView();
			if (commit) {
				if (cdoView instanceof CDOTransaction) {
					CDOTransaction cdoTransaction = (CDOTransaction) cdoView;
					try {
						cdoTransaction.commit();
					} catch (TransactionException e) {
						e.printStackTrace();
					}
				}
			}
			CDOSession cdoSession = cdoView.getSession();
			ITCPConnector connector = JavaDiscovererCDO.connectors.get(cdoView);
			cdoView.close();
			if (cdoSession != null) {
				cdoSession.close();
			}
			if (connector != null) {
				connector.close();
			}
			JavaDiscovererCDO.connectors.remove(cdoView);
		}
	}

	@Override
	protected Resource getResource(final String resourceName,
			final IJavaProject javaProject) {
		String resourceName2 = resourceName + System.currentTimeMillis();
		IStatus status = new Status(IStatus.INFO, Activator.PLUGIN_ID,
				Messages.JavaDiscovererCDO_10 + resourceName2);
		Activator.getDefault().getLog().log(status);
		EPackage ePackage = this.getEFactory().getEPackage();
		if (ePackage == null) {
			throw new RuntimeException(Messages.JavaDiscovererCDO_11
					+ this.getEFactory());
		}
		CDOResource result = getOrCreateCDOResource(resourceName2, ePackage);
		result.getContents().add(this.getEFactory().createModel());
		dbInit(result);
		return result;
	}

	private CDOResource getOrCreateCDOResource(final String resourceName,
			final EPackage ePackage) {
		CDOResource resource = null;
		// Enable logging and tracing
		OMPlatform.INSTANCE.setDebugging(true);
		OMPlatform.INSTANCE.addLogHandler(PrintLogHandler.CONSOLE);
		OMPlatform.INSTANCE.addTraceHandler(PrintTraceHandler.CONSOLE);

		// Prepare container
		IManagedContainer container = ContainerUtil.createContainer();
		// Register Net4j factories
		Net4jUtil.prepareContainer(container);
		// Register TCP factories
		TCPUtil.prepareContainer(container);
		// Register CDO factories
		CDONet4jUtil.prepareContainer(container);
		container.activate();
		// Create connector
		ITCPConnector connector = TCPUtil.getConnector(container,
				AbstractCDOServer.HOSTNAME + Messages.JavaDiscovererCDO_12 + this.port);
		// Create configuration
		CDORevisionCache revisionCache = null;
		if (this.cacheType.equals(JavaDiscovererCDO.MEM_CACHE)) {
			revisionCache = CDORevisionCacheUtil.createMEMCache();
		} else if (this.cacheType.equals(JavaDiscovererCDO.LRU_CACHE)) {
			revisionCache = CDORevisionCacheUtil.createLRUCache(
					JavaDiscovererCDO.DEFAULT_LRU_CAPACITY,
					JavaDiscovererCDO.DEFAULT_LRU_CAPACITY);
		} else if (this.cacheType.equals(JavaDiscovererCDO.TWO_LEVEL_CACHE)) {
			revisionCache = CDORevisionCacheUtil.createTwoLevelCache(
					CDORevisionCacheUtil.createLRUCache(
							JavaDiscovererCDO.LRU_CAPACITY_CURRENT,
							JavaDiscovererCDO.LRU_CAPACITY_REVISED),
					CDORevisionCacheUtil.createMEMCache());
		}

		CDOSessionConfiguration configuration = CDONet4jUtil
				.createSessionConfiguration();
		configuration.setConnector(connector);

		configuration.setRepositoryName(AbstractCDOServer.REPOSITORY_NAME);
		if (revisionCache != null) {
			configuration.setRevisionCache(revisionCache);
		}

		// Open session
		CDOSession session = configuration.openSession();
		session.getPackageRegistry().putEPackage(ePackage);

		// Open transaction
		CDOTransaction transaction = session.openTransaction();
		// Get or create resource
		JavaDiscovererCDO.connectors.put(transaction, connector);
		String resourcePath = Messages.JavaDiscovererCDO_13 + resourceName;
		IStatus status = new Status(IStatus.INFO, Activator.PLUGIN_ID,
				resourcePath);
		Activator.getDefault().getLog().log(status);
		resource = transaction.getOrCreateResource(resourcePath);
		return resource;
	}

	@Override
	protected Properties getProperties() {
		Properties properties = super.getProperties();
		properties.put(DiscoverJavaModelFromJavaProjectEMFStat.STORAGE_KIND,
				"CDO"); //$NON-NLS-1$
		properties.put(JavaDiscovererCDO.CDO_VERSION, Platform.getBundle(
				"org.eclipse.emf.cdo").getVersion().toString()); //$NON-NLS-1$
		if (JavaDiscovererCDO.LRU_CACHE.equals(this.cacheType)) {
			properties.put(JavaDiscovererCDO.CDO_CACHE_TYPE,
					JavaDiscovererCDO.LRU_CACHE);
			properties.put(JavaDiscovererCDO.CURRENT_LRU_CAPACITY, new Integer(
					JavaDiscovererCDO.DEFAULT_LRU_CAPACITY).toString());
			properties.put(JavaDiscovererCDO.REVISED_LRU_CAPACITY, new Integer(
					JavaDiscovererCDO.DEFAULT_LRU_CAPACITY).toString());
		} else if (JavaDiscovererCDO.DEFAULT_CACHE.equals(this.cacheType)) {
			properties.put(JavaDiscovererCDO.CDO_CACHE_TYPE,
					JavaDiscovererCDO.DEFAULT_CACHE);
		} else if (JavaDiscovererCDO.MEM_CACHE.equals(this.cacheType)) {
			properties.put(JavaDiscovererCDO.CDO_CACHE_TYPE,
					JavaDiscovererCDO.MEM_CACHE);
		} else {
			properties.put(JavaDiscovererCDO.CDO_CACHE_TYPE,
					"Wrong revision cache type"); //$NON-NLS-1$
			IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
					"Wrong revision cache type", new Exception()); //$NON-NLS-1$
			Activator.getDefault().getLog().log(status);
		}
		properties.put(JavaDiscovererCDO.CDO_SERVER_DESCRIPTION,
				this.serverDescription);
		properties.put(ALGO_VARIANTE, Messages.JavaDiscovererCDO_18);
		return properties;
	}

	@Override
	protected JavaFactory getEFactory() {
		org.eclipse.gmt.modisco.java.cdo.impl.JavaPackageImpl.init();
		return org.eclipse.gmt.modisco.java.cdo.meta.JavaFactory.eINSTANCE;
	}

	private void dbInit(final CDOResource resource) {
		if (this.statistics != null) {
			this.statistics.beginInit();
		}
		CDOView cdoView = resource.cdoView();
		if (cdoView instanceof CDOTransaction) {
			CDOTransaction cdoTransaction = (CDOTransaction) cdoView;
			cdoTransaction.commit();
		}
		if (this.statistics != null) {
			this.statistics.endInit();
		}
	}

	@Override
	protected JavaReaderEMFStat getJavaReader(final boolean isIncremental) {
		JavaReaderEMFStat javaReader = new JavaReaderEMFStat(getEFactory(),
				true) {
			@Override
			protected void reset() {
				if (this.statistics != null) {
					this.statistics.tabValueReport("Begin reset"); //$NON-NLS-1$
				}
				this.globalBindings.resolveBindings(this.resultModel);
				if (this.statistics != null) {
					this.statistics.tabValueReport("globalBindings resolved"); //$NON-NLS-1$
				}
				if (this.statistics != null) {
					this.statistics.beginSave();
				}
				if (this.resultModel instanceof CDOObject) {
					CDOObject cdoObject = (CDOObject) this.resultModel;
					CDOView cdoView = cdoObject.cdoView();
					if (cdoView instanceof CDOTransaction) {
						CDOTransaction cdoTransaction = (CDOTransaction) cdoView;
						cdoTransaction.commit();
					}
				}
				this.globalBindings = getBindingManager();
				if (this.statistics != null) {
					this.statistics.endSave();
				}
				if (this.statistics != null) {
					this.statistics.tabValueReport("End reset"); //$NON-NLS-1$
				}
			}
		};
		javaReader.setStatistics(this.statistics);
		return javaReader;
	}

	@Override
	protected void saveResource(final URI target, final IPath path,
			final Resource resource, final IProgressMonitor monitor)
			throws IOException {
		monitor.subTask(Messages.JavaDiscovererCDO_22);
		resource.save(new HashMap<Object, Object>());
	}

	public void setServerDescription(final String serverDescription) {
		this.serverDescription = serverDescription;
	}

	public void setPort(final int port) {
		this.port = port;
	}

	public int getPort() {
		return this.port;
	}

}
