/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Grgoire Dup (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.role.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmt.modisco.infra.query.ModelQuery;
import org.eclipse.gmt.modisco.infra.query.core.AbstractModelQuery;
import org.eclipse.gmt.modisco.infra.query.core.Activator;
import org.eclipse.gmt.modisco.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.gmt.modisco.infra.query.core.exception.ModelQueryException;
import org.eclipse.gmt.modisco.infra.role.Role;
import org.eclipse.gmt.modisco.infra.role.RoleAttribute;
import org.eclipse.gmt.modisco.infra.role.RoleReference;
import org.eclipse.gmt.modisco.infra.role.RoleSet;
import org.eclipse.gmt.modisco.infra.role.RoleStructuralFeature;
import org.eclipse.gmt.modisco.infra.role.Shortcut;
import org.eclipse.gmt.modisco.infra.role.core.exception.ModiscoRoleException;
import org.eclipse.gmt.modisco.infra.role.core.exception.StructuralConstraintViolationException;

/**
 * @author Gregoire Dupe
 * 
 */
public class RoleContext {

	private HashSet<Role> appliedRoles = new HashSet<Role>();
	private List<EStructuralFeature> appliedFeatures = new ArrayList<EStructuralFeature>();
	private HashSet<Resource> resources = new HashSet<Resource>();

	public void addComposedResource(final Resource resource) {
		this.resources.add(resource);
		TreeIterator<EObject> allContents = EcoreUtil.getAllContents(resource,
				false);
		while (allContents.hasNext()) {
			EObject element = allContents.next();
			if (element.eResource() != resource) {
				this.resources.add(element.eResource());
			}
		}
	}

	/**
	 * @param roleSet
	 */
	public void addRoleSet(final RoleSet roleSet) {
		// this.appliedRoleSet.add(roleSet);
		for (Role role : roleSet.getRoles()) {
			this.addRole(role);
		}
	}

	/**
	 * @param role
	 */
	public void addRole(final Role role) {
		if (!this.appliedRoles.contains(role)) {
			this.appliedRoles.add(role);
			for (EStructuralFeature feature : role.getEStructuralFeatures()) {
				if (feature instanceof RoleStructuralFeature
						|| feature instanceof Shortcut) {
					this.appliedFeatures.add(feature);
				}
			}
		}
	}

	/**
	 * @param ecorePackage
	 * @param role
	 * @return
	 * @throws ModelQueryException
	 */
	public boolean isInstance(final EObject eObject, final Role role)
			throws ModelQueryException {
		ModelQuery modelQuery = role.getConditionQuery();
		boolean result = false;
		boolean isSuperType = false;
		for (EClass superType : role.getESuperTypes()) {
			if (superType.isSuperTypeOf(eObject.eClass())) {
				isSuperType = true;
				break;
			}
		}
		if (isSuperType) {
			if (modelQuery == null) {
				result = true;
			} else {
				AbstractModelQuery abstractModelQuery = ModelQuerySetCatalog
						.getSingleton().getModelQueryImpl(modelQuery);
				result = (Boolean) abstractModelQuery.basicEvaluate(eObject);
			}
		}
		return result;
	}

	/**
	 * @param structuralFeature
	 * @return
	 * @throws ModelQueryException
	 * @throws ModiscoRoleException
	 * @throws StructuralConstraintViolationException
	 */
	public Object get(final EObject eObject,
			final EStructuralFeature structuralFeature)
			throws ModelQueryException, ModiscoRoleException {
		if (structuralFeature == null) {
			throw new ModiscoRoleException(
					"The parameter named structuralFeature is null"); //$NON-NLS-1$
		}
		Object result = null;
		if (this.appliedFeatures.contains(structuralFeature)) {
			Role role = (Role) structuralFeature.eContainer();
			if (this.isInstance(eObject, role)) {
				if (structuralFeature instanceof RoleStructuralFeature) {
					RoleStructuralFeature roleStructuralFeature = (RoleStructuralFeature) structuralFeature;
					ModelQuery modelQuery = roleStructuralFeature
							.getValueQuery();
					AbstractModelQuery abstractModelQuery = ModelQuerySetCatalog
							.getSingleton().getModelQueryImpl(modelQuery);
					result = abstractModelQuery.basicEvaluate(eObject);
				} else if (structuralFeature instanceof Shortcut) {
					Shortcut shortcut = (Shortcut) structuralFeature;
					List<EObject> resultList = new BasicEList<EObject>();
					EReference oppositeRef = shortcut
							.getOppositeReference();
					EClass eClass = oppositeRef.getEContainingClass();
					List<EObject> list = this.allInstances(eClass);
					for (EObject modelElement : list) {
						EObject referedByModelElement = (EObject) modelElement
								.eGet(oppositeRef);
						if (referedByModelElement == eObject) {
							EObject element = modelElement;
							for (EReference eReference : shortcut.getPath()) {
								element = (EObject) element.eGet(eReference);
							}
							resultList.add(element);
						}
					}
					if (shortcut.getUpperBound() > 1) {
						result = resultList;
					} else {
						if (resultList.size() > 1) {
							throw new StructuralConstraintViolationException();
						} else if (resultList.size() == 1) {
							result = resultList.get(0);
						}
						// else result = null;
					}
				} else {
					throw new ModiscoRoleException(
							"The structuralFeature parameter must be a RoleStructuralFeature: " //$NON-NLS-1$
									+ structuralFeature.toString());
				}
			} else {
				throw new ModiscoRoleException(
						"The eObject parameter must be a instance of : " //$NON-NLS-1$
								+ role.getName());
			}
		} else {
			throw new ModiscoRoleException(
					"The structuralFeature must be registered into the context, please use the " //$NON-NLS-1$
							+ RoleContext.class.getSimpleName()
							+ ".add(RoleSet) method."); //$NON-NLS-1$
		}
		return result;
	}

	/**
	 * @param eClass
	 */
	protected List<EObject> allInstances(final EClass eClass) {
		List<EObject> result = new ArrayList<EObject>();
		for (Resource resource : this.resources) {
			TreeIterator<EObject> treeIterator = resource.getAllContents();
			while (treeIterator.hasNext()) {
				EObject eObject = treeIterator.next();
				if (eClass.isInstance(eObject)) {
					result.add(eObject);
				}
			}
		}
		return result;
	}

	/**
	 * @param eObject
	 * @return
	 */
	public List<EStructuralFeature> getRoleFeatures(final EObject eObject) {
		List<EStructuralFeature> result = new ArrayList<EStructuralFeature>();
		for (Role role : this.appliedRoles) {
			try {
				if (isInstance(eObject, role)) {
					result.addAll(role.getEStructuralFeatures());
				}
			} catch (ModelQueryException e) {
				IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
						e.getMessage(), e);
				Activator.getDefault().getLog().log(status);
			}
		}
		return result;
	}

	/**
	 * @param allRoleSets
	 */
	public void addAll(final Collection<RoleSet> allRoleSets) {
		for (RoleSet roleSet : allRoleSets) {
			this.addRoleSet(roleSet);
		}
	}

	/**
	 * @param eObject
	 * @return
	 * @throws ModelQueryException
	 */
	public EList<EAttribute> getAttributes(final EObject eObject)
			throws ModelQueryException {
		EList<EAttribute> result = new BasicEList<EAttribute>();
		for (Role role : this.appliedRoles) {
			if (this.isInstance(eObject, role)) {
				for (EStructuralFeature structuralFeature : role
						.getEStructuralFeatures()) {
					if (structuralFeature instanceof RoleAttribute) {
						RoleAttribute roleAttribute = (RoleAttribute) structuralFeature;
						result.add(roleAttribute);
					}
				}
			}
		}
		return result;
	}

	/**
	 * @param eObject
	 * @return
	 * @throws ModelQueryException
	 */
	public EList<EReference> getReferences(final EObject eObject)
			throws ModelQueryException {
		EList<EReference> result = new BasicEList<EReference>();
		for (Role role : this.appliedRoles) {
			if (this.isInstance(eObject, role)) {
				for (EStructuralFeature structuralFeature : role
						.getEStructuralFeatures()) {
					if (structuralFeature instanceof RoleReference
							|| structuralFeature instanceof Shortcut) {
						EReference roleAttribute = (EReference) structuralFeature;
						result.add(roleAttribute);
					}
				}
			}
		}
		return result;
	}

	/**
	 * @return
	 */
	public List<EClass> getMetaClasses() {
		List<EClass> metaClasses = new ArrayList<EClass>();
		metaClasses.addAll(this.appliedRoles);
		return metaClasses;
	}

	/**
	 * @return
	 * @throws ModelQueryException
	 */
	public List<Role> getRoles(final EObject eObject)
			throws ModelQueryException {
		List<Role> result = new ArrayList<Role>();
		for (Role role : this.appliedRoles) {
			try {
				if (this.isInstance(eObject, role)) {
					result.add(role);
				}
			} catch (Exception e) {
				IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
						e.getMessage(), e);
				Activator.getDefault().getLog().log(status);
			}
		}
		return result;
	}

	/**
	 * @return
	 */
	public HashSet<Role> getAppliedRoles() {
		return this.appliedRoles;
	}

	/**
	 * 
	 */
	public void clean() {
		this.appliedFeatures.clear();
		this.appliedRoles.clear();
	}

	protected List<EStructuralFeature> getAppliedFeatures() {
		return this.appliedFeatures;
	}
}
