/**
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Gregoire DUPE (Mia-Software) - initial API and implementation
 */
package org.eclipse.gmt.modisco.infra.role.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.gmt.modisco.common.core.builder.EcoreCatalog;
import org.eclipse.gmt.modisco.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.gmt.modisco.infra.role.RoleSet;
import org.eclipse.gmt.modisco.infra.role.core.exception.ModiscoRoleException;

public final class RoleSetCatalog extends EcoreCatalog {

	private static final String REGISTRATION_EXTENSION_POINT_ID = "org.eclipse.gmt.modisco.infra.role.registration"; //$NON-NLS-1$

	private static RoleSetCatalog singleton = null;

	public static RoleSetCatalog getSingleton() {
		if (RoleSetCatalog.singleton == null) {
			EcoreCatalog.getSingleton();
			ModelQuerySetCatalog.getSingleton();
			RoleSetCatalog.singleton = new RoleSetCatalog();
		}
		return RoleSetCatalog.singleton;
	}

	private HashMap<EPackage, RoleSet> ePackageToDefaultRoleSetMap;

	public HashMap<EPackage, RoleSet> getEPackageToDefaultRoleSetMap() {
		if (this.ePackageToDefaultRoleSetMap == null) {
			this.ePackageToDefaultRoleSetMap = new HashMap<EPackage, RoleSet>();
		}
		return this.ePackageToDefaultRoleSetMap;
	}

	private RoleSetCatalog() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRootObjectName(final EObject rootObject) {
		return ((RoleSet) rootObject).getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRootObjectNsUri(final EObject rootObject) {
		return ((RoleSet) rootObject).getNsURI();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Class<?> getRootClass() {
		return RoleSet.class;
	}

	/**
	 * This method returns the role set of the resource having for name the
	 * "name" parameter value.
	 * 
	 * @param name
	 *            the name of a resource contained in the catalog
	 * @return return a role set
	 */
	public RoleSet getRoleSet(final String name) {
		return (RoleSet) this.getRootObject(name);
	}

	public RoleSet getDefaultRoleSet(final EPackage ePackage) {
		return this.getEPackageToDefaultRoleSetMap().get(ePackage);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRegistryExtensionPoint() {
		return RoleSetCatalog.REGISTRATION_EXTENSION_POINT_ID;
	}

	/**
	 * This methods returns the role sets of all the resources contained in the
	 * catalog.
	 * 
	 * @return returns all the role sets
	 */
	public Collection<RoleSet> getAllRoleSets() {
		Collection<RoleSet> result = new ArrayList<RoleSet>();
		for (EObject eObject : this.getAllRootObjects()) {
			if (eObject instanceof RoleSet) {
				RoleSet roleSet = (RoleSet) eObject;
				result.add(roleSet);
			}
		}
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getMoDiscoSubProtocol() {
		return "role"; //$NON-NLS-1$
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws ModiscoRoleException
	 */
	@Override
	protected EObject openResource(final URI uri, final IResource file) {
		EObject result = super.openResource(uri, file);
		if (result instanceof RoleSet) {
			RoleSet roleSet = (RoleSet) result;
			EAnnotation annotation = roleSet.getEAnnotation("default");//$NON-NLS-1$
			if (annotation != null) {
				if (annotation.getReferences().size() == 1
						&& annotation.getReferences().get(0) instanceof EPackage) {
					EPackage ePackage = (EPackage) annotation.getReferences()
							.get(0);
					this.getEPackageToDefaultRoleSetMap()
							.put(ePackage, roleSet);
				}

			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.gmt.modisco.common.core.builder.EcoreCatalog#getActivator()
	 */
	@Override
	protected Plugin getActivator() {
		return Activator.getDefault();
	}
}