/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    Grgoire Dup (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.facet.ui.views;

import java.net.URI;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.ui.URIEditorInput;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.gmt.modisco.common.core.builder.AbstractMoDiscoCatalog.ModiscoCatalogChangeListener;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.common.core.utils.ImageUtils;
import org.eclipse.gmt.modisco.infra.facet.Facet;
import org.eclipse.gmt.modisco.infra.facet.FacetPackage;
import org.eclipse.gmt.modisco.infra.facet.FacetSet;
import org.eclipse.gmt.modisco.infra.facet.core.FacetSetCatalog;
import org.eclipse.gmt.modisco.infra.facet.ui.Activator;
import org.eclipse.gmt.modisco.infra.facet.ui.Messages;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

/**
 * An Eclipse view that displays a list of available MoDisco facet sets. Clicking
 * on a facet set opens it in the facet set editor.
 */
public class FacetView extends ViewPart implements ModiscoCatalogChangeListener {

	private static final int JOB_SCHEDULE_DELAY = 500;
	private static final int COLUMN_WIDTH = 200;
	private TreeViewer facetSetViewer;
	private TreeViewerColumn nameColumn;
	private TreeViewerColumn locationColumn;
	private Job refreshJob = null;
	private TreeViewerColumn defaultColumn;
	private TreeViewerColumn extendsColumn;

	public TreeViewer getFacetSetViewer() {
		return this.facetSetViewer;
	}

	public TreeViewerColumn getNameColumn() {
		return this.nameColumn;
	}

	@Override
	public void createPartControl(final Composite parent) {

		parent.setLayout(new FillLayout());

		this.facetSetViewer = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE
				| SWT.FULL_SELECTION);
		this.facetSetViewer.getTree().setLinesVisible(true);
		this.facetSetViewer.getTree().setHeaderVisible(true);

		this.facetSetViewer.addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				IStructuredSelection selection = (IStructuredSelection) FacetView.this
						.getFacetSetViewer().getSelection();

				EObject modelElement = (EObject) selection.getFirstElement();
				EPackage facetSet = null;
				if (modelElement instanceof FacetSet) {
					facetSet = (FacetSet) modelElement;
				} else if (modelElement instanceof Facet) {
					Facet facet = (Facet) modelElement;
					facetSet = facet.getEPackage();
				}
				if (facetSet != null) {
					// open facet set in editor
					URI uri = null;
					try {
						String uriString = FacetSetCatalog.getSingleton().getURI(facetSet.getName())
								.toString();
						uri = URI.create(uriString);
						if ("file".equals(uri.getScheme())) { //$NON-NLS-1$
							IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
									.getActivePage(), new URI(facetSet.eResource().getURI()
									.toString()), Activator.FACET_EDITOR_ID, true);
						} else {
							URIEditorInput uriEditorInput = new URIEditorInput(
									org.eclipse.emf.common.util.URI.createURI(uri.toString()));
							IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
									.getActivePage(), uriEditorInput, Activator.FACET_EDITOR_ID,
									true);
						}

					} catch (Exception e) {
						MoDiscoLogger.logError(e, "Failed to open: " + uri, Activator.getDefault()); //$NON-NLS-1$
					}
				}
			}
		});

		createNameColumn();
		createExtendsColumn();
		createLocationColumn();

		createDefaultColumn();

		this.facetSetViewer.setContentProvider(new FacetSetContentProvider(FacetPackage.eINSTANCE
				.getFacetStructuralFeature()));

		getSite().setSelectionProvider(this.facetSetViewer);
		refresh(false);

		FacetSetCatalog.getSingleton().addChangeListener(this);
	}

	private void createLocationColumn() {
		this.locationColumn = new TreeViewerColumn(this.facetSetViewer, SWT.NONE);
		this.locationColumn.getColumn().setText(Messages.FacetView_location);
		this.locationColumn.getColumn().setWidth(FacetView.COLUMN_WIDTH * 2);
		this.locationColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (!(element instanceof FacetSet)) {
					return ""; //$NON-NLS-1$
				}
				FacetSet facetSet = (FacetSet) element;
				return FacetSetCatalog.getSingleton().getURI(facetSet.getName()).toString();
			}
		});
		addSorter(this.locationColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				FacetSet facetSet1 = (FacetSet) e1;
				FacetSet facetSet2 = (FacetSet) e2;
				String uri1 = FacetSetCatalog.getSingleton().getURI(facetSet1.getName()).toString();
				String uri2 = FacetSetCatalog.getSingleton().getURI(facetSet2.getName()).toString();
				int compare = uri1.compareTo(uri2);
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createNameColumn() {
		this.nameColumn = new TreeViewerColumn(this.facetSetViewer, SWT.NONE);
		this.nameColumn.getColumn().setText(Messages.FacetView_name);
		this.nameColumn.getColumn().setWidth(FacetView.COLUMN_WIDTH);
		this.nameColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof FacetSet) {
					FacetSet facetSet = (FacetSet) element;
					return facetSet.getName();
				} else if (element instanceof Facet) {
					Facet facet = (Facet) element;
					return facet.getName();
				} else if (element instanceof EStructuralFeature) {
					EStructuralFeature feature = (EStructuralFeature) element;
					return feature.getName();
				} else {
					return ""; //$NON-NLS-1$
				}

			}

			@Override
			public Image getImage(final Object element) {
				return ImageUtils.getImage(element);
			}
		});

		addSorter(this.nameColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				FacetSet facetSet1 = (FacetSet) e1;
				FacetSet facetSet2 = (FacetSet) e2;
				int compare = facetSet1.getName().compareTo(facetSet2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createDefaultColumn() {
		this.defaultColumn = new TreeViewerColumn(this.facetSetViewer, SWT.NONE);
		this.defaultColumn.getColumn().setText(Messages.FacetView_1);
		this.defaultColumn.getColumn().setWidth(FacetView.COLUMN_WIDTH);
		this.defaultColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof FacetSet) {
					FacetSet facetSet = (FacetSet) element;
					EAnnotation annotation = facetSet.getEAnnotation("default");//$NON-NLS-1$
					if (annotation != null) {
						if (annotation.getReferences().size() == 1
								&& annotation.getReferences().get(0) instanceof EPackage) {
							EPackage ePackage = (EPackage) annotation.getReferences().get(0);
							return ePackage.getName() + " " + ePackage.getNsURI(); //$NON-NLS-1$
						}

					}
				}
				return ""; //$NON-NLS-1$
			}

		});

		addSorter(this.defaultColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				FacetSet facetSet1 = (FacetSet) e1;
				FacetSet facetSet2 = (FacetSet) e2;
				int compare = facetSet1.getName().compareTo(facetSet2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void createExtendsColumn() {
		this.extendsColumn = new TreeViewerColumn(this.facetSetViewer, SWT.NONE);
		this.extendsColumn.getColumn().setText(Messages.FacetView_3);
		this.extendsColumn.getColumn().setWidth(FacetView.COLUMN_WIDTH);
		this.extendsColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof FacetSet) {
					FacetSet facetSet = (FacetSet) element;
					return facetSet.getExtendedPackage().getName() + " " //$NON-NLS-1$
							+ facetSet.getExtendedPackage().getNsURI();
				}
				return ""; //$NON-NLS-1$
			}

		});

		addSorter(this.extendsColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				FacetSet facetSet1 = (FacetSet) e1;
				FacetSet facetSet2 = (FacetSet) e2;
				int compare = facetSet1.getName().compareTo(facetSet2.getName());
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void addSorter(final TreeViewerColumn column, final ViewerSorter viewerSorter) {
		column.getColumn().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				TreeViewer treeView = FacetView.this.getFacetSetViewer();
				treeView.setSorter(viewerSorter);
				treeView.getTree().setSortColumn(column.getColumn());
				if (treeView.getTree().getSortDirection() == SWT.UP) {
					treeView.getTree().setSortDirection(SWT.DOWN);
				} else if (treeView.getTree().getSortDirection() == SWT.DOWN) {
					treeView.getTree().setSortDirection(SWT.UP);
				} else {
					treeView.getTree().setSortDirection(SWT.UP);
				}
				treeView.refresh();
			}
		});

	}

	@Override
	public void setFocus() {
		this.facetSetViewer.getControl().setFocus();
	}

	@Override
	public void dispose() {
		super.dispose();
		FacetSetCatalog.getSingleton().removeChangeListener(this);
	}

	/** Called when a facet set changes */
	public void changed(final EObject changedFacetSet, final IFile file) {
		refresh(true);
	}

	/** Called when a facet set is added */
	public void added(final EObject addedFacetSet, final IFile file) {
		refresh(true);
	}

	/** Called when a facet set is removed */
	public void removed(final IFile facetSetFile) {
		refresh(true);
	}

	/** Optionally delayed refresh */
	private void refresh(final boolean delayed) {
		if (this.refreshJob == null) {
			this.refreshJob = new Job(Messages.FacetView_JobName_refreshingFacetSetsView) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							FacetView.this.getFacetSetViewer().setInput(
									FacetSetCatalog.getSingleton().getAllFacetSets().toArray());
							FacetView.this.getNameColumn().getColumn().pack();
							FacetView.this.getFacetSetViewer().refresh();
						}
					});
					return Status.OK_STATUS;
				}
			};
		}
		if (delayed) {
			// delayed until it stops changing
			this.refreshJob.cancel();
			this.refreshJob.setPriority(Job.DECORATE);
			this.refreshJob.schedule(FacetView.JOB_SCHEDULE_DELAY);
		} else {
			// force synchronous execution
			this.refreshJob.schedule();
			try {
				this.refreshJob.join();
			} catch (InterruptedException e) {
				MoDiscoLogger.logError(e, Activator.getDefault());
			}
		}
	}

}
