/*******************************************************************************
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.facet.core.adapters.instances;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.adapters.instances.MetaclassInstancesAdapter;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.infra.facet.Facet;
import org.eclipse.gmt.modisco.infra.facet.core.Activator;
import org.eclipse.gmt.modisco.infra.facet.core.FacetContext;
import org.eclipse.gmt.modisco.infra.query.core.exception.ModelQueryException;

/**
 * An adapter that acts as a cache for lists of instances by EClass (extending
 * {@link MetaclassInstancesAdapter} for supporting {@link Facet}s). It works by
 * listening to changes in the model and updating the cache accordingly.
 * <p>
 * It must be created on a {@link Resource} through
 * {@link MetaclassInstancesAdapterFactoryWithFacet}, by doing the following:<p>
 * <code>
 * MetaclassInstancesAdapterFactoryWithFacet.getInstance().setFacetContext(facetContext);<br>
 * MetaclassInstances instances = (MetaclassInstances) MetaclassInstancesAdapterFactoryWithFacet.getInstance().adapt(resource, MetaclassInstances.class);
 * </code>
 */
public class MetaclassInstancesAdapterWithFacet extends MetaclassInstancesAdapter {

	private final FacetContext facetContext;

	protected MetaclassInstancesAdapterWithFacet(final Resource resource,
			final FacetContext facetContext) {
		super(resource);
		this.facetContext = facetContext;
	}

	/**
	 * Normally, an {@link EObject} only has one {@link EClass}, but we override
	 * this to handle {@link Facet}s.
	 */
	@Override
	protected List<EClass> getEClasses(final EObject element) {
		final List<EClass> eClasses = new ArrayList<EClass>();
		eClasses.add(element.eClass());

		try {
			eClasses.addAll(this.facetContext.getFacets(element));
		} catch (final ModelQueryException e) {
			MoDiscoLogger.logError(e, Activator.getDefault());
		}

		return eClasses;
	}
}
