/*******************************************************************************
 * Copyright (c) 2008, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.TreeMap;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.editors.BrowserConfiguration;
import org.eclipse.gmt.modisco.infra.browser.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.facet.Facet;
import org.eclipse.gmt.modisco.infra.query.core.exception.ModelQueryException;

/**
 * A list of metaclasses from the model, each metaclass having an associated
 * list of instances of the metaclass.
 */
public class InstancesForMetaclasses {

	/**
	 * Model elements sorted by EClass (full qualified name). TreeMap (red-black
	 * tree), automatically sorted by String.
	 */
	private final TreeMap<String, InstancesForMetaclass> modelElements = new TreeMap<String, InstancesForMetaclass>();

	/**
	 * A list of all metaclasses which do not have a parent. That is, a list of
	 * metaclasses from which all other metaclasses can be found by derivation.
	 */
	private final ArrayList<InstancesForMetaclass> rootMetaclasses = new ArrayList<InstancesForMetaclass>();

	private final BrowserConfiguration browserConfiguration;

	public InstancesForMetaclasses(final BrowserConfiguration browserConfiguration) {
		this.browserConfiguration = browserConfiguration;
	}

	public InstancesForMetaclass[] getInstancesForMetaclasses() {
		final Collection<InstancesForMetaclass> instancesForMetaclasses = this.modelElements
				.values();
		return instancesForMetaclasses.toArray(new InstancesForMetaclass[instancesForMetaclasses
				.size()]);
	}

	/**
	 * @param metaclassQualifiedName
	 *            the full qualified name of the metaclass
	 * @return the {@link InstancesForMetaclass} object or <code>null</code> if
	 *         the metaclass was not found
	 */
	public InstancesForMetaclass getInstancesForMetaclass(final String metaclassQualifiedName) {
		return this.modelElements.get(metaclassQualifiedName);
	}

	/**
	 * Initialize the list of instances by metaclass from the resource set.
	 * 
	 * @param resourceSet
	 *            the resource set from the model in its initial state
	 */
	public void addElementsFrom(final ResourceSet resourceSet) {
		final EList<Resource> resources = resourceSet.getResources();
		if (resources.size() == 0) {
			return;
		}
		// consider only instances from the first resource
		final Resource resource = resources.get(0);
		addElementsFrom(resource);
	}

	public void addElementsFrom(final Resource resource) {
		final TreeIterator<EObject> allContents = resource.getAllContents();
		while (allContents.hasNext()) {
			final EObject eObject = allContents.next();
			addModelElement(eObject);
		}
	}

	/**
	 * Add a new element to the MetaclassListItemProvider corresponding to its
	 * metaclass.
	 * <p>
	 * Create a new MetaclassListItemProvider if necessary.
	 * 
	 * @param element
	 *            the element to add
	 */
	private void addModelElement(final EObject element) {
		final String classQualifiedName = EMFUtil.getMetaclassQualifiedName(element.eClass());

		InstancesForMetaclass instancesForMetaclass = this.modelElements.get(classQualifiedName);

		if (instancesForMetaclass == null) {
			instancesForMetaclass = new InstancesForMetaclass(element.eClass(), this,
					this.browserConfiguration);
			this.modelElements.put(classQualifiedName, instancesForMetaclass);
		}
		instancesForMetaclass.add(element, true);
		try {
			for (final Facet facet : this.browserConfiguration.getFacetContext().getFacets(element)) {
				final String facetQualifiedName = EMFUtil.getMetaclassQualifiedName(facet);
				InstancesForMetaclass instancesForFacet = this.modelElements.get(facetQualifiedName);
				if (instancesForFacet == null) {
					instancesForFacet = new InstancesForMetaclass(facet, this,
							this.browserConfiguration);
					this.modelElements.put(facetQualifiedName, instancesForFacet);
				}
				instancesForFacet.add(element, false);
			}
		} catch (final ModelQueryException e) {
			MoDiscoBrowserPlugin.logException(e);
		}
	}

	/** Adds the metaclasses from the given list to the list of metaclasses */
	public void addMetaclasses(final Collection<EClass> metaclasses) {
		for (final EClass eClass : metaclasses) {
			addMetaclass(eClass);
		}
	}

	/** Adds the given metaclass to the list of metaclasses */
	public void addMetaclass(final EClass eClass) {
		final String metaclassQualifiedName = EMFUtil.getMetaclassQualifiedName(eClass);

		if (this.modelElements.get(metaclassQualifiedName) == null) {
			final InstancesForMetaclass instancesForMetaclass = new InstancesForMetaclass(eClass,
					this, this.browserConfiguration);
			this.modelElements.put(metaclassQualifiedName, instancesForMetaclass);
		}
	}

	/** Builds the derivation tree of metaclasses */
	public void buildDerivationTree() {
		final InstancesForMetaclass[] instancesByMetaclass = getInstancesForMetaclasses();

		// clear previous state
		this.rootMetaclasses.clear();
		for (final InstancesForMetaclass instancesForMetaclass : instancesByMetaclass) {
			instancesForMetaclass.clearSubclasses();
		}

		// build the derivation tree
		for (final InstancesForMetaclass instancesForMetaclass : instancesByMetaclass) {
			instancesForMetaclass.buildParentsSubclasses();
			if (instancesForMetaclass.getEClass().getESuperTypes().isEmpty()) {
				this.rootMetaclasses.add(instancesForMetaclass);
			}
		}
	}

	public InstancesForMetaclass[] getRootMetaclasses() {
		return this.rootMetaclasses.toArray(new InstancesForMetaclass[this.rootMetaclasses.size()]);
	}

	public void removeMetaClasses(final Collection<? extends EClass> metaclasses) {
		for (final EClass eClass : metaclasses) {
			final String metaclassQualifiedName = EMFUtil.getMetaclassQualifiedName(eClass);
			this.modelElements.remove(metaclassQualifiedName);
		}
	}
}
