/*******************************************************************************
 * Copyright (c) 2008, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.ui.viewer.IViewerProvider;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory.Descriptor.Registry;
import org.eclipse.emf.edit.ui.provider.UnwrappingSelectionProvider;
import org.eclipse.emf.edit.ui.util.EditUIUtil;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.common.ui.controls.FormStyleSashForm;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.adapters.PropertySourceAdapterFactory;
import org.eclipse.gmt.modisco.infra.browser.core.AttributeItem;
import org.eclipse.gmt.modisco.infra.browser.core.BigListItem;
import org.eclipse.gmt.modisco.infra.browser.core.ITreeElement;
import org.eclipse.gmt.modisco.infra.browser.core.InstancesForMetaclass;
import org.eclipse.gmt.modisco.infra.browser.core.InstancesForMetaclasses;
import org.eclipse.gmt.modisco.infra.browser.core.LinkItem;
import org.eclipse.gmt.modisco.infra.browser.core.MetaclassList;
import org.eclipse.gmt.modisco.infra.browser.core.ModelElementItem;
import org.eclipse.gmt.modisco.infra.browser.core.QueryItem;
import org.eclipse.gmt.modisco.infra.browser.core.SearchResults;
import org.eclipse.gmt.modisco.infra.browser.custom.MetamodelView;
import org.eclipse.gmt.modisco.infra.browser.custom.core.CustomizationsCatalog;
import org.eclipse.gmt.modisco.infra.browser.customization.CustomizationEngine;
import org.eclipse.gmt.modisco.infra.browser.editors.BrowserConfiguration.ChangeListener;
import org.eclipse.gmt.modisco.infra.browser.preferences.PreferenceConstants;
import org.eclipse.gmt.modisco.infra.browser.queries.SelectedQueriesManager;
import org.eclipse.gmt.modisco.infra.browser.queries.SelectedQuery;
import org.eclipse.gmt.modisco.infra.browser.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.common.ui.editorInputs.IResourceEditorInput;
import org.eclipse.gmt.modisco.infra.facet.FacetSet;
import org.eclipse.gmt.modisco.infra.facet.core.FacetSetCatalog;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.INavigationLocation;
import org.eclipse.ui.INavigationLocationProvider;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.IPropertySource;
import org.eclipse.ui.views.properties.IPropertySourceProvider;
import org.eclipse.ui.views.properties.PropertySheetPage;

/**
 * MoDisco model editor: a two pane editor. The left pane shows the list of
 * metaclasses and the right pane shows the list of instances of the metaclass
 * selected in the left pane.
 */
public class EcoreBrowser extends EditorPart implements ISelectionProvider, IMenuListener,
		IViewerProvider, INavigationLocationProvider {

	private static final int MAIN_FORM_MARGIN_HEIGHT = 6;
	private static final int MAIN_FORM_MARGIN_WIDTH = 4;
	private static final int RIGHT_PANE_SASH_WEIGHT = 65;
	private static final int LEFT_PANE_SASH_WEIGHT = 35;
	public static final String EDITOR_ID = "org.eclipse.gmt.modisco.infra.browser.editorID"; //$NON-NLS-1$
	private static final String PROPERTY_VIEW_ID = "org.eclipse.ui.views.PropertySheet"; //$NON-NLS-1$

	private ResourceSet fResourceSet;

	/** An adapter factory that uses the registry */
	private ComposedAdapterFactory adapterFactoryWithRegistry;

	/** The property sheet page. */
	private PropertySheetPage propertySheetPage;

	/** The tree part of the model editor. */
	private TreeViewer treeViewer;

	/** The tree component (instances pane) */
	private Tree fTree;

	/** The list of metaclasses that appears in the left pane of the editor */
	private MetaclassViewer metaclassViewer;

	/** This listens to the tree viewer's selection. */
	private ISelectionChangedListener selectionChangedListener;

	/**
	 * This keeps track of all the
	 * {@link org.eclipse.jface.viewers.ISelectionChangedListener}s that are
	 * listening to this editor.
	 */
	private final Collection<ISelectionChangedListener> selectionChangedListeners = new ArrayList<ISelectionChangedListener>();

	/** This keeps track of the selection of the editor as a whole. */
	private ISelection editorSelection = StructuredSelection.EMPTY;

	/** This listens for workspace changes. */
	private final WorkspaceChangeListener resourceChangeListener;

	/**
	 * The configuration of this editor, which is passed to the adapters, so
	 * that they can use it to change their behavior accordingly
	 */
	private final BrowserConfiguration browserConfiguration;

	/** The composite in which the editor is displayed */
	private Composite parentComposite;

	/** The search box */
	private Text searchBox;

	/** The search job, to allow cancellation in case of a big model */
	private Job searchJob = null;

	/**
	 * The input of the tree viewer, saved before filtering by search, and
	 * restored afterwards
	 */
	private Object savedInput;

	/** A listener on changes to fonts settings */
	private IPropertyChangeListener fontChangeListener;

	/** Listens to changes in the loaded customizations, and updates the viewers */
	private ChangeListener fCustomizationEngineChangeListener = null;

	/** Allows optimized refreshes of the viewer */
	private Job refreshJob = null;

	/**
	 * Whether the resource being viewed comes from the package registry, in
	 * which case it must not be unloaded
	 */
	private boolean browingRegistryResource;
	private FormStyleSashForm sashForm;
	private CustomTreePainter customTreePainter;

	private FormToolkit formToolkit;
	private Form form;
	private LeftPane leftPane;
	private TreeToolBar instancesToolBar;

	/** Create the Mia Ecore model editor. */
	public EcoreBrowser() {
		super();
		this.browserConfiguration = new BrowserConfiguration(this);

		// load user settings
		this.browserConfiguration.load();

		this.resourceChangeListener = new WorkspaceChangeListener(this);
	}

	/** Set the selection into the tree viewer */
	public void setSelectionToViewer(final Collection<?> selection) {
		// Try to select the items in the tree viewer
		if (selection != null && !selection.isEmpty() && this.treeViewer != null) {
			this.treeViewer.setSelection(new StructuredSelection(selection.toArray()), true);
		}
	}

	/** @return the viewer as required by the {@link IViewerProvider} interface. */
	public Viewer getViewer() {
		return this.treeViewer;
	}

	public MetaclassViewer getMetaclassViewer() {
		return this.metaclassViewer;
	}

	/**
	 * Create a context menu for the viewer, add a listener, and register the
	 * menu for extension.
	 */
	protected void createContextMenuFor(final StructuredViewer viewer) {
		final MenuManager contextMenu = new MenuManager("#PopUp"); //$NON-NLS-1$
		contextMenu.add(new Separator("additions")); //$NON-NLS-1$
		contextMenu.setRemoveAllWhenShown(true);
		contextMenu.addMenuListener(this);
		final Menu menu = contextMenu.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(contextMenu, new UnwrappingSelectionProvider(viewer));
	}

	/** Load a resource based on the editor's input. */
	protected Resource createModel() {
		Resource result = null;
		this.fResourceSet.getURIConverter().getURIMap().putAll(EcorePlugin.computePlatformURIMap());
		this.browingRegistryResource = false;
		final URI resourceURI = EditUIUtil.getURI(getEditorInput());
		try {
			final IEditorInput editorInput = getEditorInput();
			if (editorInput instanceof FileEditorInput) {
				// Load the resource
				result = this.fResourceSet.getResource(resourceURI, true);
			} else if (editorInput instanceof IResourceEditorInput) {
				IResourceEditorInput resourceEditorInput = (IResourceEditorInput) editorInput;
				result = resourceEditorInput.getResource();
				this.fResourceSet.getResources().add(result);
				this.browingRegistryResource = true;
				setPartName(result.getURI().toString());
			} else {
				final String resolveURI = resourceURI.toString();
				final EPackage ePackage = this.fResourceSet.getPackageRegistry().getEPackage(
						resolveURI);
				if (ePackage != null) {
					result = ePackage.eResource();
					if (result != null) {
						this.fResourceSet.getResources().add(result);
						this.browingRegistryResource = true;
						setPartName(ePackage.getName());
					}
				} else {
					MessageDialog.openWarning(getSite().getShell(),
							Messages.EcoreBrowser_cannotOpenModelFromURI, NLS
									.bind(Messages.EcoreBrowser_cannotFindEPackageWithThatURI,
											resolveURI));
				}
			}
		} catch (final Exception e) {
			MoDiscoBrowserPlugin.logException(e);
			MessageDialog.openError(this.parentComposite.getShell(),
					Messages.EcoreBrowser_errorLoadingModel, e.toString());
			this.fResourceSet.getResource(resourceURI, false);
		}
		return result;
	}

	/** Create the editor composite. */
	@Override
	public void createPartControl(final Composite parent) {

		this.formToolkit = new FormToolkit(parent.getDisplay());
		this.form = this.formToolkit.createForm(parent);

		// final URI resourceURI = EditUIUtil.getURI(getEditorInput());
		// this.form.setText(resourceURI.toString());
		// this.formToolkit.decorateFormHeading(this.form);

		// Composite toolBarComposite = new Composite(form.getHead(), SWT.WRAP);
		// final IToolBarManager toolBarManager = this.form.getToolBarManager();
		// MainToolbar.create(toolBarManager, this.browserConfiguration,
		// getSite().getShell());
		// this.form.setHeadClient(toolBarComposite);

		this.parentComposite = parent;
		parent.setLayout(new FillLayout());

		final FillLayout fillLayout = new FillLayout();
		fillLayout.marginWidth = EcoreBrowser.MAIN_FORM_MARGIN_WIDTH;
		fillLayout.marginHeight = EcoreBrowser.MAIN_FORM_MARGIN_HEIGHT;
		this.form.getBody().setLayout(fillLayout);

		// Create an adapter factory that looks in the registry
		final Registry registry = ComposedAdapterFactory.Descriptor.Registry.INSTANCE;
		this.adapterFactoryWithRegistry = new ComposedAdapterFactory(registry);
		this.browserConfiguration.setAdapterFactoryWithRegistry(this.adapterFactoryWithRegistry);

		// register close listener to save user settings when editor is closed
		getSite().getPage().addPartListener(new SaveSettingsOnCloseListener());

		initResourceSet();
		this.browserConfiguration.setSelectedQueriesManager(new SelectedQueriesManager());
		this.sashForm = new FormStyleSashForm(this.form.getBody(), SWT.HORIZONTAL, this.formToolkit) {
			@Override
			protected void createContents() {
				createLeftPane(this);
				createTreePane(this);
			}
		};

		// ToolBar toolBar = this.leftPane.getToolbar().getToolBar();
		// ToolItem[] toolItems = toolBar.getItems();
		// int leftToolbarWidth = 25;
		// for (ToolItem toolItem : toolItems) {
		// leftToolbarWidth += toolItem.getWidth();
		// }
		// int width = this.leftPane.getToolBarComposite().getgetBounds().width;

		setupInitialTreeFont();

		this.sashForm.setWeights(new int[] { EcoreBrowser.LEFT_PANE_SASH_WEIGHT,
				EcoreBrowser.RIGHT_PANE_SASH_WEIGHT });
		// this.topSashForm.setMinWidth(leftToolbarWidth);

		// keep track of the selection, to be able to restore it when using the
		// "back" button
		addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(final SelectionChangedEvent event) {
				markNavigationLocation();
			}
		});

		/*
		 * Select the metaclass corresponding to the 'root' element (that is,
		 * the 'model' element). This in turn sets the input of the treeview.
		 */
		this.metaclassViewer.selectRootElement();

		createContextMenuFor(this.treeViewer);
		new DragAdapter(this.treeViewer);
		loadDefaultCustomization();
	}

	protected void initResourceSet() {
		// create an empty resource set
		this.fResourceSet = createResourceSet();
		this.browserConfiguration.setResourceSet(this.fResourceSet);

		HashSet<Resource> browsedResources = new HashSet<Resource>();
		Resource mainResource = createModel();
		browsedResources.add(mainResource);

		loadDefaultFacetSet(mainResource);

		addReferencedResources(mainResource, browsedResources);
		this.browserConfiguration.setBrowsedResources(browsedResources);

		this.browserConfiguration.initInstancesForMetaclasses();
		InstancesForMetaclasses instancesForMetaclasses = this.browserConfiguration
				.getInstancesForMetaclasses();
		// add all known metaclasses to the list (so that empty metaclasses can
		// be displayed)
		instancesForMetaclasses.addMetaclasses(this.browserConfiguration.getAllClasses());
		instancesForMetaclasses.buildDerivationTree();

		for (Resource resource : browsedResources) {
			instancesForMetaclasses.addElementsFrom(resource);
			this.browserConfiguration.getFacetContext().addComposedResource(resource);
		}
	}

	/**
	 * Create the resourceSet. This method can be overridden by subclasses.
	 * 
	 * @return The ResourceSet.
	 */
	protected ResourceSet createResourceSet() {
		return new ResourceSetImpl();
	}

	/**
	 * Load FacetSet defined as default for the root EPackage of root elements
	 * in the given resource
	 */
	private void loadDefaultFacetSet(final Resource resource) {
		for (EObject root : resource.getContents()) {
			EPackage rootEPackage = root.eClass().getEPackage();
			FacetSet facetSet = FacetSetCatalog.getSingleton().getDefaultFacetSet(rootEPackage);
			if (facetSet != null) {
				this.browserConfiguration.getFacetContext().addFacetSet(facetSet);
			}
		}
	}

	/**
	 * If proxies are found, ask the user whether to load resources referenced
	 * by proxies found in the given resource, and up to which depth.
	 * 
	 * @param resource
	 *            the resource in which to look for proxies to resolve
	 * @param allResources
	 *            the set of resources to which to add resources referenced by
	 *            proxies
	 */
	private void addReferencedResources(final Resource resource, final Set<Resource> allResources) {
		addReferencedResourcesInternal(resource, allResources, 1);
	}

	/**
	 * Adds resources referenced by proxies found in the given resources to the
	 * set <code>allResources</code>
	 * 
	 * @param resource
	 *            the resource in which to look for proxies to resolve
	 * @param allResources
	 *            the set of resources to which to add resources referenced by
	 *            proxies
	 * @param currentDepth
	 *            the current depth
	 * 
	 * @see EcoreBrowser#addReferencedResources(Resource, Set)
	 */
	private void addReferencedResourcesInternal(final Resource resource,
			final Set<Resource> allResources, final int currentDepth) {

		final Set<Resource> directReferencedResources = new HashSet<Resource>();

		for (EObject root : resource.getContents()) {
			Map<EObject, Collection<EStructuralFeature.Setting>> externalReferences = EcoreUtil.ExternalCrossReferencer
					.find(root);
			if (!externalReferences.isEmpty()) {
				int loadingDepth = this.browserConfiguration.getReferencedResourcesLoadingDepth();
				if (loadingDepth == BrowserConfiguration.NOT_YET_ASKED) {
					final int defaultLoadingDepth = MoDiscoBrowserPlugin.getPlugin()
							.getPreferenceStore().getInt(
									PreferenceConstants.P_BROWSER_LOADING_DEPTH);
					if (MoDiscoBrowserPlugin.getPlugin().getPreferenceStore().getBoolean(
							PreferenceConstants.P_BROWSER_ASK_LOADING_DEPTH)) {
						loadingDepth = askUserForDepthToLoadReferencedResources(defaultLoadingDepth);
					} else {
						loadingDepth = defaultLoadingDepth;
						if (loadingDepth < -1) {
							loadingDepth = 0;
						}
					}
					this.browserConfiguration.setReferencedResourcesLoadingDepth(loadingDepth);
					if (loadingDepth != -1 && loadingDepth < currentDepth) {
						return;
					}
				}
			}
			for (EObject external : externalReferences.keySet()) {
				Resource externalResource;
				if (external.eIsProxy()) {
					externalResource = EcoreUtil.resolve(external, root).eResource();
				} else {
					externalResource = external.eResource();
				}
				if (externalResource != null) {
					directReferencedResources.add(externalResource);
				} else {
					MoDiscoLogger.logError(
							"Failed to load an external element: " + external.eClass().getName() //$NON-NLS-1$
									+ " " + EcoreUtil.getURI(external), //$NON-NLS-1$
							MoDiscoBrowserPlugin.getPlugin());
				}
			}
		}

		int loadingDepth = this.browserConfiguration.getReferencedResourcesLoadingDepth();

		// recurse on sub-resources
		for (Resource directlyReferencedResource : directReferencedResources) {
			// avoid infinite recursion in the case of mutually referencing
			// resources
			if (!allResources.contains(directlyReferencedResource)) {
				allResources.add(directlyReferencedResource);
				if (loadingDepth == -1 || currentDepth + 1 <= loadingDepth) {
					addReferencedResourcesInternal(directlyReferencedResource, allResources,
							currentDepth + 1);
				}
			}
		}
	}

	private int askUserForDepthToLoadReferencedResources(final int initialValue) {
		IInputValidator validator = new IInputValidator() {
			public String isValid(final String inputText) {
				if ("*".equals(inputText)) { //$NON-NLS-1$
					return null;
				}
				try {
					Integer.parseInt(inputText);
				} catch (NumberFormatException e) {
					return NLS.bind(Messages.EcoreBrowser_notAPositiveInteger, inputText);
				}
				return null;
			}
		};
		final String initialValueStr;
		if (initialValue == -1) {
			initialValueStr = "*"; //$NON-NLS-1$
		} else {
			initialValueStr = Integer.toString(initialValue);
		}

		final InputDialog inputDialog = new InputDialog(this.getSite().getShell(),
				Messages.EcoreBrowser_selectDepthReferencedResources,
				Messages.EcoreBrowser_modelContainsProxies
						+ Messages.EcoreBrowser_chooseLoadProxiesDepth, initialValueStr, validator);

		if (inputDialog.open() == Window.OK) {
			String value = inputDialog.getValue();
			if (value == null) {
				return 0;
			}
			if ("*".equals(value)) { //$NON-NLS-1$
				return -1;
			}
			try {
				return Integer.parseInt(value);
			} catch (NumberFormatException e) {
				MoDiscoBrowserPlugin.logException(e);
				return 0;
			}
		}
		return 0;
	}

	public void reloadModel() {
		initResourceSet();
		this.metaclassViewer.reload();
		internalRefreshTree();
	}

	protected void reloadModelViewer() {
		this.metaclassViewer.reload();
		internalRefreshTree();
	}

	/**
	 * Create the left pane, containing the metaclass viewer.
	 * 
	 * @param parent
	 *            the parent composite
	 */
	protected void createLeftPane(final Composite parent) {
		final Section section = this.formToolkit.createSection(parent,
				ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setText(Messages.EcoreBrowser_metaclasses);
		// section.setDescription(Messages.EcoreBrowser_description_displaysTypes);
		// section.setLayoutData(new TableWrapData(TableWrapData.FILL));
		final Composite listPaneComposite = this.formToolkit.createComposite(section);
		section.setClient(listPaneComposite);
		this.leftPane = new LeftPane(section, listPaneComposite, this.browserConfiguration,
				this.formToolkit);
		final Composite toolBar = this.leftPane.getToolBarComposite();
		section.setTextClient(toolBar);
		listPaneComposite.setLayout(new FillLayout());
		// Composite listPaneComposite = new Composite(section, SWT.NONE);
		this.metaclassViewer = this.leftPane.getMetaclassViewer();
		this.metaclassViewer
				.addSelectionChangedListener(new MetaclassListViewerSelectionChangedListener());
	}

	/**
	 * Create the tree and its viewer
	 * 
	 * @param parent
	 *            the parent composite of the tree
	 */
	protected void createTreePane(final Composite parent) {

		final Section section = this.formToolkit.createSection(parent,
				ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setText(Messages.EcoreBrowser_instances);
		// section.setDescription(Messages.EcoreBrowser_description_displaysTypeInstances);
		// section.setLayoutData(new TableWrapData(TableWrapData.FILL));
		final Composite treePane = this.formToolkit.createComposite(section, SWT.NONE);
		section.setClient(treePane);

		// treePane.setLayout(new GridLayout());
		// treePane = formToolkit.createComposite(treePane, SWT.BORDER);

		// Composite treePane = new Composite(parent, SWT.NONE);

		final GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		treePane.setLayout(layout);

		final Composite treeToolBar = createTreeToolBar(section);
		section.setTextClient(treeToolBar);

		createSearchPanel(treePane);

		this.fTree = this.formToolkit.createTree(treePane, SWT.MULTI | SWT.BORDER);
		final GridData treeGridData = new GridData();
		treeGridData.grabExcessHorizontalSpace = true;
		treeGridData.grabExcessVerticalSpace = true;
		treeGridData.horizontalAlignment = SWT.FILL;
		treeGridData.verticalAlignment = SWT.FILL;
		this.fTree.setLayoutData(treeGridData);

		this.fTree.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.keyCode == SWT.ESC) {
					// Escape on the tree: restore view (after search)
					EcoreBrowser.this.searchBox.setText(""); //$NON-NLS-1$
					search();
				} else if (e.keyCode == SWT.CR) {
					// Enter: browse selected item
					handleEnter();
				} else if (e.keyCode == 'c' && e.stateMask == SWT.CTRL) {
					// Ctrl+C : copy element text to clipboard
					handleCopyToClipboard();
				} else if (e.keyCode == SWT.DEL) {
					handleDel();
				}
			}
		});

		this.customTreePainter = new CustomTreePainter(this.fTree, this.browserConfiguration);
		// create a tooltip to show facet names
		FacetTooltip.create(this.fTree, this.customTreePainter);

		this.treeViewer = new TreeViewer(this.fTree);
		// speeds up all operations on a big model
		this.treeViewer.setUseHashlookup(true);

		hookViewerToSelection(this.treeViewer);

		this.treeViewer.setContentProvider(new BrowserContentProvider());
		this.treeViewer.setLabelProvider(new BrowserLabelProvider());

		registerShowPropertyViewOnDblClick(this.fTree);
	}

	/**
	 * The user pressed the "Enter" key : if selected element is a model
	 * element, then browse to it. If it is an attribute, then open properties
	 * view.
	 */
	protected void handleEnter() {
		final ISelection selection = getEditorSelection();
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			final Object selectedElement = treeSelection.getFirstElement();
			if (selectedElement instanceof ModelElementItem) {
				browse();
			} else if (selectedElement instanceof AttributeItem
					|| selectedElement instanceof LinkItem) {
				openPropertiesView();
			} else if (selectedElement instanceof QueryItem) {
				QueryItem queryItem = (QueryItem) selectedElement;
				SelectedQuery query = queryItem.getQuery();
				query.executeQuery();
			}
		}
	}

	/** The user pressed the "Delete" key */
	protected void handleDel() {
		final ISelection selection = getEditorSelection();
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			Iterator<?> iterator = treeSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof QueryItem) {
					QueryItem queryItem = (QueryItem) selectedElement;
					this.browserConfiguration.getSelectedQueriesManager().remove(
							queryItem.getQuery());
				}
			}
		}
	}

	/** Copy the text of the selected element to clipboard */
	private void handleCopyToClipboard() {
		final ISelection selection = getEditorSelection();
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			StringBuilder builder = new StringBuilder();
			Iterator<?> iterator = treeSelection.iterator();
			while (iterator.hasNext()) {
				if (builder.length() > 0) {
					builder.append("\n"); //$NON-NLS-1$
				}
				Object selectedElement = iterator.next();
				if (selectedElement instanceof ModelElementItem) {
					ModelElementItem modelElementItem = (ModelElementItem) selectedElement;
					builder.append(modelElementItem.getName());
				} else if (selectedElement instanceof AttributeItem) {
					AttributeItem attributeItem = (AttributeItem) selectedElement;
					String valueText = attributeItem.getValueText();
					final String strEquals = " = "; //$NON-NLS-1$
					if (valueText.startsWith(strEquals)) {
						builder.append(valueText.substring(strEquals.length()));
					} else {
						builder.append(valueText);
					}
				} else if (selectedElement instanceof LinkItem) {
					LinkItem linkItem = (LinkItem) selectedElement;
					builder.append(linkItem.getReference().getName());
				} else if (selectedElement instanceof ITreeElement) {
					ITreeElement treeElement = (ITreeElement) selectedElement;
					builder.append(treeElement.getText());
				} else {
					builder.append(selectedElement.toString());
				}
			}
			Clipboard clipboard = new Clipboard(Display.getDefault());
			clipboard.setContents(new Object[] { builder.toString() },
					new Transfer[] { TextTransfer.getInstance() });
			clipboard.dispose();
		}
	}

	/**
	 * Set the initial font using saved user preferences, and add a listener to
	 * be notified of changes in the preferences
	 */
	private void setupInitialTreeFont() {
		setupPreferredFont();

		this.fontChangeListener = new IPropertyChangeListener() {
			public void propertyChange(final PropertyChangeEvent event) {
				if (event.getProperty().equals(JFaceResources.DIALOG_FONT)) {
					setupPreferredFont();
				}
			}
		};
		JFaceResources.getFontRegistry().addListener(this.fontChangeListener);
	}

	/** Set the user preferred font on the editor */
	private void setupPreferredFont() {
		FontDescriptor descriptor = FontDescriptor.createFrom(JFaceResources.getDialogFont());
		descriptor = descriptor.increaseHeight(this.browserConfiguration.getFontSizeDelta());
		final Font customFont = descriptor.createFont(Display.getDefault());
		descriptor = descriptor.setStyle(SWT.ITALIC);
		final Font customItalicFont = descriptor.createFont(Display.getDefault());
		this.browserConfiguration.setCustomFont(customFont);
		this.browserConfiguration.setCustomItalicFont(customItalicFont);
		this.fTree.setFont(customFont);
		this.treeViewer.refresh();
		this.metaclassViewer.setFont(customFont);
		this.metaclassViewer.refresh();
	}

	private Composite createTreeToolBar(final Composite parent) {

		final Composite treeToolBarComposite = new Composite(parent, SWT.WRAP);
		// GridData treeToolBarGridData = new GridData();
		// treeToolBarGridData.grabExcessHorizontalSpace = true;
		// treeToolBarGridData.horizontalAlignment = SWT.FILL;
		// treeToolBarComposite.setLayoutData(treeToolBarGridData);

		this.instancesToolBar = TreeToolBar.create(treeToolBarComposite, this);
		return treeToolBarComposite;
	}

	/**
	 * Register a listener to show the property view or execute a query when the
	 * tree is double-clicked
	 */
	private void registerShowPropertyViewOnDblClick(final Tree tree) {
		tree.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDoubleClick(final MouseEvent event) {
				boolean handled = false;
				if (event.button == 1) {
					TreeItem[] selection = tree.getSelection();
					for (TreeItem treeItem : selection) {
						Object element = treeItem.getData();
						if (element instanceof QueryItem) {
							QueryItem queryItem = (QueryItem) element;
							SelectedQuery query = queryItem.getQuery();
							query.executeQuery();
							handled = true;
						}
					}
					if (!handled) {
						openPropertiesView();
					}
				}
			}
		});
	}

	/** A listener to save user settings when the editor is closed */
	private class SaveSettingsOnCloseListener implements IPartListener {
		public void partClosed(final IWorkbenchPart part) {
			if (part == EcoreBrowser.this) {
				EcoreBrowser.this.browserConfiguration.save();
				getSite().getPage().removePartListener(this);
			}
		}

		public void partOpened(final IWorkbenchPart part) {
		}

		public void partDeactivated(final IWorkbenchPart part) {
		}

		public void partBroughtToTop(final IWorkbenchPart part) {
		}

		public void partActivated(final IWorkbenchPart part) {
		}
	}

	/**
	 * Create the search panel, containing a label and search field.
	 * 
	 * @param parent
	 *            the parent composite, which must have a GridLayout
	 */
	protected void createSearchPanel(final Composite parent) {
		final Composite searchPanel = new Composite(parent, SWT.NONE);
		searchPanel.setLayout(new GridLayout());

		searchPanel.setLayout(new FillLayout());
		final GridData searchPanelGridData = new GridData();
		searchPanelGridData.grabExcessHorizontalSpace = true;
		searchPanelGridData.horizontalAlignment = SWT.FILL;
		searchPanel.setLayoutData(searchPanelGridData);

		// Label searchLabel = new Label(searchPanel, SWT.NONE);
		// searchLabel.setText("&Search:");

		this.searchBox = new Text(searchPanel, SWT.SEARCH);
		this.searchBox.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
					// ENTER: start a search
					search();
				} else if (e.keyCode == SWT.ESC) {
					// ESCAPE: restore the view after a search
					EcoreBrowser.this.searchBox.setText(""); //$NON-NLS-1$
					search();
				}
			}
		});

		this.searchBox.setMessage(Messages.EcoreBrowser_typeFilterText);
	}

	/**
	 * Called when the user initiates a search. Find the list of all the
	 * instances of the selected metaclass that have a name containing the
	 * search string (entered by the user in the search box), in a case
	 * insensitive way. Set this list as input of the tree viewer.
	 * <p>
	 * Do the search in a Job that can be canceled by the user.
	 * <p>
	 * If the search box is empty, then restore the previous tree viewer input.
	 */
	protected void search() {
		final String searchText = this.searchBox.getText().toLowerCase();

		if (searchText.trim().length() == 0) {
			if (this.savedInput != null && this.savedInput != this.treeViewer.getInput()) {
				this.treeViewer.setInput(this.savedInput);
			}
			return;
		}

		final List<EClass> selectedMetaclasses = findSelectedEClasses();
		if (selectedMetaclasses.isEmpty()) {
			return;
		}

		this.searchJob = new Job(Messages.EcoreBrowser_jobName_SearchInModelEditor) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {

				final ResourceSet resourceSet = EcoreBrowser.this.fResourceSet;
				final EList<Resource> resources = resourceSet.getResources();
				if (resources.size() < 1) {
					return Status.OK_STATUS;
				}

				// the search results : will become the input of the tree viewer
				final SearchResults searchResults = new SearchResults(
						EcoreBrowser.this.browserConfiguration);

				final Resource resource = resources.get(0);
				final TreeIterator<EObject> allContents = resource.getAllContents();
				while (allContents.hasNext()) {
					final EObject eObject = allContents.next();
					if (selectedMetaclasses.contains(eObject.eClass())) {
						final ModelElementItem modelElementItem = new ModelElementItem(eObject,
								null, EcoreBrowser.this.browserConfiguration);
						final String objectText = modelElementItem.getName().toLowerCase();
						if (objectText.contains(searchText)) {
							searchResults.add(eObject);
						}
					}

					// if the user canceled the search
					if (monitor.isCanceled()) {
						return Status.CANCEL_STATUS;
					}
				}

				// do only the minimum amount of work in the UI thread
				Display.getDefault().syncExec(new Runnable() {
					public void run() {
						// do not try to restore selection
						EcoreBrowser.this.treeViewer.setSelection(null);
						EcoreBrowser.this.treeViewer.setInput(searchResults);
					}
				});
				return Status.OK_STATUS;
			}
		};
		this.searchJob.setUser(true);
		this.searchJob.setPriority(Job.LONG);
		this.searchJob.schedule();
	}

	/**
	 * @return the {@link EClass}es corresponding to the metaclasses selected in
	 *         the metaclass list (empty list if none)
	 */
	private List<EClass> findSelectedEClasses() {

		final ArrayList<EClass> selectedMetaclasses = new ArrayList<EClass>();

		final ISelection selection = this.metaclassViewer.getSelection();
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;

			final Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				final Object element = iterator.next();
				if (element instanceof InstancesForMetaclass) {
					final InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
					addMetaclassToSelection(instancesForMetaclass, selectedMetaclasses,
							this.browserConfiguration.isDisplayInstancesOfSubclasses());
				}
			}
		}
		return selectedMetaclasses;
	}

	/**
	 * Adds the given metaclass to the given selection.
	 * 
	 * @param addSubclasses
	 *            also adds subclasses
	 */
	private void addMetaclassToSelection(final InstancesForMetaclass instancesForMetaclass,
			final ArrayList<EClass> selectedMetaclasses, final boolean addSubclasses) {
		selectedMetaclasses.add(instancesForMetaclass.getEClass());
		// add subclasses recursively
		if (addSubclasses) {
			final InstancesForMetaclass[] subclasses = instancesForMetaclass.getSubclasses();
			for (final InstancesForMetaclass subclass : subclasses) {
				addMetaclassToSelection(subclass, selectedMetaclasses, addSubclasses);
			}
		}
	}

	/**
	 * Listen to selection changes in the metaclass list viewer, and update the
	 * tree viewer accordingly.
	 */
	private final class MetaclassListViewerSelectionChangedListener implements
			ISelectionChangedListener {

		/** Memorize the tree selection for each metaclass */
		private final HashMap<String, ITreeSelection> selections = new HashMap<String, ITreeSelection>();
		/** The metaclass that was selected before the selection changed */
		private String lastSelectedMetaclass = null;

		/**
		 * Get the selected entries corresponding to the given selection in the
		 * metaclass viewer (may be empty)
		 */
		private InstancesForMetaclass[] getEntriesForSelection(final ISelection selection) {
			final ArrayList<InstancesForMetaclass> instancesForMetaclasses = new ArrayList<InstancesForMetaclass>();

			if (selection instanceof IStructuredSelection) {
				final IStructuredSelection structuredSelection = (IStructuredSelection) selection;

				for (final Iterator<?> iterator = structuredSelection.iterator(); iterator
						.hasNext();) {
					final Object element = iterator.next();
					if (element instanceof InstancesForMetaclass) {
						final InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
						instancesForMetaclasses.add(instancesForMetaclass);
					}
				}
			}

			return instancesForMetaclasses
					.toArray(new InstancesForMetaclass[instancesForMetaclasses.size()]);
		}

		/**
		 * Updates the tree viewer when the selection changes in the metaclass
		 * viewer. Keeps track of the tree selection for each metaclass, and
		 * restores it when possible.
		 */
		public void selectionChanged(final SelectionChangedEvent event) {

			// save the selection
			if (this.lastSelectedMetaclass != null) {
				this.selections.put(this.lastSelectedMetaclass,
						(ITreeSelection) EcoreBrowser.this.treeViewer.getSelection());
			}

			final ISelection metaclassSelection = event.getSelection();
			final InstancesForMetaclass[] entries = getEntriesForSelection(metaclassSelection);

			ITreeSelection savedSelection = null;
			Object input = null;
			if (entries.length > 0) {
				savedSelection = this.selections.get(entries[0].getClassQualifiedName());

				final String[] selectedMetaclassesNames = new String[entries.length];
				int i = 0;
				for (final InstancesForMetaclass entry : entries) {
					selectedMetaclassesNames[i++] = entry.getClassQualifiedName();
				}

				input = new MetaclassList(EcoreBrowser.this.browserConfiguration,
						selectedMetaclassesNames);
				this.lastSelectedMetaclass = entries[0].getClassQualifiedName();
			} else {
				this.lastSelectedMetaclass = null;
			}

			try {
				// speedup and avoid flickering
				EcoreBrowser.this.treeViewer.getTree().setRedraw(false);

				EcoreBrowser.this.searchBox.setText(""); //$NON-NLS-1$

				// do not restore selection twice
				EcoreBrowser.this.treeViewer.setSelection(null);

				// set the list of model elements as the input of the treeview
				EcoreBrowser.this.treeViewer.setInput(input);
				EcoreBrowser.this.savedInput = input;
				EcoreBrowser.this.treeViewer.setSelection(savedSelection, true);
			} finally {
				EcoreBrowser.this.treeViewer.getTree().setRedraw(true);
			}

			markNavigationLocation();
		}
	}

	public void setShowEmptyLinks(final boolean value) {
		this.browserConfiguration.setShowEmptyLinks(value);
		internalRefreshTree();
	}

	public boolean isShowEmptyLinks() {
		return this.browserConfiguration.isShowEmptyLinks();
	}

	public void setSortInstances(final boolean value) {
		this.browserConfiguration.setSortInstances(value);
		internalRefreshTree();
	}

	public boolean isSortInstances() {
		return this.browserConfiguration.isSortInstances();
	}

	public void setShowDerivedLinks(final boolean value) {
		this.browserConfiguration.setShowDerivedLinks(value);
		internalRefreshTree();
	}

	public boolean isShowDerivedLinks() {
		return this.browserConfiguration.isShowDerivedLinks();
	}

	public void setShowFullQualifiedNames(final boolean value) {
		this.browserConfiguration.setShowFullQualifiedNames(value);
		internalRefreshTree();
		this.metaclassViewer.refresh();
	}

	public boolean isShowFullQualifiedNames() {
		return this.browserConfiguration.isShowFullQualifiedNames();
	}

	public void setShowMultiplicity(final boolean value) {
		this.browserConfiguration.setShowMultiplicity(value);
		internalRefreshTree();
	}

	public boolean isShowMultiplicity() {
		return this.browserConfiguration.isShowMultiplicity();
	}

	public void setShowOppositeLinks(final boolean value) {
		this.browserConfiguration.setShowOppositeLinks(value);
		internalRefreshTree();
	}

	public boolean isShowOppositeLinks() {
		return this.browserConfiguration.isShowOppositeLinks();
	}

	public void setShowContainer(final boolean value) {
		this.browserConfiguration.setShowContainer(value);
		internalRefreshTree();
	}

	public boolean isShowContainer() {
		return this.browserConfiguration.isShowContainer();
	}

	public void setSortLinks(final boolean value) {
		this.browserConfiguration.setSortLinks(value);
		internalRefreshTree();
	}

	public boolean isSortLinks() {
		return this.browserConfiguration.isSortLinks();
	}

	public void setSortLinksByType(final boolean value) {
		this.browserConfiguration.setSortLinksByType(value);
		internalRefreshTree();
	}

	public boolean isSortLinksByType() {
		return this.browserConfiguration.isSortLinksByType();
	}

	public void setShowAttributes(final boolean value) {
		this.browserConfiguration.setShowAttributes(value);
		internalRefreshTree();
	}

	public boolean isShowAttributes() {
		return this.browserConfiguration.isShowAttributes();
	}

	public void setShowEmptyAttributes(final boolean value) {
		this.browserConfiguration.setShowEmptyAttributes(value);
		internalRefreshTree();
	}

	public boolean isShowEmptyAttributes() {
		return this.browserConfiguration.isShowEmptyAttributes();
	}

	public void setFontSizeDelta(final int value) {
		this.browserConfiguration.setFontSizeDelta(value);
		setupPreferredFont();
	}

	public int getFontSizeDelta() {
		return this.browserConfiguration.getFontSizeDelta();
	}

	public void setShowOrdering(final boolean value) {
		this.browserConfiguration.setShowOrdering(value);
		internalRefreshTree();
	}

	public boolean isShowOrdering() {
		return this.browserConfiguration.isShowOrdering();
	}

	public void setShowTypeOfLinks(final boolean value) {
		this.browserConfiguration.setShowTypeOfLinks(value);
		internalRefreshTree();
	}

	public boolean isShowTypeOfLinks() {
		return this.browserConfiguration.isShowTypeOfLinks();
	}

	public void setShowElementIDs(final boolean value) {
		this.browserConfiguration.setShowElementIDs(value);
		internalRefreshTree();
	}

	public boolean isShowElementIDs() {
		return this.browserConfiguration.isShowElementIDs();
	}

	/** This is how the framework determines which interfaces we implement. */
	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(final Class key) {
		if (key.equals(IPropertySheetPage.class)) {
			return getPropertySheetPage();
		} else {
			return super.getAdapter(key);
		}
	}

	/** This accesses a cached version of the property sheet page. */
	public IPropertySheetPage getPropertySheetPage() {
		if (this.propertySheetPage == null) {
			this.propertySheetPage = new PropertySheetPage() {
				// @Override
				// public void selectionChanged(IWorkbenchPart part, ISelection
				// selection) {
				// // avoid the slow view redrawing
				// getControl().setRedraw(false);
				// try {
				// super.selectionChanged(part, selection);
				// } finally {
				// getControl().setRedraw(true);
				// }
				// }
			};
			this.propertySheetPage.setPropertySourceProvider(new IPropertySourceProvider() {
				public IPropertySource getPropertySource(final Object object) {
					// The property source is not implemented by the EObject
					// directly, but by an adapter created for it with the
					// help of an adapter factory.
					PropertySourceAdapterFactory propertySourceAdapterFactory = new PropertySourceAdapterFactory(
							EcoreBrowser.this.browserConfiguration);
					Object adapted = propertySourceAdapterFactory.adapt(object,
							IPropertySource.class);
					if (adapted instanceof IPropertySource) {
						IPropertySource propertySource = (IPropertySource) adapted;
						return propertySource;
					}
					return null;
				}
			});
		}

		return this.propertySheetPage;
	}

	// /** Handles activation of the editor or its associated views. */
	// protected void handleActivate() {
	// // Recompute the read only state.
	// if (this.editingDomain.getResourceToReadOnlyMap() != null) {
	// this.editingDomain.getResourceToReadOnlyMap().clear();
	//
	// // Refresh any actions that may become enabled or disabled.
	// setSelection(getSelection());
	// }
	// }

	// /**
	// * Handles what to do with changed resources on activation.
	// *
	// * @param changedResources
	// * the resources that have changed
	// */
	// protected void handleChangedResources(Collection<Resource>
	// changedResources) {
	//
	// if (isDirty()) {
	// changedResources.addAll(this.editingDomain.getResourceSet().getResources());
	// }
	// this.editingDomain.getCommandStack().flush();
	//
	// for (Resource resource : changedResources) {
	// if (resource.isLoaded()) {
	// resource.unload();
	// try {
	// resource.load(Collections.EMPTY_MAP);
	// } catch (IOException exception) {
	// MoDiscoEditorPlugin.INSTANCE.log(exception);
	// MessageDialog.openError(this.parentComposite.getShell(),
	// "Error loading resource", exception.toString());
	// }
	// }
	// }
	//
	// if (AdapterFactoryEditingDomain.isStale(this.editorSelection)) {
	// setSelection(StructuredSelection.EMPTY);
	// }
	// }

	@Override
	public boolean isDirty() {
		return false;
	}

	/**
	 * This is for implementing {@link org.eclipse.ui.IEditorPart} : saves the
	 * model file.
	 */
	@Override
	public void doSave(final IProgressMonitor progressMonitor) {
	}

	@Override
	public void doSaveAs() {
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	// /** Implements {@link IGotoMarker}. Select the EObject corresponding to
	// the given marker */
	// public void gotoMarker(IMarker marker) {
	// String uriAttribute = marker.getAttribute(EValidator.URI_ATTRIBUTE,
	// null);
	// if (uriAttribute != null) {
	// URI uri = URI.createURI(uriAttribute);
	// EObject eObject = this.resourceSet.getEObject(uri, true);
	// if (eObject != null) {
	// markNavigationLocation();
	// String classQualifiedName =
	// EMFUtil.getMetaclassQualifiedName(eObject.eClass());
	// // Select the metaclass in the metaclass list viewer
	// this.metaclassViewer.selectMetaclass(classQualifiedName);
	// selectElement(eObject);
	// markNavigationLocation();
	// }
	// }
	// }

	/**
	 * This is called during startup to initialize the editor with its site and
	 * input.
	 */
	@Override
	public void init(final IEditorSite site, final IEditorInput editorInput) {
		// TODO: check input is of expected type
		try {
			setSite(site);
			setInputWithNotify(editorInput);
			setPartName(editorInput.getName());
			site.setSelectionProvider(this);
			ResourcesPlugin.getWorkspace().addResourceChangeListener(this.resourceChangeListener,
					IResourceChangeEvent.POST_CHANGE);
		} catch (final Exception e) {
			e.printStackTrace();
			MoDiscoBrowserPlugin.logException(e);
		}
	}

	/** Focus the tree viewer when the editor is focused */
	@Override
	public void setFocus() {
		if (this.treeViewer != null) {
			this.treeViewer.getTree().setFocus();
		}
	}

	/** This implements {@link org.eclipse.jface.viewers.ISelectionProvider}. */
	public void addSelectionChangedListener(final ISelectionChangedListener listener) {
		this.selectionChangedListeners.add(listener);
	}

	/** This implements {@link org.eclipse.jface.viewers.ISelectionProvider}. */
	public void removeSelectionChangedListener(final ISelectionChangedListener listener) {
		this.selectionChangedListeners.remove(listener);
	}

	/**
	 * Returns the "unwrapped" selection of the editor, with Ecore elements
	 * instead of the proxies used internally
	 */
	public ISelection getSelection() {
		return unwrapSelection(this.editorSelection);
	}

	/** @return the "real" selection, with proxy objects (ModelElementItem, etc.) */
	public ISelection getEditorSelection() {
		return this.editorSelection;
	}

	/**
	 * Replace proxy objects by their underlying Ecore objects in the given
	 * selection
	 * 
	 * @return the unwrapped version of the selection
	 */
	public static ISelection unwrapSelection(final ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			final List<Object> elements = new ArrayList<Object>();
			final Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				final Object element = iterator.next();
				if (element instanceof ModelElementItem) {
					final ModelElementItem modelElementItem = (ModelElementItem) element;
					elements.add(modelElementItem.getEObject());
				} else if (element instanceof LinkItem) {
					final LinkItem linkItem = (LinkItem) element;
					elements.add(linkItem.getReference());
				} else if (element instanceof AttributeItem) {
					final AttributeItem attributeItem = (AttributeItem) element;
					elements.add(attributeItem.getAttribute());
				} else {
					elements.add(element);
				}
			}
			return new StructuredSelection(elements);

		}
		return selection;
	}

	/**
	 * This implements {@link org.eclipse.jface.viewers.ISelectionProvider} to
	 * set this editor's overall selection. Calling this result will notify the
	 * listeners.
	 */
	public void setSelection(final ISelection selection) {
		this.editorSelection = selection;

		for (final ISelectionChangedListener listener : this.selectionChangedListeners) {
			// depending on the listener type, unwrap the selection
			if (listener instanceof IBrowserSelectionChangedListener) {
				// expose the proxy objects
				listener.selectionChanged(new SelectionChangedEvent(this, selection));
			} else {
				// expose the underlying Ecore elements
				listener.selectionChanged(new SelectionChangedEvent(this,
						unwrapSelection(selection)));
			}
		}
		setStatusLineMessage(selection);
	}

	/** Connect the given viewer to the editor's selection */
	private void hookViewerToSelection(final Viewer viewer) {
		if (viewer == null) {
			return;
		}

		if (this.selectionChangedListener == null) {
			// Create the listener on demand
			this.selectionChangedListener = new ISelectionChangedListener() {
				// This just notifies those things that are affected by the
				// selection
				public void selectionChanged(final SelectionChangedEvent selectionChangedEvent) {
					setSelection(selectionChangedEvent.getSelection());
				}
			};
		}

		viewer.addSelectionChangedListener(this.selectionChangedListener);

		// Set the editor's selection based on the viewer's selection
		setSelection(viewer.getSelection());
	}

	/** Set the status line message depending on the given selection */
	public void setStatusLineMessage(final ISelection selection) {
		// TODO: maybe display more useful information?
		final IStatusLineManager statusLineManager = getActionBars().getStatusLineManager();

		if (statusLineManager != null) {
			if (selection instanceof IStructuredSelection) {
				final Collection<?> collection = ((IStructuredSelection) selection).toList();
				switch (collection.size()) {
				case 0:
					statusLineManager.setMessage(Messages.EcoreBrowser_noObjectSelected);
					break;
				case 1:
					final Object object = collection.iterator().next();
					if (object instanceof ITreeElement) {
						final ITreeElement treeElement = (ITreeElement) object;
						statusLineManager.setMessage(NLS.bind(
								Messages.EcoreBrowser_singleObjectSelected, treeElement.getText()));
					}
					break;
				default:
					statusLineManager.setMessage(NLS.bind(
							Messages.EcoreBrowser_multiObjectSelected, Integer.toString(collection
									.size())));
					break;
				}
			} else {
				statusLineManager.setMessage(""); //$NON-NLS-1$
			}
		}
	}

	/**
	 * This implements {@link org.eclipse.jface.action.IMenuListener} to help
	 * fill the context menus with contributions from the Edit menu.
	 */
	public void menuAboutToShow(final IMenuManager menuManager) {
		((IMenuListener) getEditorSite().getActionBarContributor()).menuAboutToShow(menuManager);
	}

	public BrowserActionBarContributor getActionBarContributor() {
		return (BrowserActionBarContributor) getEditorSite().getActionBarContributor();
	}

	public IActionBars getActionBars() {
		return getActionBarContributor().getActionBars();
	}

	/** Clean up */
	@Override
	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this.resourceChangeListener);

		this.adapterFactoryWithRegistry.dispose();
		this.sashForm.dispose();

		if (!this.browingRegistryResource) {
			// unload all resources
			final EList<Resource> resources = this.fResourceSet.getResources();
			final Job cleanupJob = new Job(Messages.EcoreBrowser_jobName_UnloadResources) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					for (final Resource resource : resources) {
						resource.unload();
					}
					return Status.OK_STATUS;
				}
			};
			cleanupJob.setPriority(Job.DECORATE);
			cleanupJob.setSystem(true);
			cleanupJob.schedule();
		}

		if (getActionBarContributor().getActiveEditor() == this) {
			getActionBarContributor().setActiveEditor(null);
		}

		if (this.propertySheetPage != null) {
			this.propertySheetPage.dispose();
		}

		if (this.treeViewer != null && this.selectionChangedListener != null) {
			this.treeViewer.removeSelectionChangedListener(this.selectionChangedListener);
			this.selectionChangedListener = null;
		}

		// dispose fonts
		this.browserConfiguration.setCustomFont(null);
		this.browserConfiguration.setCustomItalicFont(null);

		// avoid leaking editor
		this.browserConfiguration.clearEditor();

		JFaceResources.getFontRegistry().removeListener(this.fontChangeListener);

		this.customTreePainter.dispose();
		// dispose the customization engine, which removes listeners too
		this.browserConfiguration.getCustomizationEngine().dispose();
		this.fCustomizationEngineChangeListener = null;
		this.formToolkit.dispose();
		super.dispose();
	}

	/** Open the selected element in the list of elements of the same metaclass */
	public void browse() {
		final ISelection selection = getEditorSelection();
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			final Object selectedElement = treeSelection.getFirstElement();
			if (selectedElement instanceof ModelElementItem) {
				final ModelElementItem modelElementItem = (ModelElementItem) selectedElement;
				browseTo(modelElementItem);
			}
		}
	}

	/** Open the given element in the list of elements of the same metaclass */
	public void browseTo(final ModelElementItem modelElementItem) {
		final EObject modelElement = modelElementItem.getEObject();
		browseTo(modelElement);
	}

	/** Open the given element in the list of elements of the same metaclass */
	public void browseTo(final EObject modelElement) {
		markNavigationLocation();
		final String classQualifiedName = EMFUtil.getMetaclassQualifiedName(modelElement.eClass());
		// Select the metaclass in the metaclass list viewer
		this.metaclassViewer.selectMetaclass(classQualifiedName);
		TreePath treePath = findTreePathForElement(modelElement);
		if (treePath != null) {
			// deselect any previously selected elements
			this.treeViewer.setSelection(null);
			this.treeViewer.setSelection(new TreeSelection(treePath), true);
			markNavigationLocation();
		}
	}

	public void browseToByURI(final URI eProxyURI) {
		EObject eObject = EMFUtil.findElementByURI(eProxyURI, this.fResourceSet);
		if (eObject != null) {
			browseTo(eObject);
		}
	}

	/**
	 * Finds the path of the given element in the tree viewer. The element is
	 * looked for at the root and under BigListItemProviders in the case the
	 * elements are split.
	 * 
	 * @return the path or <code>null</code> if the element was not found
	 */
	private TreePath findTreePathForElement(final EObject eObject) {
		final Object input = this.treeViewer.getInput();
		if (input instanceof MetaclassList) {
			final MetaclassList metaclassListItemProvider = (MetaclassList) input;
			final Collection<?> elements = metaclassListItemProvider.getElements();
			for (final Object element : elements) {
				// if the elements are split
				if (element instanceof BigListItem) {
					final BigListItem bigListItemProvider = (BigListItem) element;
					final Collection<?> subElements = bigListItemProvider.getChildren();
					for (final Object subElement : subElements) {
						if (subElement instanceof ModelElementItem) {
							final ModelElementItem other = (ModelElementItem) subElement;
							if (other.getEObject().equals(eObject)) {
								return new TreePath(new Object[] { subElement });
							}
						}
					}
				}

				if (element instanceof ModelElementItem) {
					final ModelElementItem other = (ModelElementItem) element;
					if (other.getEObject().equals(eObject)) {
						return new TreePath(new Object[] { element });
					}
				}
			}
		}
		return null;
	}

	/** Implements {@link INavigationLocationProvider} */
	public INavigationLocation createEmptyNavigationLocation() {
		return new BrowserNavigationLocation(this);
	}

	/**
	 * Implements {@link INavigationLocationProvider}. Create a location that
	 * can be navigated back to, using the "back" button
	 */
	public INavigationLocation createNavigationLocation() {
		// navigation to the first element of the selection
		final String firstSelectedMetaclass = this.metaclassViewer
				.getFirstSelectedMetaclassQualifiedName();
		return new BrowserNavigationLocation(this, firstSelectedMetaclass,
				(ITreeSelection) this.treeViewer.getSelection());
	}

	/** Mark a location that can be navigated back to, using the "back" button */
	protected void markNavigationLocation() {
		getSite().getPage().getNavigationHistory().markLocation(this);
	}

	/**
	 * Restore a location (used for implementing history).
	 * 
	 * @param metaclassQualifiedName
	 *            the metaclass viewer's selection
	 * @param selection
	 *            the tree viewer's selection
	 */
	public void restoreLocation(final String metaclassQualifiedName, final ITreeSelection selection) {
		this.metaclassViewer.selectMetaclass(metaclassQualifiedName);
		this.treeViewer.setSelection(selection);
	}

	/** Refresh the tree viewer part of the editor */
	public void refreshTreeDelayed() {
		if (this.refreshJob == null) {
			this.refreshJob = new Job(Messages.EcoreBrowser_refreshingModelViewers) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							internalRefreshTree();
						}
					});
					return Status.OK_STATUS;
				}
			};
		} else {
			/*
			 * If refreshTreeDelayed is called again before the job delay
			 * passed, then cancel the job and re-schedule a new one.
			 */
			this.refreshJob.cancel();
		}
		this.refreshJob.setPriority(Job.INTERACTIVE);
		this.refreshJob.schedule();
	}

	private void internalRefreshTree() {
		if (!this.treeViewer.getTree().isDisposed()) {
			try {
				this.treeViewer.getTree().setRedraw(false);
				this.treeViewer.refresh();
			} finally {
				this.treeViewer.getTree().setRedraw(true);
			}
		}
	}

	public BrowserConfiguration getBrowserConfiguration() {
		return this.browserConfiguration;
	}

	private void openPropertiesView() {
		try {
			getEditorSite().getPage().showView(EcoreBrowser.PROPERTY_VIEW_ID);
		} catch (final PartInitException e) {
			MoDiscoBrowserPlugin.logException(e);
		}
	}

	public ResourceSet getResourceSet() {
		return this.fResourceSet;
	}

	public void loadCustomizations(final List<MetamodelView> customizationsToLoad) {
		final CustomizationEngine customizationEngine = this.browserConfiguration
				.getCustomizationEngine();

		if (this.fCustomizationEngineChangeListener == null) {
			this.fCustomizationEngineChangeListener = new ChangeListener() {
				public void changed() {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							EcoreBrowser.this.browserConfiguration.getEditor().getMetaclassViewer()
									.refresh();
							EcoreBrowser.this.browserConfiguration.getEditor().refreshTreeDelayed();
						}
					});
				}
			};
			customizationEngine.addChangeListener(this.fCustomizationEngineChangeListener);
		}

		customizationEngine.clear();
		for (final MetamodelView customizationToLoad : customizationsToLoad) {
			customizationEngine.registerCustomization(customizationToLoad);
		}
		customizationEngine.loadCustomizations();
	}

	private void loadDefaultCustomization() {
		loadCustomizations(CustomizationsCatalog.getInstance().getRegistryDefaultCustomizations());
	}

	/** used by unit tests */
	protected TreeToolBar getInstancesToolBar() {
		return this.instancesToolBar;
	}
}
