/**
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Gregoire DUPE (Mia-Software) - initial API and implementation
 */
package org.eclipse.gmt.modisco.common.core.builder;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EventListener;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.resource.IMoDiscoResourceListener;
import org.eclipse.gmt.modisco.common.core.resource.MoDiscoResourceSet;
import org.eclipse.gmt.modisco.common.core.utils.ValidationUtils;
import org.osgi.framework.Bundle;

/**
 * @author Grgoire Dup
 */
public abstract class AbstractMoDiscoCatalog implements IMoDiscoResourceListener {

	private static List<AbstractMoDiscoCatalog> catalogs = new ArrayList<AbstractMoDiscoCatalog>();
	private final MoDiscoResourceSet resourceSet = MoDiscoResourceSet.getResourceSetSingleton();
	private final HashMap<String, EObject> nameToInstalledEObjectMap = new HashMap<String, EObject>();
	private final HashMap<String, EObject> nameToWorkspaceEObjectMap = new HashMap<String, EObject>();
	private final HashMap<String, EObject> uriToEObjectMap = new HashMap<String, EObject>();
	private final HashMap<EObject, Bundle> eObjectToBundleMap = new HashMap<EObject, Bundle>();
	private final File registryFile;
	private final HashMap<String, URI> uriMap = new HashMap<String, URI>();
	private final List<ModiscoCatalogChangeListener> changeListeners = new ArrayList<ModiscoCatalogChangeListener>();
	private final HashSet<IResource> nonValidFiles = new HashSet<IResource>();

	protected AbstractMoDiscoCatalog() {
		AbstractMoDiscoCatalog.getCatalogs().add(this);
		IPath absolutPath = getActivator().getStateLocation().append(getRegistryFile());
		this.registryFile = absolutPath.toFile();
		initInstalledRootObject();
		initWorkspaceRootObject();
	}

	protected abstract Plugin getActivator();

	private void initWorkspaceRootObject() {
		try {
			if (this.registryFile.exists()) {
				BufferedReader br = new BufferedReader(new FileReader(this.registryFile));
				String line = br.readLine();
				while (line != null) {
					URI uri = URI.createURI(line);
					String path = uri.toPlatformString(true);
					IResource querySetResource = ResourcesPlugin.getWorkspace().getRoot()
							.findMember(path);
					if (querySetResource instanceof IFile) {
						IFile querySetFile = (IFile) querySetResource;
						addWSFile(querySetFile);
					}
					line = br.readLine();
				}
				br.close();
			}
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR,
					getActivator().getBundle().getSymbolicName(), e.getMessage(), e);
			getActivator().getLog().log(status);
		}
	}

	/**
	 * Save method called by the plug-in activator
	 */
	public void save() {
		try {
			PrintStream ps = new PrintStream(this.registryFile);
			for (EObject rootObject : this.nameToWorkspaceEObjectMap.values()) {
				ps.println(getURI(getRootObjectName(rootObject)));
			}
			ps.close();
		} catch (FileNotFoundException e) {
			IStatus status = new Status(IStatus.ERROR,
					getActivator().getBundle().getSymbolicName(), e.getMessage(), e);
			getActivator().getLog().log(status);
		}

	}

	protected HashMap<String, EObject> getNameToInstalledEObjectMap() {
		return this.nameToInstalledEObjectMap;
	}

	protected HashMap<String, EObject> getNameToWorkspaceEObjectMap() {
		return this.nameToWorkspaceEObjectMap;
	}

	private void initInstalledRootObject() {
		if (getRegistryExtensionPoint() != null) {
			IConfigurationElement[] configs = Platform.getExtensionRegistry()
					.getConfigurationElementsFor(getRegistryExtensionPoint());
			for (IConfigurationElement config : configs) {
				String fileName = config.getAttribute("file"); //$NON-NLS-1$
				URI uri = URI.createURI("platform:/plugin/" //$NON-NLS-1$
						+ config.getContributor().getName() + "/" + fileName); //$NON-NLS-1$
				EObject rootObject = openResource(uri, null);
				if (rootObject != null) {
					Bundle bundle = Platform.getBundle(config.getContributor().getName());
					this.eObjectToBundleMap.put(rootObject, bundle);
					String name = getRootObjectName(rootObject);
					this.nameToInstalledEObjectMap.put(name, rootObject);
				}
			}
		}
	}

	/**
	 * This method returns the name of a given rootObject. This method must be
	 * implemented by {@link AbstractMoDiscoCatalog} sub classes.
	 * 
	 * @param rootObject
	 *            a resource root object
	 * @return rootObject name
	 */
	protected abstract String getRootObjectName(EObject rootObject);

	/**
	 * This method returns the id of the extension point that will be used to
	 * refer installed models which must be stored in the catalog . This method
	 * must be implemented by {@link AbstractMoDiscoCatalog} sub classes.
	 * 
	 * @return the extension point id
	 */
	protected abstract String getRegistryExtensionPoint();

	/**
	 * Contains the procedure to open a resource, check it, and add it into the
	 * specific maps.
	 * 
	 * @param uri
	 *            the resource uri to open
	 * @param file
	 *            the file containing the resource (markers will be applied on
	 *            this file).
	 * @return the root object of the resource
	 */
	protected EObject openResource(final URI uri, final IResource file) {
		EObject rootObject = null;
		Class<?> expectedClass = getRootClass();
		try {
			Resource resource = this.resourceSet.getResource(uri, this);
			if (ValidationUtils.validate(resource, file)) {
				if (resource.getContents().size() != 1) {
					throw new Exception("One and only one root is expected in a query model: " //$NON-NLS-1$
							+ resource.getContents().size() + " found."); //$NON-NLS-1$
				}
				EObject root = resource.getContents().get(0);
				if (expectedClass.isInstance(root)) {
					rootObject = root;
					this.uriToEObjectMap.put(uri.toString(), rootObject);
					this.uriMap.put(getRootObjectName(rootObject), uri);
					if (getMoDiscoSubProtocol() != null) {
						URI modiscoUri = URI.createURI("modisco:/" //$NON-NLS-1$
								+ getMoDiscoSubProtocol() + "/" //$NON-NLS-1$
								+ getRootObjectName(rootObject));
						Resource modiscoResource = this.resourceSet.getResource(modiscoUri, false);
						if (modiscoResource == null) {
							modiscoResource = this.resourceSet.createResource(modiscoUri);
						}
						modiscoResource.getContents().clear();
						modiscoResource.getContents().addAll(resource.getContents());
					}
				} else {
					Exception e = new Exception("Wrong kind of root in " //$NON-NLS-1$
							+ uri.toString() + " : " //$NON-NLS-1$
							+ root.getClass().getSimpleName());
					IStatus status = new Status(IStatus.ERROR, getActivator().getBundle()
							.getSymbolicName(), e.getMessage(), e);
					getActivator().getLog().log(status);
				}
			} else {
				this.nonValidFiles.add(file);
			}
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR,
					getActivator().getBundle().getSymbolicName(), "Failed to load: " + uri, e); //$NON-NLS-1$
			getActivator().getLog().log(status);
		}
		return rootObject;
	}

	/**
	 * This method returns the string representing the modisco sub protocol
	 * (modisco:/&lt;subprotocol&gt;/), that will be used to access the
	 * resources stored by the {@link AbstractMoDiscoCatalog} sub class. This
	 * method must be implemented by {@link AbstractMoDiscoCatalog} sub classes.
	 * 
	 * @return the string representing the modisco sub protocol
	 */
	protected abstract String getMoDiscoSubProtocol();

	/**
	 * This method returns the expected root element. This method must be
	 * implemented by {@link AbstractMoDiscoCatalog} sub classes.
	 */
	protected abstract Class<?> getRootClass();

	/**
	 * This methods returns the root objects of all the resources contained in
	 * the catalog.
	 * 
	 * @return root objects
	 */
	public final Collection<EObject> getAllRootObjects() {
		return getAllRootObjectMap().values();
	}

	protected Map<String, EObject> getAllRootObjectMap() {
		HashMap<String, EObject> allRootObject = new HashMap<String, EObject>();
		allRootObject.putAll(this.nameToInstalledEObjectMap);
		allRootObject.putAll(this.nameToWorkspaceEObjectMap);
		return allRootObject;
	}

	/**
	 * This method returns the root object of the resource having for name the
	 * "name" parameter value.
	 * 
	 * @param name
	 *            the name of a resource contained in the catalog
	 * @return a root object
	 */
	public final EObject getRootObject(final String name) {
		return getAllRootObjectMap().get(name);
	}

	/**
	 * This method is used by builders to add resources into the catalog.
	 * 
	 * @param declarationFile
	 *            the EMF resource file
	 * @return the root object of the resource
	 */
	public final EObject addWSFile(final IFile declarationFile) {
		EObject rootEObject = null;
		try {
			try {
				declarationFile.deleteMarkers(ValidationUtils.EMF_PROBLEM_MARKER, true,
						IResource.DEPTH_ZERO);
			} catch (CoreException e) {
				IStatus status = new Status(IStatus.ERROR, getActivator().getBundle()
						.getSymbolicName(), "An exception happened while removing markers", e); //$NON-NLS-1$
				getActivator().getLog().log(status);
			}
			String pathName = declarationFile.getProject().getName() + "/" //$NON-NLS-1$
					+ declarationFile.getProjectRelativePath().toString();
			URI rootObjectURI = URI.createPlatformResourceURI(pathName, false);
			rootEObject = openResource(rootObjectURI, declarationFile);
			if (rootEObject != null) {
				this.nameToWorkspaceEObjectMap.put(getRootObjectName(rootEObject), rootEObject);
				this.resourceSet.aResourceHasBeenLoaded(rootEObject.eResource());
				addNotify(rootEObject, declarationFile);
			}
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR,
					getActivator().getBundle().getSymbolicName(),
					"Failed to add a workspace file in " //$NON-NLS-1$
							+ this.getClass().getSimpleName() + " : " //$NON-NLS-1$
							+ declarationFile.getLocation().toString(), e);
			getActivator().getLog().log(status);
		}
		return rootEObject;
	}

	private void addNotify(final EObject rootObject, final IFile file) {
		for (ModiscoCatalogChangeListener listener : this.changeListeners) {
			listener.added(rootObject, file);
		}
	}

	/**
	 * This method is used by builders to remove resources from the catalog.
	 * 
	 * @param declarationFile
	 *            the EMF resource file
	 */
	public void removeWSFile(final IFile declarationFile) {
		String pathName = declarationFile.getProject().getName() + "/" //$NON-NLS-1$
				+ declarationFile.getProjectRelativePath().toString();
		URI rootObjectURI = URI.createPlatformResourceURI(pathName, false);
		EObject oldRootObject = this.uriToEObjectMap.get(rootObjectURI.toString());
		if (oldRootObject != null) {
			EPackage.Registry.INSTANCE.remove(getRootObjectNsUri(oldRootObject));
			this.nameToWorkspaceEObjectMap.remove(getRootObjectName(oldRootObject));
			this.uriToEObjectMap.remove(getRootObjectName(oldRootObject));
			if (oldRootObject.eResource() != null) {
				oldRootObject.eResource().unload();
			}
		}
		try {
			if (declarationFile.exists()) {
				declarationFile.deleteMarkers(ValidationUtils.EMF_PROBLEM_MARKER, true,
						IResource.DEPTH_ZERO);
			}
		} catch (CoreException e) {
			IStatus status = new Status(IStatus.ERROR,
					getActivator().getBundle().getSymbolicName(),
					"An exception happened while removing markers", e); //$NON-NLS-1$
			getActivator().getLog().log(status);
		}
		removeNotify(declarationFile);
	}

	/**
	 * This method returns the nsURI of a given rootObject. This method must be
	 * implemented by {@link AbstractMoDiscoCatalog} sub classes.
	 * 
	 * @param rootObject
	 *            a root eObject
	 * @return a nsURI
	 */
	protected abstract String getRootObjectNsUri(EObject rootObject);

	private void removeNotify(final IFile file) {
		for (ModiscoCatalogChangeListener listener : this.changeListeners) {
			listener.removed(file);
		}

	}

	/**
	 * This method returns the URI of the file that has been used to create a
	 * resource having another URI (given by the parameter value).
	 * 
	 * @param name
	 *            a resource URI
	 * @return the URI of a file
	 */
	public URI getURI(final String name) {
		return this.uriMap.get(name);
	}

	/** A listener for change notifications */
	public interface ModiscoCatalogChangeListener extends EventListener {
		void changed(final EObject eObject, final IFile file);

		void added(final EObject eObject, final IFile file);

		void removed(final IFile file);
	}

	/**
	 * This method is used to add a listener for catalog changes. This method is
	 * called by the views presenting the catalog contents
	 * 
	 * @param modiscoCatalogChangeListener
	 *            a listener
	 */
	public void addChangeListener(final ModiscoCatalogChangeListener modiscoCatalogChangeListener) {
		this.changeListeners.add(modiscoCatalogChangeListener);
	}

	/**
	 * This method is used to remove a catalog change listener. This method is
	 * called by the views presenting the catalog contents
	 * 
	 * @param modiscoCatalogChangeListener
	 *            listener to remove
	 */
	public void removeChangeListener(final ModiscoCatalogChangeListener modiscoCatalogChangeListener) {
		this.changeListeners.remove(modiscoCatalogChangeListener);

	}

	/**
	 * This method is called by builders to clean the catalog contents.
	 * 
	 * @param project
	 *            project to clean
	 */
	public void clean(final IProject project) {
		for (IResource resource : this.nonValidFiles) {
			try {
				if (resource != null) {
					resource.deleteMarkers(ValidationUtils.EMF_PROBLEM_MARKER, true,
							IResource.DEPTH_ONE);
				}
			} catch (CoreException e) {
				IStatus status = new Status(IStatus.WARNING, getActivator().getBundle()
						.getSymbolicName(), "An error happened while removing markers", e); //$NON-NLS-1$
				getActivator().getLog().log(status);
			}
		}
		this.nonValidFiles.clear();
		List<String> toBeRemovedList = new ArrayList<String>();
		List<IFile> toBeRemovedFileList = new ArrayList<IFile>();
		for (String rootObjectName : this.nameToWorkspaceEObjectMap.keySet()) {
			EObject rootObject = this.nameToWorkspaceEObjectMap.get(rootObjectName);
			if (rootObject.eResource() == null) {
				toBeRemovedList.add(rootObjectName);
			} else {
				URI uri = getURI(rootObjectName);
				if (uri.isPlatformResource()) {
					IFile file = (IFile) ResourcesPlugin.getWorkspace().getRoot().findMember(
							uri.toPlatformString(true));
					if (file != null && project == file.getProject()) {
						toBeRemovedFileList.add(file);
					}
				}
			}
		}
		for (String toBeRemoved : toBeRemovedList) {
			this.nameToWorkspaceEObjectMap.remove(toBeRemoved);
		}
		if (!toBeRemovedList.isEmpty()) {
			removeNotify(null);
		}
		for (IFile file : toBeRemovedFileList) {
			removeWSFile(file);
		}
		if (!toBeRemovedFileList.isEmpty()) {
			removeNotify(null);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void aListenedResourceHasChanged(final URI resourceUri, final URI dependingResourceURI) {
		IFile declarationFile = null;
		URI uri;
		if (dependingResourceURI.isPlatformResource()) {
			uri = dependingResourceURI;
			declarationFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
					new Path(uri.toPlatformString(true)));
		} else if (dependingResourceURI.scheme().equals("modisco")) { //$NON-NLS-1$
			uri = getURI(dependingResourceURI.segment(1));
			declarationFile = ResourcesPlugin.getWorkspace().getRoot().getFile(
					new Path(uri.toPlatformString(true)));
		} else {
			throw new RuntimeException("Unexpected uri: " //$NON-NLS-1$
					+ dependingResourceURI);
		}
		if (MoDiscoResourceSet.DEBUG) {
			IStatus status = new Status(IStatus.INFO, getActivator().getBundle().getSymbolicName(),
					"[" //$NON-NLS-1$
							+ this.getClass().getSimpleName()
							+ ".aListenedResourceHasChanged()] Reloading: file= " //$NON-NLS-1$
							+ declarationFile);
			getActivator().getLog().log(status);
		}
		addWSFile(declarationFile);
	}

	private final String getRegistryFile() {
		return this.getClass().getName() + "Registry"; //$NON-NLS-1$
	}

	/**
	 * This method returns the installed bundle containing the resource file
	 * from which the eObject parameter has been created
	 * 
	 * @param rootObject
	 *            an eObject (must be a resource root)
	 * @return the installed bundle containing eObject
	 */
	protected Bundle getBundle(final EObject rootObject) {
		return this.eObjectToBundleMap.get(rootObject);
	}

	public static List<AbstractMoDiscoCatalog> getCatalogs() {
		return AbstractMoDiscoCatalog.catalogs;
	}

}