/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.common.core.examples;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.Messages;
import org.eclipse.m2m.atl.common.ATLLaunchConstants;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.core.IExtractor;
import org.eclipse.m2m.atl.core.IInjector;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.ModelFactory;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.core.service.CoreService;
import org.eclipse.m2m.atl.core.service.LauncherService;
import org.eclipse.m2m.atl.core.ui.vm.asm.ASMModelWrapper;
import org.eclipse.m2m.atl.drivers.emf4atl.ASMEMFModel;
import org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel;

/**
 * @author Gabriel Barbier
 */
public class AtlTransformation {

	// resources location
	private final String kdmMMUri = "http://www.eclipse.org/MoDisco/kdm/action"; //$NON-NLS-1$
	private final String j2se5MMUri = "http://www.eclipse.org/MoDisco/J2SE5"; //$NON-NLS-1$

	public static final String TEST_IN_MODEL = "platform:/resource/test/test.in"; //$NON-NLS-1$

	public Resource testDetaillePourRegularVm(final URI j2se5SourceModelUri,
			final URL transformation, final URI kdmTargetModelUri) throws ATLCoreException {
		/*
		 * Attention: URIs used to indicate that the model is registered in the
		 * EMFRegistry must not have the "uri:" prefix anymore.
		 */

		/*
		 * Attention: this collection is very important; it associates the name
		 * of a metamodel to the name of the model handler to use.
		 */
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String j2se5MetaModelName = "j2se5"; //$NON-NLS-1$
		modelHandlers.put(j2se5MetaModelName, "EMF"); //$NON-NLS-1$
		String kdmMetaModelName = "kdm"; //$NON-NLS-1$
		modelHandlers.put(kdmMetaModelName, "EMF"); //$NON-NLS-1$

		/*
		 * Launcher initialization. Now all instances are provided by the
		 * CoreServices class and differences are made through strings.
		 * Thankfully, the ATLLaunchConstants class contains most of these
		 * constants.
		 */
		Map<String, Object> launcherOptions = new HashMap<String, Object>();
		launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);

		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);
		// launcher.initialize(Collections.<String, Object> emptyMap());

		/*
		 * Initialization of the old model handlers; they are now called
		 * injectors/extractors.
		 */
		ModelFactory factory = CoreService
				.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		/*
		 * Still concerning the old VM, we have to fill each model's options
		 * with the name of the model, its path, etc. There is a little
		 * redundancy, but that is required! So, we have to provide this
		 * information for all the models.
		 */
		// load the Java metamodel
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(j2se5MetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", j2se5MetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.j2se5MMUri); //$NON-NLS-1$
		IReferenceModel j2se5MM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(j2se5MM, this.j2se5MMUri);

		// load the Java model
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", j2se5SourceModelUri.toString()); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(j2se5MM, modelOptions);
		injector.inject(input, j2se5SourceModelUri.toString());
		launcher.addInModel(input, inModelName, j2se5MetaModelName);

		// load the KDM metamodel
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.kdmMMUri); //$NON-NLS-1$
		IReferenceModel kdmMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(kdmMM, this.kdmMMUri);
		// load the KDM model
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		if (kdmTargetModelUri != null) {
			modelOptions.put("path", kdmTargetModelUri.toString()); //$NON-NLS-1$
		} else {
			modelOptions
					.put(
							"path", j2se5SourceModelUri.trimFileExtension().appendFileExtension("kdm").toString()); //$NON-NLS-1$ //$NON-NLS-2$
		}
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(kdmMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, kdmMetaModelName);

		/*
		 * Encapsulation of the transformation in a new Thread in order to get
		 * an empty execution stack.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true"); //$NON-NLS-1$ //$NON-NLS-2$
		options.put("printExecutionTime", "true"); //$NON-NLS-1$ //$NON-NLS-2$

		Job transformationThread = new Job(Messages.AtlTransformation_14) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options, transformation
							.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}

		if (kdmTargetModelUri != null) {
			extractor.extract(outputInstance, kdmTargetModelUri.toString());
		}

		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		}
		return output;

	}

	private Resource resourceTemp;

	public Resource testSimplifieUtilisantLauncherServicePourRegularVm(
			final URI j2se5SourceModelUri, final URL transformation, final URI kdmTargetModelUri)
			throws ATLCoreException {
		/*
		 * This version seems simpler, but it does not provide an easy way to
		 * retrieve the result of a transformation. We will have to use a
		 * collection, or the initialization of an instance variable...
		 * 
		 * Attention: with this version, the target resource has to be saved by
		 * ATL.
		 */
		/**
		 * Launches a transformation using the given parameters. Process the
		 * injections, extractions.
		 * 
		 * @param mode
		 *            the launching mode
		 * @param monitor
		 *            the progress monitor
		 * @param launcher
		 *            the {@link ILauncher}
		 * @param inModels
		 *            a map defining the source models names: [model name,
		 *            reference model name]
		 * @param inoutModels
		 *            a map defining the inout models names: [model name,
		 *            reference model name]
		 * @param outModels
		 *            a map defining the target models names: [model name,
		 *            reference model name]
		 * @param paths
		 *            a map defining the location of each model name: [model
		 *            name, model location]
		 * @param options
		 *            a map of transformation options: [key, value]
		 * @param libraries
		 *            a map of libraries modules: [library name, module]
		 * @param modules
		 *            a list of modules to superimpose. The first module
		 *            overrides the next ones.
		 * @return the transformation result
		 */

		/*
		 * Attention: this collection is very important; it associates the name
		 * of a metamodel to the name of the model handler to use.
		 */
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String j2se5MetaModelName = "j2se5"; //$NON-NLS-1$
		modelHandlers.put(j2se5MetaModelName, "EMF"); //$NON-NLS-1$
		String kdmMetaModelName = "kdm"; //$NON-NLS-1$
		modelHandlers.put(kdmMetaModelName, "EMF"); //$NON-NLS-1$

		/*
		 * Launcher initialization. Now all instances are provided by the
		 * CoreServices class and differences are made through strings.
		 * Thankfully, the ATLLaunchConstants class contains most of these
		 * constants.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);

		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(options);
		// launcher.initialize(Collections.<String, Object> emptyMap());

		/* Model handling: only the models declared in the transformation */
		final Map<String, String> sourceModels = new HashMap<String, String>();
		sourceModels.put("IN", "j2se5"); //$NON-NLS-1$ //$NON-NLS-2$
		final Map<String, String> targetModels = new HashMap<String, String>();
		targetModels.put("OUT", "kdm"); //$NON-NLS-1$ //$NON-NLS-2$
		/* Model paths: all the models concerned by the transformation */
		final Map<String, String> modelPaths = new HashMap<String, String>();
		modelPaths.put("j2se5", this.j2se5MMUri); //$NON-NLS-1$
		modelPaths.put("IN", j2se5SourceModelUri.toString()); //$NON-NLS-1$
		modelPaths.put("kdm", this.kdmMMUri); //$NON-NLS-1$
		if (kdmTargetModelUri != null) {
			modelPaths.put("OUT", kdmTargetModelUri.toString()); //$NON-NLS-1$
		} else {
			modelPaths
					.put(
							"OUT", j2se5SourceModelUri.trimFileExtension().appendFileExtension("kdm").toString()); //$NON-NLS-1$ //$NON-NLS-2$
		}

		/*
		 * Encapsulation of the transformation in a new Thread in order to get
		 * an empty execution stack.
		 */
		options.put("continueAfterError", "true"); //$NON-NLS-1$ //$NON-NLS-2$
		options.put("printExecutionTime", "true"); //$NON-NLS-1$ //$NON-NLS-2$

		Job transformationThread = new Job(Messages.AtlTransformation_33) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					final InputStream[] modules = { transformation.openStream() };
					final Map<String, InputStream> libraries = Collections.emptyMap();
					Object resultModel = LauncherService.launch(ILauncher.RUN_MODE, monitor,
							launcher, sourceModels, Collections.<String, String> emptyMap(),
							targetModels, modelPaths, options, libraries, modules);
					if (resultModel instanceof ASMModelWrapper) {
						ASMModelWrapper wrapper = (ASMModelWrapper) resultModel;
						ASMModel model = wrapper.getAsmModel();
						if (model instanceof ASMEMFModel) {
							AtlTransformation.this.resourceTemp = ((ASMEMFModel) model).getExtent();
						}
					}
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				} catch (ATLCoreException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}

		Resource output = this.resourceTemp;

		return output;

	}

	public void testDetaillePourEmfVm(final URL asmURL, final IFile file) throws ATLCoreException,
			IOException {
		// Getting the launcher
		ILauncher launcher = null;
		launcher = CoreService.getLauncher(ATLLaunchConstants.EMF_VM_NAME);
		launcher.initialize(Collections.<String, Object> emptyMap());
		// Defaults
		ModelFactory factory = CoreService
				.createModelFactory(launcher.getDefaultModelFactoryName());
		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		// Metamodels
		IReferenceModel umlMetamodel = factory.newReferenceModel();
		injector.inject(umlMetamodel, "http://www.eclipse.org/uml2/2.1.0/UML"); //$NON-NLS-1$
		IReferenceModel refiningTraceMetamodel = factory.getBuiltInResource("RefiningTrace.ecore"); //$NON-NLS-1$

		// Creating models
		IModel refiningTraceModel = factory.newModel(refiningTraceMetamodel);
		IModel umlModel = factory.newModel(umlMetamodel);

		// Loading existing model
		injector.inject(umlModel, file.getFullPath().toString());

		// Launching
		launcher.addOutModel(refiningTraceModel, "refiningTrace", "RefiningTrace"); //$NON-NLS-1$ //$NON-NLS-2$
		launcher.addInOutModel(umlModel, "IN", "UML"); //$NON-NLS-1$ //$NON-NLS-2$

		launcher.launch(ILauncher.RUN_MODE, new NullProgressMonitor(), Collections
				.<String, Object> emptyMap(), asmURL.openStream());

		// Saving model
		extractor.extract(umlModel, file.getFullPath().toString());

	}
}
