/*******************************************************************************
 * Copyright (c) 2008, 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *    Fabien Giquel (Mia-Software)
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.core.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Comparator;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.gmt.modisco.common.core.CommonModiscoActivator;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;

/**
 * @author Gabriel Barbier
 * 
 */
public final class FolderUtils {
	private static final int COPY_BUFFER_SIZE = 512 * 1024;

	private static boolean debug = false;

	private static FolderUtils.ConfigurationManagementFilter configurationManagementFilter;

	private static class ConfigurationManagementFilter implements FilenameFilter {
		private final String filterName = "CVS"; //$NON-NLS-1$

		/**
		 * Tests if a specified file should be included in a file list.
		 * 
		 * 
		 * @param dir
		 *            the directory in which the file was found.
		 * @param name
		 *            the name of the file.
		 * 
		 * @return <code>true</code> if and only if the name should be included
		 *         in the file list; <code>false</code> otherwise.
		 */
		public boolean accept(final File dir, final String name) {
			boolean result = false;
			if (!name.equals(this.filterName)) {
				result = true;
			}
			return result;
		}
	}

	private FolderUtils() {
		// prevent instantiation
	}

	public static final void clearFolder(final File dirtyFolder) {
		assert dirtyFolder != null;
		assert dirtyFolder.exists();
		assert dirtyFolder.isDirectory();
		/*
		 * pour supprimer le contenu du rpertoire, il faut faire une rcursion
		 * sur tous les objets (File) de ce rpertoire. Dans le cas d'un
		 * rpertoire, il faut d'abord supprimer son propre contenu avant de
		 * pouvoir le supprimer  son tour. Dans le cas d'un fichier simple, il
		 * suffit de le supprimer.
		 */
		File[] files = dirtyFolder.listFiles();
		for (File file : files) {
			if (file.isDirectory()) {
				// c'est un rpertoire
				clearFolder(file);
			}
			file.delete();
		}
	}

	/**
	 * This method compares two folders (in fact the contents of these two
	 * folders). It is recursive into each folder
	 */
	public static final boolean compareFolders(final File folderSource, final File folderTarget) {
		return compareFolders(folderSource, folderTarget,
				FolderUtils.configurationManagementFilter, FolderUtils.getDefaultFileComparator());
	}

	public static final boolean compareFolders(final File folderSource, final File folderTarget,
			final FilenameFilter filter) {
		return compareFolders(folderSource, folderTarget, filter, FolderUtils
				.getDefaultFileComparator());
	}

	public static final boolean compareFolders(final File folderSource, final File folderTarget,
			final FilenameFilter filter, final Comparator<File> fileComparison) {
		assert ((folderSource != null) && (folderTarget != null));
		assert ((folderSource.isDirectory()) && (folderTarget.isDirectory()));
		boolean result = false;
		if (folderSource.equals(folderTarget)) {
			result = true;
		} else {
			result = recursiveCompareFolders(folderSource, folderTarget, filter, fileComparison);
			if (FolderUtils.debug && !result) {
				MoDiscoLogger
						.logError(
								"folders " + folderSource.getName() + " and " + folderTarget.getName() + " are not equal.", CommonModiscoActivator.getDefault()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
		}
		return result;
	}

	public static final boolean compareFiles(final File source, final File target) {
		return compareFiles(source, target, getDefaultFileComparator());
	}

	private static final Comparator<File> getDefaultFileComparator() {
		return new Comparator<File>() {
			public int compare(final File source, final File target) {
				boolean result = true;
				if (!source.getName().equals(target.getName())) {
					result = false;
				} else {
					/*
					 * Compare file line by line
					 */
					try {
						BufferedReader sourceReader = new BufferedReader(new FileReader(source));
						BufferedReader targetReader = new BufferedReader(new FileReader(target));
						String sourceLine = sourceReader.readLine();
						String targetLine = targetReader.readLine();
						while ((sourceLine != null) && (targetLine != null)) {
							result = result && sourceLine.equals(targetLine);
							sourceLine = sourceReader.readLine();
							targetLine = targetReader.readLine();
						}
						if ((sourceLine != null) || (targetLine != null)) {
							result = false;
							MoDiscoLogger
									.logError(
											"These files do not have the same number of lines.", CommonModiscoActivator.getDefault()); //$NON-NLS-1$
						}
					} catch (FileNotFoundException e) {
						result = false;
					} catch (IOException e) {
						result = false;
					}
				}
				if (result) {
					return 0;
				}
				return -1;
			}
		};
	}

	public static final boolean compareFiles(final File source, final File target,
			final Comparator<File> fileComparator) {
		assert ((source.isFile()) && (target.isFile()));
		boolean result = (fileComparator.compare(source, target) == 0);
		if (FolderUtils.debug && result) {
			MoDiscoLogger
					.logInfo(
							"Files " + source.getName() + " and " + target.getName() + " are equal.", CommonModiscoActivator.getDefault()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		return result;
	}

	public static final String getFileContent(final File source) {
		StringBuilder result = new StringBuilder();
		try {
			BufferedReader sourceReader = new BufferedReader(new FileReader(source));
			String sourceLine = sourceReader.readLine();
			while (sourceLine != null) {
				result.append(sourceLine);
				result.append("\n"); //$NON-NLS-1$
				sourceLine = sourceReader.readLine();
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return result.toString();
	}

	/**
	 * This method compares two folders (in fact the contents of these two
	 * folders). It is recursive into each folder
	 */
	private static final boolean recursiveCompareFolders(final File folderSource,
			final File folderTarget, final FilenameFilter filter,
			final Comparator<File> fileComparison) {
		if (FolderUtils.debug) {
			MoDiscoLogger
					.logError(
							"comparison of " + folderSource.getName() + " and " + folderTarget.getName(), CommonModiscoActivator.getDefault()); //$NON-NLS-1$ //$NON-NLS-2$
		}
		boolean result = true;

		File[] sourceContents = folderSource.listFiles(filter);
		File[] targetContents = folderTarget.listFiles(filter);
		if (sourceContents.length != targetContents.length) {
			result = false;
			if (FolderUtils.debug) {
				MoDiscoLogger.logError("folders " + folderSource.getName() + " and " //$NON-NLS-1$//$NON-NLS-2$
						+ folderTarget.getName() + " do not have the same number of children (" //$NON-NLS-1$
						+ sourceContents.length + ", " + targetContents.length + ")", //$NON-NLS-1$ //$NON-NLS-2$
						CommonModiscoActivator.getDefault());
			}
		} else {
			for (File sourceContent : sourceContents) {
				/*
				 * pour chaque objet File du rpertoire courant, il faut tester
				 * si il existe son quivalent dans le contenu du rpertoire
				 * cible. La recherche est base sur le nom (path name), il peut
				 * donc y avoir des problmes si un rpertoire et un fichier
				 * porte le mme nom.
				 */
				File targetContent = getCorrespondingTargetContent(sourceContent, targetContents);
				if (targetContent == null) {
					result = false;
					if (FolderUtils.debug) {
						MoDiscoLogger
								.logError(
										"There is no corresponding element in target folder for " + sourceContent.getName(), CommonModiscoActivator.getDefault()); //$NON-NLS-1$
					}
				} else {
					/*
					 * il y a maintenant deux cas  distinguer : le cas d'un
					 * fichier simple et le cas d'un rpertoire. Dans le cas
					 * d'un rpertoire, il suffit de faire la rcursion. Dans le
					 * cas d'un fichier, il faut comparer le contenu des
					 * fichiers.
					 */
					if (sourceContent.isDirectory()) {
						boolean subResult = recursiveCompareFolders(sourceContent, targetContent,
								filter, fileComparison);
						result = result && subResult;

						if (FolderUtils.debug && !subResult) {
							MoDiscoLogger
									.logError(
											"folders " + sourceContent.getName() + " and " + targetContent.getName() + " are not equal.", CommonModiscoActivator.getDefault()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						}
					} else {
						boolean subResult = compareFiles(sourceContent, targetContent,
								fileComparison);
						result = result && subResult;
						if (FolderUtils.debug && !subResult) {
							MoDiscoLogger
									.logError(
											"files " + sourceContent.getName() + " and " + targetContent.getName() + " are not equal.", CommonModiscoActivator.getDefault()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						}
					}
				}
			}
		}
		return result;
	}

	private static final File getCorrespondingTargetContent(final File sourceContent,
			final File[] targetContents) {
		File targetContent = null;
		for (File temp : targetContents) {
			if (sourceContent.getName().equals(temp.getName())) {
				targetContent = temp;
			}
		}
		return targetContent;
	}

	/**
	 * Copies the source directory to the target directory. The target is
	 * created if it does not exist.
	 * 
	 * @param srcDir
	 * @param destDir
	 * @throws IOException
	 */
	public static final void copyDirectory(final File srcDir, final File destDir)
			throws IOException {

		if (!destDir.exists()) {
			destDir.mkdirs();
		}
		File[] filesList = srcDir.listFiles();
		File dest;
		// Copies each file and directory, one by one
		for (File src : filesList) {
			dest = new File(destDir.getPath() + File.separator + src.getName());
			if (src.isDirectory()) {
				copyDirectory(src, dest);
			} else {
				copyFile(src, dest);
			}
		}

	}

	/**
	 * Copies the source file to the target file.
	 * 
	 * @return <code>true</code> if successful, <code>false</code> otherwise
	 */
	public static final boolean copyFile(final File source, final File destination) {
		boolean result = false;
		FileInputStream sourceFile = null;
		FileOutputStream destinationFile = null;
		try {
			// File creation
			destination.createNewFile();
			sourceFile = new FileInputStream(source);
			destinationFile = new FileOutputStream(destination);
			// 0.5 MB buffer for reading
			byte[] buffer = new byte[FolderUtils.COPY_BUFFER_SIZE];
			int nbRead;
			while ((nbRead = sourceFile.read(buffer)) != -1) {
				destinationFile.write(buffer, 0, nbRead);
			}

			// Copied
			result = true;
		} catch (java.io.FileNotFoundException f) {
			result = false;
		} catch (java.io.IOException e) {
			result = false;
		} finally {
			try {
				if (sourceFile != null) {
					sourceFile.close();
				}
				if (destinationFile != null) {
					destinationFile.close();
				}
			} catch (Exception e) {
				result = false;
			}
		}
		return result;
	}

	/**
	 * Recursively delete a directory.
	 * 
	 * @param directory
	 *            directory to delete
	 * @throws IOException
	 *             in case deletion is unsuccessful
	 */
	public static final void deleteDirectory(final File directory) throws IOException {
		if (!directory.exists()) {
			return;
		}

		clearFolder(directory);
		if (directory.list().length == 0) {
			// delete directory
			if (!directory.delete()) {
				String message = "Unable to delete directory " + directory + "."; //$NON-NLS-1$ //$NON-NLS-2$
				throw new IOException(message);
			}
		}
	}

	public static void createFolder(final IFolder folder) throws CoreException {
		if (!folder.getParent().exists()) {
			createFolder((IFolder) folder.getParent());
		}
		folder.create(true, true, new NullProgressMonitor());
	}

	/**
	 * write a new content into a given file.
	 * 
	 * @param source
	 *            file. It will be created if it does not exist.
	 * @param newContent
	 * @throws IOException
	 */
	public static final void writeFileContent(final File source, final String newContent)
			throws IOException {
		FileWriter fw = new FileWriter(source, false);
		fw.write(newContent);
		fw.close();
	}
}
