/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.custom.ui.views;

import java.net.URI;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.ui.URIEditorInput;
import org.eclipse.emf.common.util.EList;
import org.eclipse.gmt.modisco.infra.browser.custom.MetamodelView;
import org.eclipse.gmt.modisco.infra.browser.custom.core.CustomizationsCatalog;
import org.eclipse.gmt.modisco.infra.browser.custom.core.CustomizationsCatalog.CustomizationChangeListener;
import org.eclipse.gmt.modisco.infra.browser.custom.ui.Activator;
import org.eclipse.gmt.modisco.infra.browser.custom.ui.Messages;
import org.eclipse.gmt.modisco.infra.browser.util.ColorProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

/**
 * An Eclipse view that displays a list of available MoDisco Browser
 * Customizations. Clicking on a customization opens it in the customization
 * editor.
 */
public class CustomizationsView extends ViewPart implements CustomizationChangeListener {

	private static final int QUERY_SETS_COLUMN_WIDTH = 400;
	private static final int METAMODEL_COLUMN_WIDTH = 300;
	private static final int NAME_COLUMN_WIDTH = 200;
	private TableViewer fViewer;
	private TableViewerColumn nameColumn;
	private TableViewerColumn metamodelColumn;
	private TableViewerColumn querySetsColumn;

	private Job refreshJob = null;

	public CustomizationsView() {
	}

	@Override
	public void createPartControl(final Composite parent) {
		parent.setLayout(new FillLayout());

		this.fViewer = new TableViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE
				| SWT.FULL_SELECTION);
		this.fViewer.getTable().setLinesVisible(true);
		this.fViewer.getTable().setHeaderVisible(true);

		this.fViewer.addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				final IStructuredSelection selection = (IStructuredSelection) CustomizationsView.this.fViewer
						.getSelection();
				final MetamodelView metamodelView = (MetamodelView) selection.getFirstElement();
				// open customization in editor
				try {
					final URI uri = new URI(metamodelView.getLocation());
					if ("file".equals(uri.getScheme())) { //$NON-NLS-1$
						IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
								.getActivePage(), new URI(metamodelView.getLocation()),
								Activator.CUSTOMIZATION_EDITOR_ID, true);
					} else {
						final URIEditorInput uriEditorInput = new URIEditorInput(
								org.eclipse.emf.common.util.URI.createURI(uri.toString()));
						IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow()
								.getActivePage(), uriEditorInput,
								Activator.CUSTOMIZATION_EDITOR_ID, true);
					}

				} catch (final Exception e) {
					Activator.logException(e);
				}
			}
		});

		this.nameColumn = new TableViewerColumn(this.fViewer, SWT.NONE);
		this.nameColumn.getColumn().setText(Messages.CustomizationsView_nameColumn);
		this.nameColumn.getColumn().setWidth(CustomizationsView.NAME_COLUMN_WIDTH);
		this.nameColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				final MetamodelView metamodelView = (MetamodelView) element;
				return metamodelView.getName();
			}

			@Override
			public Color getForeground(final Object element) {
				final MetamodelView metamodelView = (MetamodelView) element;
				if (!metamodelView.getLocation().startsWith("file:/")) { //$NON-NLS-1$
					return ColorProvider.getInstance().getExternalResourceColor();
				}
				return super.getForeground(element);
			}
		});
		addSorter(this.nameColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				final MetamodelView view1 = (MetamodelView) e1;
				final MetamodelView view2 = (MetamodelView) e2;
				return view1.getName().compareTo(view2.getName());
			}
		});

		this.metamodelColumn = new TableViewerColumn(this.fViewer, SWT.NONE);
		this.metamodelColumn.getColumn().setText(Messages.CustomizationsView_metamodelColumn);
		this.metamodelColumn.getColumn().setWidth(CustomizationsView.METAMODEL_COLUMN_WIDTH);
		this.metamodelColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				final MetamodelView metamodelView = (MetamodelView) element;
				return metamodelView.getMetamodelURI();
			}
		});
		addSorter(this.metamodelColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				final MetamodelView view1 = (MetamodelView) e1;
				final MetamodelView view2 = (MetamodelView) e2;
				return view1.getMetamodelURI().compareTo(view2.getMetamodelURI());
			}
		});

		this.querySetsColumn = new TableViewerColumn(this.fViewer, SWT.NONE);
		this.querySetsColumn.getColumn().setText(Messages.CustomizationsView_querySetsColumn);
		this.querySetsColumn.getColumn().setWidth(CustomizationsView.QUERY_SETS_COLUMN_WIDTH);
		this.querySetsColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				final MetamodelView metamodelView = (MetamodelView) element;
				if (metamodelView.isAllQuerySetsAvailable()) {
					return Messages.CustomizationsView_allQuerySets;
				}
				final EList<String> availableQuerySets = metamodelView.getAvailableQuerySets();
				final StringBuilder builder = new StringBuilder();
				for (int i = 0; i < availableQuerySets.size(); i++) {
					builder.append(availableQuerySets.get(i));
					if (i != availableQuerySets.size() - 1) {
						builder.append(", "); //$NON-NLS-1$
					}
				}
				return builder.toString();
			}
		});

		// locationColumn = new TableViewerColumn(viewer, SWT.NONE);
		// locationColumn.getColumn().setText("Location");
		// locationColumn.getColumn().setWidth(400);
		// locationColumn.setLabelProvider(new ColumnLabelProvider() {
		// @Override
		// public String getText(Object element) {
		// MetamodelView metamodelView = (MetamodelView) element;
		// return metamodelView.getLocation();
		// }
		// });

		this.fViewer.setContentProvider(new IStructuredContentProvider() {
			public Object[] getElements(final Object inputElement) {
				@SuppressWarnings("unchecked")
				final List<MetamodelView> metamodelViews = (List<MetamodelView>) inputElement;
				return metamodelViews.toArray();
			}

			public void inputChanged(final Viewer viewer, final Object oldInput,
					final Object newInput) {
			}

			public void dispose() {
			}
		});

		// getSite().setSelectionProvider(viewer);
		refresh(false);

		CustomizationsCatalog.getInstance().addChangeListener(this);
	}

	private void addSorter(final TableViewerColumn column, final ViewerSorter viewerSorter) {
		column.getColumn().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				if (CustomizationsView.this.fViewer.getSorter() == null
						|| CustomizationsView.this.fViewer.getSorter() != viewerSorter) {
					CustomizationsView.this.fViewer.setSorter(viewerSorter);
					CustomizationsView.this.fViewer.getTable().setSortColumn(column.getColumn());
					CustomizationsView.this.fViewer.getTable().setSortDirection(SWT.UP);
				} else {
					CustomizationsView.this.fViewer.setSorter(null);
					CustomizationsView.this.fViewer.getTable().setSortDirection(SWT.NONE);
				}
			}
		});

	}

	@Override
	public void setFocus() {
		this.fViewer.getControl().setFocus();
	}

	@Override
	public void dispose() {
		super.dispose();
		CustomizationsCatalog.getInstance().removeChangeListener(this);
	}

	/** Called when a customization changes */
	public void changed(final MetamodelView changedCustomization, final IFile file) {
		refresh(true);
	}

	/** Called when a customization is added */
	public void added(final MetamodelView addedCustomization, final IFile file) {
		refresh(true);
	}

	/** Called when a customization is removed */
	public void removed(final IFile customizationFile) {
		refresh(true);
	}

	/** Optionally delayed refresh */
	private void refresh(final boolean delayed) {
		if (this.refreshJob == null) {
			this.refreshJob = new Job(Messages.CustomizationsView_jobName_RefreshCustomizationsView) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							CustomizationsView.this.fViewer.setInput(CustomizationsCatalog
									.getInstance().getAllCustomizations());
							CustomizationsView.this.nameColumn.getColumn().pack();
							CustomizationsView.this.metamodelColumn.getColumn().pack();
							CustomizationsView.this.querySetsColumn.getColumn().pack();
							CustomizationsView.this.fViewer.refresh();
						}
					});
					return Status.OK_STATUS;
				}
			};
		}
		if (delayed) {
			// delayed until it stops changing
			this.refreshJob.cancel();
			this.refreshJob.setPriority(Job.DECORATE);
			final int delay = 500;
			this.refreshJob.schedule(delay);
		} else {
			// force synchronous execution
			this.refreshJob.schedule();
			try {
				this.refreshJob.join();
			} catch (final InterruptedException e) {
				Activator.logException(e);
			}
		}
	}

}
