/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.workflow.controller;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.internal.ui.launchConfigurations.LaunchGroupExtension;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.gmt.modisco.workflow.Activator;
import org.eclipse.gmt.modisco.workflow.Constants;
import org.eclipse.gmt.modisco.workflow.MoDiscoWorkflowLaunchConstants;
import org.eclipse.gmt.modisco.workflow.WorkflowExecutionInterface;
import org.eclipse.gmt.modisco.workflow.WorkflowModelInterface;
import org.eclipse.gmt.modisco.workflow.actions.LaunchDriversWizard;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.Element;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.ModiscoworkflowFactory;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.ModiscoworkflowPackage;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.Work;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.WorkParameter;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.WorkParameterStringListValue;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.WorkParameterStringValue;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.WorkParameterValue;
import org.eclipse.gmt.modisco.workflow.modiscoworkflow.Workflow;
import org.eclipse.gmt.modisco.workflow.ui.common.AdditionalButtonsComposite;
import org.eclipse.gmt.modisco.workflow.ui.common.CreatePluginProject;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jdt.core.JavaCore;

/**
 * This class defines one controller which control all actions proceed to user.
 * It makes the link between drivers and workflow engine.
 * @author Nicolas Payneau
 */
@SuppressWarnings("restriction")
public class ControllerWorkflowTransformation {

	private AdditionalButtonsComposite additionalButton;
	private Map<String,WorkflowModelInterface> workflowInterfaceMap;
	private Map<String,WorkflowExecutionInterface> workflowExecutionInterfaceMap;
	private LaunchDriversWizard wizardDrivers;
	private ControllerDriverPage controllerDriver;
	private int editionIndice;
	private LaunchGroupExtension group;
	private String currentIdWorkflow;
	private String nameRoot;
	
	/**
	 * Class constructor
	 */
	public ControllerWorkflowTransformation(String idWorkflow, String nameRoot){
		this.nameRoot = nameRoot;
		this.workflowExecutionInterfaceMap = new HashMap<String, WorkflowExecutionInterface>();
		this.workflowInterfaceMap = new HashMap<String, WorkflowModelInterface>();
		this.controllerDriver = new ControllerDriverPage();
		this.wizardDrivers = new LaunchDriversWizard(this,this.controllerDriver);
		this.currentIdWorkflow = idWorkflow;
		this.initializeWorkflow();
	}
	
	/**
	 * Open the wizard to add a new MoDiscoWork in the MoDiscoWorkflow
	 */
	public void newElement() {
		this.wizardDrivers.run(null,Constants.addMode);
		refresh(-1);
	}
	
	/**
	 * Open the wizard to edit a present element in the MoDiscoWorkflow
	 * @param index Element selected in the table
	 * @return true if the element found otherwise false.
	 */
	public Constants editElement(int index) {
		Constants status = Constants.Ok;
		this.editionIndice = index;
		Element modiscoWork = getComponent(index);
		if(modiscoWork!=null){
			if(!modiscoWork.getType().equals(MoDiscoWorkflowLaunchConstants.launchTypeMoDiscoWorkflow)){
				ILaunchConfiguration conf = getLaunchConfigurationFromName(modiscoWork.getName());
				ILaunchConfigurationWorkingCopy config = null;
				try {
					config = conf.getWorkingCopy();
				} catch (CoreException e) {
					e.printStackTrace();
				}
				if(config!=null){
					this.controllerDriver.setConfigurationEdition(config);
					this.controllerDriver.setTypeEdition(modiscoWork.getType());
					this.wizardDrivers.run(null,Constants.editMode);
				}else{
					status = Constants.LaunchConfigurationNotFound;
				}
			}else{
				status = Constants.MoDiscoWorkflowSelected;
			}
		}
		refresh(index);
		return status;
	}
	
	/**
	 * Remove a element in the MoDiscoWorkflow
	 * @param tableIndex index element selected in the table
	 */
	public void removeFromWorkflow(int tableIndex){
		List<Element> modiscoWorkList = getAllMoDiscoElement();
		Element modisco = modiscoWorkList.get(tableIndex);
		this.workflowInterfaceMap.get(this.currentIdWorkflow).removeComponent(tableIndex);
		updateAllElementIndex(modisco.getIndex());
		refresh(-1);
	}
	
	/**
	 * Allows to update all element index after have deleted one element
	 * @param modiscoWorkIndex index of the MoDiscoWork deleted
	 */
	private void updateAllElementIndex(int elementIndexDeleted){
		int i=0;
		for(Element component : getAllMoDiscoElement()){
			if(component.getIndex()>elementIndexDeleted){
				Element modisco = (Element)this.workflowInterfaceMap.get(this.currentIdWorkflow).getComponent(i);
				modisco.setIndex(modisco.getIndex()-1);
				setComponent(i,modisco);
			}
			i++;
		}
	}
	
	/**
	 * Remove all occurrences of MoDiscoWork selected and its launcher
	 * @param indexList List of index. It corresponds to the selected MoDiscoWork and all these occurrences in the table
	 */
	public void removeFromLauncher(List<Integer> indexList){
		try {
			//0 because they have all the same of ILaunchConfiguration
			Element modiscoWork = getComponent(indexList.get(0));
			ILaunchConfiguration config = getLaunchConfigurationFromName(modiscoWork.getName());
			if(config.exists()){
				ILaunchConfigurationWorkingCopy wc = config.getWorkingCopy();
				if(wc!=null){
					wc.getOriginal().delete();
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		for(int index : indexList){
			removeFromWorkflow(index);
		}
		refresh(-1);
	}
	
	/**
	 * Allows to update element index to up the element selected in the MoDiscoWorkflow
	 * @param tableIndex index of table selected
	 */
	public void upElement(int tableIndex) {
		this.workflowInterfaceMap.get(this.currentIdWorkflow).upMoDisoWork(tableIndex);
	}
	
	/**
	 * Allows to update element index to down the element selected in the MoDiscoWorkflow
	 * @param tableIndex index of table selected
	 */
	public void downElement(int tableIndex) {
		this.workflowInterfaceMap.get(this.currentIdWorkflow).downMoDiscoWork(tableIndex);
	}
	
	/**
	 * Open the wizard to load a element in the MoDiscoWorkflow
	 */
	public void loadElement() {
		this.wizardDrivers.run(null,Constants.loadMode);
		refresh(-1);
	}
	
	/**
	 * Allows to add a MoDiscoWork in the MoDiscoWorkflow
	 * @param type Launcher type
	 * @param index	Index in the MoDiscoWorkflow
	 * @param name Transformation name
	 * @param configuration The configuration
	 * @param refresh true if it is necessary to refresh otherwise false
	 */
	public void addElement(String type, final int index, String name, boolean refresh){
		boolean loop = false;
		int indexLocal = index;
		if(indexLocal==-1){
			indexLocal = getAllMoDiscoElement().size();
		}
		Element modisco;
		if(type.equals(MoDiscoWorkflowLaunchConstants.launchTypeMoDiscoWorkflow)){
			modisco = ModiscoworkflowFactory.eINSTANCE.createWorkflow();
			((Workflow)modisco).getElements().addAll(createElementsOfWorkflow(name,type));
			loop = verifyLoops(((Workflow)modisco).getElements(),this.nameRoot);
		}else{
			modisco = ModiscoworkflowFactory.eINSTANCE.createWork();
			List<WorkParameter> parameters = this.controllerDriver.getDriverFromLaunchType(type).getParameters(this.getLaunchConfigurationFromName(name));
			if(parameters!=null){
				((Work)modisco).getParameters().clear();
				((Work)modisco).getParameters().addAll(parameters);
			}
		}
		modisco.setIndex(indexLocal);
		modisco.setName(name);
		modisco.setType(type);
		if(loop==false&&modisco.getName()!=this.nameRoot){
			this.workflowInterfaceMap.get(this.currentIdWorkflow).addComponentWorkflow(modisco);
			if(refresh){
				refresh(-1);
			}
		}else{
			this.additionalButton.dialogLoop();
		}
	}

	/**
	 * Allows to verify the loop in the MoDiscoWorkflow
	 * @param elements All elements contained in MoDiscoWorkflow
	 * @param name The name of root
	 * @return True if MoDiscoWorkflow contains a loop otherwise false
	 */
	private boolean verifyLoops(List<Element> elements, String name) {
		boolean loop = false;
		for(Element elmt : elements){
			if(elmt instanceof Workflow){
				if(elmt.getName().equals(name)){
					loop=true;
					break;
				}else{
					loop=verifyLoops(((Workflow)elmt).getElements(),name);
				}
			}
		}
		return loop;
	}
	
	/**
	 * Load the plug-in to refer the point extension Constant.nameExtensionPointWorkflow
	 */
	private void initializeWorkflow(){
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint(Constants.nameExtensionPointWorkflow); 
		
		for (IExtension extension : point.getExtensions()) {
			for (IConfigurationElement element : extension.getConfigurationElements()) {
				//We take into account the first extension
				if(element.getName().equals(Constants.workflowCoreExtensionPointWorkflow)){
					try {
						this.workflowInterfaceMap.put(extension.getUniqueIdentifier(),(WorkflowModelInterface)element.createExecutableExtension("class")); //$NON-NLS-1$
					} catch (Throwable e) {
						e.printStackTrace();
					}
				}
				if(element.getName().equals(Constants.workflowExecutionExtensionPointWorkflow)){
					try {
						this.workflowExecutionInterfaceMap.put(extension.getUniqueIdentifier(),(WorkflowExecutionInterface)element.createExecutableExtension("class")); //$NON-NLS-1$
					} catch (Throwable e) {
						e.printStackTrace();
					}
				}
			}
		}
	}
	
	/**
	 * Refresh the view (Table of MoDiscoWork)
	 * @param index index for the next selection
	 */
	public void refresh(int index){
		this.additionalButton.getTable().removeAll();
		List<Element> modiscoWorkList = getAllMoDiscoElement();
		for(Element component : modiscoWorkList){
			this.additionalButton.createItemTable(component.getName(),component.getType());
		}
		if(index!=-1){
			this.additionalButton.getTable().select(index);
		}else{
			this.additionalButton.getTable().select(this.additionalButton.getTable().getItemCount()-1);
		}
		this.additionalButton.refreshTable();
	}
	
	/**
	 * Allows to retrieve an instance of {@link Element} corresponding at the item selected in the table
	 * @param tableIndex Index of item selected
	 * @return an instance of {@link Element} corresponding at the item selected in the table
	 */
	public Element getComponent(int tableIndex){
		return (Element)this.workflowInterfaceMap.get(this.currentIdWorkflow).getComponent(tableIndex);
	}
	
	/**
	 * Modify a component of MoDiscoWorkflow
	 * @param index Component index
	 * @param modiscoWork an instance of {@link Element}
	 */
	private void setComponent(int index, Element modiscoWork){
		this.workflowInterfaceMap.get(this.currentIdWorkflow).setComponent(index,modiscoWork);
	}
	
	/**
	 * Allows to retrieve all components in the MoDiscoWorkflow.
	 * @return a list of Element
	 */
	@SuppressWarnings("unchecked")
	public List<Element> getAllMoDiscoElement(){
		List<? extends EObject> listComponent = this.workflowInterfaceMap.get(this.currentIdWorkflow).getAllComponents();
		List<Element> listTmp = (List<Element>)listComponent;
		List<Element> modiscoList = new ArrayList<Element>();
		for(Element elmt : listTmp){
			modiscoList.add(createElement(getLaunchConfigurationFromName(elmt.getName()), elmt.getName(), elmt.getIndex(), elmt.getType()));
		}
		return  modiscoList;
	}

	/**
	 * Allows to save the modifications to do in the edition of the Element
	 * @param transformationName The new transformation name
	 * @param configuration The new configuration
	 */
	public void performFinishEditTransformation(String transformationName) {
		Element modiscoElement = getComponent(this.editionIndice);
		//updateName
		modiscoElement.setName(transformationName);
		//Update parameters if it's a work
		if(modiscoElement instanceof Work){
			List<WorkParameter> parameters = this.controllerDriver.getDriverFromLaunchType(modiscoElement.getType()).getParameters(this.getLaunchConfigurationFromName(modiscoElement.getName()));
			if(parameters!=null){
				((Work)modiscoElement).getParameters().clear();
				((Work)modiscoElement).getParameters().addAll(parameters);
			}
		}else{
			//Update elements if it's a workflow
			updateElementsOfWorkflow(((Workflow)modiscoElement).getElements());
		}
		setComponent(modiscoElement.getIndex(),modiscoElement);
	}
	
	/**
	 * Allows to update all element of workflow
	 * @param elements Elements of workflow
	 */
	private void updateElementsOfWorkflow(EList<Element> elements) {
		for(Element elmt : elements){
			if(elmt instanceof Work){
				List<WorkParameter> parameters = this.controllerDriver.getDriverFromLaunchType(elmt.getType()).getParameters(this.getLaunchConfigurationFromName(elmt.getName()));
				if(parameters!=null){
					((Work)elmt).getParameters().clear();
					((Work)elmt).getParameters().addAll(parameters);
				}
			}else{
				updateElementsOfWorkflow(((Workflow)elmt).getElements());
			}
		}
	}

	/**
	 * Verify that all the elements have a configuration
	 * @return the Element name which didn't have a configuration
	 */
	public String allComponentsHaveConfiguration() {
		String transformationName = null;
		for(Element  modiscoWork: getAllMoDiscoElement()){
			if(modiscoWork!=null){
				ILaunchConfiguration config = getLaunchConfigurationFromName(modiscoWork.getName());
				if(config==null){
					transformationName = modiscoWork.getName();
					break;
				}
				if(!config.exists()){
					transformationName = modiscoWork.getName();
					break;
				}
			}
		}
		return transformationName;
	}
	
	/**
	 * Allows to save the resource
	 * @param configuration LaunchConfiguration of MoDiscoWorkflow
	 */
	public void saveWorkflowResource(ILaunchConfigurationWorkingCopy configuration, boolean onlyLaunch) {
		this.nameRoot = configuration.getName();
		this.workflowInterfaceMap.get(this.currentIdWorkflow).save(configuration, onlyLaunch);
	}
	
	/**
	 * Allows to move the resource
	 * @param path Workflow resource path
	 */
	public void moveWorkflowResource(String path) {
		this.workflowInterfaceMap.get(this.currentIdWorkflow).moveResource(path);
	}
	
	/**
	 * Allows to initialize or load the workflow
	 * @param path workflow path
	 */
	public void initializationWorkflowResource(ILaunchConfiguration configuration, String path) {
		WorkflowModelInterface model = this.workflowInterfaceMap.get(this.currentIdWorkflow);
		model.initialization(configuration,path);	
		//Synchronization for MWE
		if(this.currentIdWorkflow.equals(MoDiscoWorkflowLaunchConstants.idMinimaliste)==false){
			this.workflowInterfaceMap.get(this.currentIdWorkflow).synchronization(getAllMoDiscoElement());
		}
	}
	
	/**
	 * Allows to remove the resource
	 * @param resourcePath Path of the .workflow resource
	 */
	public void removeResource(String resourcePath) {
		this.workflowInterfaceMap.get(this.currentIdWorkflow).removeResource(resourcePath);
		
	}
	
	/**
	 * Allows to retrieve all parameters of selected driver in the table
	 * @param tableIndex Index selected of the table
	 * @return All parameters of driver
	 */
	public List<WorkParameter> getAllParameterOfDriver(int tableIndex){
		List<WorkParameter> parameters = null;
		Element modisco = getComponent(tableIndex);
		ILaunchConfiguration config = getLaunchConfigurationFromName(modisco.getName());
		if(config!=null){
			ILaunchConfigurationWorkingCopy copy = null;
			try {
				copy = config.getWorkingCopy();
			} catch (CoreException e) {
				System.err.println(Messages.ControllerWorkflowTransformation_2);
				e.printStackTrace();
			}
			parameters = this.controllerDriver.getDriverFromLaunchType(modisco.getType()).getParameters(copy);
		}
		return parameters;
	}
	
	/**
	 * Allows to find the good instance of LaunchConfiguration corresponding at name
	 * @param configName Name of LaunchConfiguration
	 * @return An instance of launchConfiguration corresponding at name
	 */
	private ILaunchConfiguration getLaunchConfigurationFromName(String configName){
		ILaunchConfiguration configuration = null;
		try {
			for(ILaunchConfiguration config : DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations()){
				if(config.getName().equals(configName)){
					configuration = config;
					break;
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return configuration;
	}
	
	/**
	 * Allows to change workflow engine (MWE to minimalist or minimalist to MWE)
	 * @param configuration LaunchConfigurationWorkingCopy of MoDiscoWorkflow
	 */
	public void changeWorkflowEngine(ILaunchConfigurationWorkingCopy configuration) {
		String newId = MoDiscoWorkflowLaunchConstants.idMinimaliste;
		try {
			newId = configuration.getAttribute(MoDiscoWorkflowLaunchConstants.idWorkflow, MoDiscoWorkflowLaunchConstants.idMinimaliste);
		} catch (CoreException e) {
			e.printStackTrace();
		}
		if(!newId.equals(this.currentIdWorkflow)){
			List<Element> listMoDisco = getAllMoDiscoElement();
			String resourcePath = null;
			if (!configuration.isLocal()) {				
				IPath path = configuration.getFile().getLocation().removeLastSegments(1).append(configuration.getName()).addFileExtension(Constants.workflowExtension);
				resourcePath = path.toString();
			} else {
				resourcePath = Activator.getDefault().getStateLocation().append(configuration.getName()).addFileExtension(Constants.workflowExtension).toString();
			}
			//MWE to minimalist.
			if(newId.equals(MoDiscoWorkflowLaunchConstants.idMinimaliste)){
				//Delete the resource
				removeResource(resourcePath);
				configuration.removeAttribute(MoDiscoWorkflowLaunchConstants.moDiscoWorkNumber);
				configuration.removeAttribute(MoDiscoWorkflowLaunchConstants.moDiscoListSerialize);
				configuration.removeAttribute(MoDiscoWorkflowLaunchConstants.resourcePath);
			}else{
				this.currentIdWorkflow = newId;
				configuration.removeAttribute(MoDiscoWorkflowLaunchConstants.moDiscoListMinimalist);
			}
			this.currentIdWorkflow = newId;
			initializationWorkflowResource(configuration.getOriginal(), resourcePath);
			for(Element modisco : listMoDisco){
				addElement(modisco.getType(), modisco.getIndex(), modisco.getName(), false);
			}
			saveWorkflowResource(configuration, false);
		}
	}

	/**
	 * Allows to export the workflow (.launch or .workflow)
	 * @param configuration The Launch configuration associated
	 * @param outputPath The output path
	 */
	public void exportWorkflow(ILaunchConfiguration configuration, String outputPath){
		this.workflowInterfaceMap.get(this.currentIdWorkflow).exportWorkflow(configuration, outputPath);
	}
	
	/**
	 * Allows to create a discoverer project from a MoDiscoWorkflow
	 * @param projectName Name of project
	 * @param basePackageName Package base name
	 */
	public void exportMoDiscoWorkflowToDiscovererProject(String projectName,String symbolicName,String basePackageName) {
		try {
			URI uri = URI.createFileURI(Activator.getDefault().getStateLocation().append(Messages.ControllerWorkflowTransformation_3).toString());
			Resource res = createResourceExport(projectName,symbolicName, basePackageName, uri);
			new CreatePluginProject(res);
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Allows to export the internal model
	 * @param outputPath The output path
	 */
	public void exportInternalModel(String outputPath){
		IPath p = new Path(outputPath);
		p = p.append(this.nameRoot).addFileExtension("modiscoworkflow"); //$NON-NLS-1$
		URI uri = URI.createFileURI(p.toString());
		createResourceExport(this.nameRoot, this.nameRoot, this.nameRoot, uri);
	}
	
	
	/**
	 * Allows to verify if all element name of workflow are valid to define a class name
	 * @return Status of MoDiscoWorks
	 */
	public IStatus verifyNameToExportDiscoverer() {
		IStatus status = null;
		for(Element modiscoWork : getAllMoDiscoElement()){
			status = JavaConventions.validateIdentifier(modiscoWork.getName(),JavaCore.COMPILER_SOURCE , JavaCore.COMPILER_COMPLIANCE);
			if(!status.isOK()){
				break;
			}
		}
		return status;
	}

	/**
	 * Allows to create a modiscoWorkflow resource
	 * @param name Name of project
	 * @param basePackageName The name of base package
	 * @return A modiscoWorkflow resource
	 */
	private Resource createResourceExport(String name, String symbolicName, String basePackageName, URI uri){
		Resource resource = null;
		ModiscoworkflowPackage.eINSTANCE.eClass();
		Workflow workflow = ModiscoworkflowFactory.eINSTANCE.createWorkflow();
		ResourceSet resSet = new ResourceSetImpl();
		resource = resSet.createResource(uri);
		workflow.setName(this.nameRoot);
		workflow.setPackageBase(basePackageName);
		workflow.setSymbolicName(symbolicName);
		List<Element> elements = getAllMoDiscoElement();
		workflow.getElements().addAll(elements);
		resource.getContents().add(workflow);
		try {
			resource.save(null);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return resource;
	}

	/**
	 * Allows to create all elements of workflow
	 * @param name Configuration name of workflow
	 * @param type Configuration type of workflow
	 * @return Created elements list 
	 */
	private Collection<? extends Element> createElementsOfWorkflow(String name, String type) {
		List<Element> elements = new ArrayList<Element>();
		List<WorkParameter> parameters = this.controllerDriver.getDriverFromLaunchType(type).getParameters(this.getLaunchConfigurationFromName(name));
		Map<String,WorkParameterValue> mapParamters = new HashMap<String, WorkParameterValue>();
		for(WorkParameter parameter : parameters){
			mapParamters.put(parameter.getName(), parameter.getValue());
		}
		WorkParameterStringValue idValue = (WorkParameterStringValue) mapParamters.get(MoDiscoWorkflowLaunchConstants.idWorkflow);
		//Minimalist case
		if(idValue.getValue().equals(MoDiscoWorkflowLaunchConstants.idMinimaliste)){
			WorkParameterStringListValue elementList = (WorkParameterStringListValue)mapParamters.get(MoDiscoWorkflowLaunchConstants.moDiscoListMinimalist);
			List<String> stringList = elementList.getValue();
			for(int i=0;i<stringList.size();i=i+3){
				ILaunchConfiguration configuration = this.getLaunchConfigurationFromName(stringList.get(i));
				elements.add(createElement(configuration,stringList.get(i),Integer.valueOf(stringList.get(i+1)),stringList.get(i+2)));
			}
		}else{
			//MWE Case
			WorkParameterStringListValue elementList = (WorkParameterStringListValue)mapParamters.get(MoDiscoWorkflowLaunchConstants.moDiscoListSerialize);
			List<String> stringList = elementList.getValue();
			int index = 0;
			for(String elmt :stringList){
				ILaunchConfiguration configuration = this.getLaunchConfigurationFromName(elmt.substring(0, elmt.indexOf(";"))); //$NON-NLS-1$
				elements.add(createElement(configuration,elmt.substring(0, elmt.indexOf(";")),index,elmt.substring(elmt.lastIndexOf(";")))); //$NON-NLS-1$ //$NON-NLS-2$
				index++;
			}
		}
		return elements;
	}
	
	/**
	 * Allows to create an element
	 * @param configuration launch configuration of element
	 * @param index Index in the workflow
	 * @return Created element
	 */
	private Element createElement(ILaunchConfiguration configuration, String name, int index, String type) {
		Element createdElement = null;
		if(configuration!=null){
			try {
				String configType = configuration.getType().getIdentifier();
				if(configType.equals(MoDiscoWorkflowLaunchConstants.launchTypeMoDiscoWorkflow)){
					//Create a workflow
					Workflow workflow = ModiscoworkflowFactory.eINSTANCE.createWorkflow();
					workflow.setIndex(index);
					workflow.setName(configuration.getName());
					workflow.setType(configType);
					workflow.getElements().addAll(createElementsOfWorkflow(configuration.getName(), configType));
					createdElement = workflow;
				}else{
					//Create a work
					Work work = ModiscoworkflowFactory.eINSTANCE.createWork();
					work.setIndex(index);
					work.setName(configuration.getName());
					work.setType(configType);
					work.getParameters().addAll(this.controllerDriver.getDriverFromLaunchType(configType).getParameters(configuration));
					createdElement = work;
				}
			} catch (CoreException e) {
				e.printStackTrace();
			}
		}else{
			Work work = ModiscoworkflowFactory.eINSTANCE.createWork();
			work.setIndex(index);
			work.setName(name);
			work.setType(type);
			createdElement = work;
		}
		return createdElement;
	}

	/*
	 * Getter and Setter
	 */
	public void setAdditionnalButton(AdditionalButtonsComposite additionalButton){
		this.additionalButton = additionalButton;
	}

	public LaunchGroupExtension getGroup() {
		return this.group;
	}

	public void setGroup(LaunchGroupExtension group) {
		this.group = group;
	}

	public WorkflowExecutionInterface getWorkflowExecutionInterface(String id) {
		return this.workflowExecutionInterfaceMap.get(id);
	}

	public String getIdWorkflowExecution() {
		return this.currentIdWorkflow;
	}
}
