/**
 * *******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * 
 *     Fabien Giquel (Mia-Software) - initial API and implementation
 * *******************************************************************************
 *
 */
package org.eclipse.gmt.modisco.xml.internal.resource;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.xml.Attribute;
import org.eclipse.gmt.modisco.xml.Comment;
import org.eclipse.gmt.modisco.xml.DocumentTypeDeclaration;
import org.eclipse.gmt.modisco.xml.Element;
import org.eclipse.gmt.modisco.xml.Namespace;
import org.eclipse.gmt.modisco.xml.ProcessingInstruction;
import org.eclipse.gmt.modisco.xml.Root;
import org.eclipse.gmt.modisco.xml.Text;
import org.eclipse.gmt.modisco.xml.emf.moDiscoXMLFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;

/**
 * A handler for reading xml files and instanciating a generic xml model.
 */
public class GenericXMLHandler extends DefaultHandler implements LexicalHandler {
	/**
	 * Prefix for namespace attributes.
	 */
	public static final String XMLNS = "xmlns"; //$NON-NLS-1$
	/**
	 * A parameter key for indicating to ignore white spaces in text portions
	 * found.
	 */
	public static final String OPTION_IGNORE_WHITESPACES = "OPTION_IGNORE_WHITESPACES"; //$NON-NLS-1$
	/**
	 * A parameter key for indicating to minimize the memory size of obtained
	 * model. If set to true : comments are ignored, text portions with only
	 * indentation and line delimiters are ignored.
	 */
	public static final String OPTION_LIGHTWEIGHT_MODEL = "OPTION_IGNORE_WHITESPACES"; //$NON-NLS-1$

	private Map<String, Object> parameters;
	private Resource container;

	private Root root;
	private Element current;
	private DocumentTypeDeclaration dtd;
	private boolean nextIsCDATA = false;
	private List<ProcessingInstruction> leadingPIs = null;

	private Boolean ignoreWhiteSpaces = null;
	private Boolean lightWeightModel = null;

	/**
	 * Constructor for GenericXMLHandler.
	 * 
	 * @param resource
	 *            the model container.
	 */
	public GenericXMLHandler(final Resource resource,
			final Map<String, Object> options) {
		this.container = resource;
		this.parameters = options;
	}

	@Override
	public final void startElement(final String uri, final String localName,
			final String qName, final Attributes attrs) throws SAXException {
		if (this.root == null) {
			this.root = moDiscoXMLFactory.eINSTANCE.createRoot();
			this.root.setName(qName);
			this.container.getContents().add(this.root);
			this.current = this.root;
			if (this.dtd != null) {
				this.root.setDtd(this.dtd);
			}
			if (this.leadingPIs != null) {
				this.root.getLeadingPIs().addAll(this.leadingPIs);
			}
		} else {
			Element newElement = moDiscoXMLFactory.eINSTANCE.createElement();
			newElement.setName(qName);

			this.current.getChildren().add(newElement);
			this.current = newElement;
		}

		for (int i = 0; i < attrs.getLength(); i++) {
			String name = attrs.getQName(i);
			if (this.current == this.root && name.startsWith(XMLNS)) {
				Namespace newNamespace = moDiscoXMLFactory.eINSTANCE
						.createNamespace();
				if (name.length() == XMLNS.length()) {
					newNamespace.setName(""); // xmlns= //$NON-NLS-1$
				} else {
					newNamespace.setName(name.substring(XMLNS.length() + 1));
					// xmlns:XX=
				}
				newNamespace.setValue(attrs.getValue(i));

				this.root.getNamespaces().add(newNamespace);
			} else {
				Attribute newAttribute = moDiscoXMLFactory.eINSTANCE
						.createAttribute();
				newAttribute.setName(name);
				newAttribute.setValue(attrs.getValue(i));

				this.current.getChildren().add(newAttribute);
			}
		}
	}

	@Override
	public final void endElement(final String uri, final String localName,
			final String qName) throws SAXException {
		this.current = this.current.getParent();
	}

	@Override
	public final void characters(final char[] ch, final int start,
			final int length) throws SAXException {
		if (this.current != null) {
			boolean hasContent = true;
			if (minimizeInformations()) {
				hasContent = false;
				int index = start;
				while (!hasContent && index < start + length) {
					hasContent = !isIndentOrLineDelimiterChar(ch[index]);
					index++;
				}
			}

			if (hasContent) {
				String text = new String(ch, start, length);
				if (ignoreWhiteSpaces()) {
					text = text.trim();
				}

				if (text.length() > 0) {
					Text newElement;
					if (this.nextIsCDATA) {
						newElement = moDiscoXMLFactory.eINSTANCE.createCDATA();
					} else {
						newElement = moDiscoXMLFactory.eINSTANCE.createText();
					}
					newElement.setName(text);
					this.current.getChildren().add(newElement);
				}
			}
		}
	}

	public final void comment(final char[] ch, final int start, final int length)
			throws SAXException {
		if (this.current != null && !minimizeInformations()) {
			String text = new String(ch, start, length);

			if (text.length() > 0) {
				Comment newElement = moDiscoXMLFactory.eINSTANCE
						.createComment();
				newElement.setName(text);
				this.current.getChildren().add(newElement);
			}
		}
	}

	public final void startDTD(final String name, final String publicId,
			final String systemId) throws SAXException {
		DocumentTypeDeclaration newDtd = moDiscoXMLFactory.eINSTANCE
				.createDocumentTypeDeclaration();
		newDtd.setName(name);
		newDtd.setPublicID(publicId);
		newDtd.setSystemID(systemId);
		this.dtd = newDtd;
		if (this.root != null) {
			this.root.setDtd(this.dtd);
		}
	}

	private boolean ignoreWhiteSpaces() {
		if (this.ignoreWhiteSpaces == null) {
			if (this.parameters != null
					&& this.parameters.get(OPTION_IGNORE_WHITESPACES) != null) {
				this.ignoreWhiteSpaces = (Boolean) this.parameters
						.get(OPTION_IGNORE_WHITESPACES);
			} else {
				this.ignoreWhiteSpaces = false;
			}
		}

		return this.ignoreWhiteSpaces;
	}

	private boolean minimizeInformations() {
		if (this.lightWeightModel == null) {
			if (this.parameters != null
					&& this.parameters.get(OPTION_LIGHTWEIGHT_MODEL) != null) {
				this.lightWeightModel = (Boolean) this.parameters
						.get(OPTION_LIGHTWEIGHT_MODEL);
			} else {
				this.lightWeightModel = false;
			}
		}

		return this.lightWeightModel;
	}

	private boolean isIndentOrLineDelimiterChar(final char ch) {
		return ch == ' ' || ch == '\t' || ch == '\n' || ch == '\r';
	}

	public final void endCDATA() throws SAXException {
		this.nextIsCDATA = false;
	}

	@Override
	public final void processingInstruction(final String target,
			final String data) throws SAXException {
		ProcessingInstruction newElement = moDiscoXMLFactory.eINSTANCE
				.createProcessingInstruction();
		newElement.setName(target);
		newElement.setData(data);

		if (this.current != null) {
			this.current.getChildren().add(newElement);
		} else {
			if (this.leadingPIs == null) {
				this.leadingPIs = new ArrayList<ProcessingInstruction>();
			}
			this.leadingPIs.add(newElement);
		}

	}

	public void endDTD() throws SAXException {
		// nothing
	}

	public void endEntity(final String arg0) throws SAXException {
		// nothing
	}

	public final void startCDATA() throws SAXException {
		this.nextIsCDATA = true;
	}

	public void startEntity(final String arg0) throws SAXException {
		// Nothing
	}

}
