/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.discoverersmanager.ui.launch.utils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.gmt.modisco.discoverersmanager.DiscoverersManager;
import org.eclipse.gmt.modisco.discoverersmanager.DiscoveryParameter;
import org.eclipse.gmt.modisco.discoverersmanager.ui.launch.data.DiscoveryParameterDisplay;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.swt.widgets.Composite;

/**
 * @author gbarbier
 * 
 */
public final class ParameterUtils {

	private final static ParameterUtils instance = new ParameterUtils();

	private ParameterUtils() {
	}

	public static ParameterUtils getInstance() {
		return ParameterUtils.instance;
	}

	private Map<Class<?>, Class<? extends CellEditor>> discovererCellEditors;

	@SuppressWarnings("unchecked")
	private Map<Class<?>, Class<? extends CellEditor>> getDiscovererCellEditors() {
		if (this.discovererCellEditors == null) {
			this.discovererCellEditors = new HashMap<Class<?>, Class<? extends CellEditor>>();
			IExtensionRegistry registry = Platform.getExtensionRegistry();
			IExtensionPoint point = registry
					.getExtensionPoint("org.eclipse.gmt.modisco.cellcontenteditors"); //$NON-NLS-1$

			for (IExtension extension : point.getExtensions()) {
				for (IConfigurationElement element : extension.getConfigurationElements()) {
					String targetType = element.getAttribute("targetType"); //$NON-NLS-1$
					String editorImplementation = element.getAttribute("editorImplementation"); //$NON-NLS-1$
					if ((targetType != null) && (editorImplementation != null)) {
						try {
							Class<?> targetTypeClass = Class.forName(targetType);
							Class<? extends CellEditor> editorClass = (Class<? extends CellEditor>) Class
									.forName(editorImplementation);

							this.discovererCellEditors.put(targetTypeClass, editorClass);
						} catch (ClassNotFoundException e) {
							String message = "A problem occured when instantiating discoverer cell editor configuration: " + element.getValue(); //$NON-NLS-1$
							IStatus status = new Status(IStatus.WARNING,
									DiscoverersManager.PLUGIN_ID, message, e);
							DiscoverersManager.getDefault().getLog().log(status);
						}
					} else {
						String message = "A problem occured when retrieving discoverer cell editor configuration: " + element.getValue(); //$NON-NLS-1$
						IStatus status = new Status(IStatus.WARNING, DiscoverersManager.PLUGIN_ID,
								message);
						DiscoverersManager.getDefault().getLog().log(status);
					}
				}
			}
		}
		return this.discovererCellEditors;
	}

	@SuppressWarnings("unchecked")
	public Object deserializeParameterValue(final DiscoveryParameter parameter,
			final ILaunchConfiguration configuration) {
		Object result = null;
		Class<?> type = parameter.getType();
		try {
			// test if configuration contains a value for this parameter
			if (configuration.hasAttribute(parameter.getName())) {
				// different options are available to retrieve the value
				if (type.equals(String.class)) {
					String value = configuration.getAttribute(parameter.getName(), ""); //$NON-NLS-1$
					result = value;
				} else if ((type.equals(Integer.TYPE)) || (type.equals(Integer.class))) {
					int value = configuration.getAttribute(parameter.getName(), 0);
					result = value; // autoboxing: int -> Integer
				} else if ((type.equals(Boolean.TYPE)) || (type.equals(Boolean.class))) {
					boolean value = configuration.getAttribute(parameter.getName(), false);
					result = value; // autoboxing: boolean -> Boolean
				} else if (List.class.isAssignableFrom(type)) {
					List<?> value = configuration.getAttribute(parameter.getName(), Collections
							.emptyList());
					result = value;
				} else if (Set.class.isAssignableFrom(type)) {
					Set<?> value = new HashSet<Object>();
					List list = configuration.getAttribute(parameter.getName(), Collections
							.emptyList());
					value.addAll(list);
					result = value;
				} else if (Map.class.isAssignableFrom(type)) {
					Map<?, ?> value = configuration.getAttribute(parameter.getName(), Collections
							.emptyMap());
					result = value;
				} else {
					// TODO Object not managed, we have to retrieve a
					// ParameterFactory from an extension point
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return result;
	}

	/**
	 * 
	 * @param parameter
	 * @param configuration
	 */
	public void updateParameterValue(final DiscoveryParameterDisplay parameter,
			final ILaunchConfiguration configuration) {
		if (configuration != null) {
			Object value = deserializeParameterValue(parameter, configuration);
			parameter.setValue(value);
		}
	}

	/**
	 * 
	 * @param parameter
	 * @param configuration
	 */
	@SuppressWarnings("unchecked")
	public void serializeParameterValue(final DiscoveryParameterDisplay parameter,
			final ILaunchConfigurationWorkingCopy configuration) {
		Object value = parameter.getValue();
		Class<?> type = parameter.getType();
		if (value != null) {
			// different options are available to retrieve the value
			if (type.equals(String.class)) {
				configuration.setAttribute(parameter.getName(), (String) value);
			} else if ((type.equals(Integer.TYPE)) || (type.equals(Integer.class))) {
				configuration.setAttribute(parameter.getName(), (Integer) value); // autoboxing:
																					// int
																					// ->
																					// Integer
			} else if ((type.equals(Boolean.TYPE)) || (type.equals(Boolean.class))) {
				configuration.setAttribute(parameter.getName(), (Boolean) value); // autoboxing:
																					// boolean
																					// ->
																					// Boolean
			} else if (List.class.isAssignableFrom(type)) {
				configuration.setAttribute(parameter.getName(), (List<?>) value);
			} else if (Set.class.isAssignableFrom(type)) {
				List<?> list = new ArrayList<Object>();
				list.addAll((Set) value);
				configuration.setAttribute(parameter.getName(), list);
			} else if (Map.class.isAssignableFrom(type)) {
				configuration.setAttribute(parameter.getName(), (Map<?, ?>) value);
			} else {
				// Object not managed, we have to retrieve a ParameterFactory
				// from an extension point
				configuration.setAttribute(parameter.getName(), value.toString());
			}
		} else {
			configuration.removeAttribute(parameter.getName());
		}
	}

	public boolean canEdit(final DiscoveryParameterDisplay parameter) {
		boolean result = false;
		Class<?> type = parameter.getType();
		result = getDiscovererCellEditors().containsKey(type);
		return result;
	}

	public CellEditor getCellEditor(final DiscoveryParameterDisplay parameter,
			final Composite parent) {
		CellEditor cellEditor = null;
		Class<?> type = parameter.getType();
		Class<? extends CellEditor> cellEditorClass = getDiscovererCellEditors().get(type);
		try {
			CellEditor cellEditorTemp = cellEditorClass.newInstance();
			cellEditorTemp.create(parent);
			cellEditor = cellEditorTemp;
			// Constructor<? extends CellEditor> constructor =
			// cellEditorClass.getConstructor(Composite.class);
			// cellEditor = constructor.newInstance(parent);
		} catch (SecurityException e) {
			String message = "A problem occured when instantiating discoverer cell editor : " + cellEditorClass; //$NON-NLS-1$
			IStatus status = new Status(IStatus.WARNING, DiscoverersManager.PLUGIN_ID, message, e);
			DiscoverersManager.getDefault().getLog().log(status);
			// } catch (NoSuchMethodException e) {
			// String message =
			// "A problem occured when instantiating discoverer cell editor : "
			// + cellEditorClass;
			// IStatus status = new Status(IStatus.WARNING,
			// DiscoverersManager.PLUGIN_ID, message, e);
			// DiscoverersManager.getDefault().getLog().log(status);
		} catch (IllegalArgumentException e) {
			String message = "A problem occured when instantiating discoverer cell editor : " + cellEditorClass; //$NON-NLS-1$
			IStatus status = new Status(IStatus.WARNING, DiscoverersManager.PLUGIN_ID, message, e);
			DiscoverersManager.getDefault().getLog().log(status);
		} catch (InstantiationException e) {
			String message = "A problem occured when instantiating discoverer cell editor : " + cellEditorClass; //$NON-NLS-1$
			IStatus status = new Status(IStatus.WARNING, DiscoverersManager.PLUGIN_ID, message, e);
			DiscoverersManager.getDefault().getLog().log(status);
		} catch (IllegalAccessException e) {
			String message = "A problem occured when instantiating discoverer cell editor : " + cellEditorClass; //$NON-NLS-1$
			IStatus status = new Status(IStatus.WARNING, DiscoverersManager.PLUGIN_ID, message, e);
			DiscoverersManager.getDefault().getLog().log(status);
			// } catch (InvocationTargetException e) {
			// String message =
			// "A problem occured when instantiating discoverer cell editor : "
			// + cellEditorClass;
			// IStatus status = new Status(IStatus.WARNING,
			// DiscoverersManager.PLUGIN_ID, message, e);
			// DiscoverersManager.getDefault().getLog().log(status);
		}
		return cellEditor;
	}

	public void initializeValueFromEditingSupport(final DiscoveryParameterDisplay parameter,
			final Object value) {
		parameter.setValue(value);
	}

	public Object retrieveValueForEditingSupport(final DiscoveryParameterDisplay parameter) {
		Object result = null;
		result = parameter.getValue();
		return result;
	}
}
