/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.browser.custom.editor.editors;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.ui.URIEditorInput;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.Resource.Diagnostic;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.gmt.modisco.infra.browser.custom.AttributeView;
import org.eclipse.gmt.modisco.infra.browser.custom.CustomView;
import org.eclipse.gmt.modisco.infra.browser.custom.CustomViewFeature;
import org.eclipse.gmt.modisco.infra.browser.custom.CustomizableFeatures;
import org.eclipse.gmt.modisco.infra.browser.custom.DerivedFeatureValue;
import org.eclipse.gmt.modisco.infra.browser.custom.FeatureValue;
import org.eclipse.gmt.modisco.infra.browser.custom.FeatureValueCase;
import org.eclipse.gmt.modisco.infra.browser.custom.MetamodelView;
import org.eclipse.gmt.modisco.infra.browser.custom.ReferenceView;
import org.eclipse.gmt.modisco.infra.browser.custom.StaticFeatureValue;
import org.eclipse.gmt.modisco.infra.browser.custom.TypeView;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.Activator;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.Messages;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.dialogs.CustomizationPropertiesDialog;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.dialogs.EditFeatureValueDialog;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.dialogs.ErrorsDialog;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.dialogs.EditFeatureValueDialog.ValueType;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.editors.MetaclassViewer.Attribute;
import org.eclipse.gmt.modisco.infra.browser.custom.editor.editors.MetaclassViewer.Reference;
import org.eclipse.gmt.modisco.infra.browser.custom.emf.UicustomFactory;
import org.eclipse.gmt.modisco.infra.browser.custom.util.OverlayIconInfo;
import org.eclipse.gmt.modisco.infra.browser.custom.util.OverlayIconPosition;
import org.eclipse.gmt.modisco.infra.browser.custom.util.UicustomUtil;
import org.eclipse.gmt.modisco.infra.browser.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.browser.util.ImageProvider;
import org.eclipse.gmt.modisco.infra.common.core.internal.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.infra.common.core.internal.utils.ModelUtils;
import org.eclipse.gmt.modisco.infra.common.ui.internal.controls.FormStyleSashForm;
import org.eclipse.gmt.modisco.infra.facet.Facet;
import org.eclipse.gmt.modisco.infra.facet.core.FacetSetCatalog;
import org.eclipse.gmt.modisco.infra.query.ModelQuery;
import org.eclipse.gmt.modisco.infra.query.ModelQuerySet;
import org.eclipse.gmt.modisco.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.ColorSelector;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FontDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.part.EditorPart;

@SuppressWarnings("synthetic-access")
public class CustomizationEditor extends EditorPart {

	private static final int VERTICAL_SPACE_BETWEEN_BUTTON_GROUPS = 30;
	private static final int VERTICAL_SPACE_BEFORE_FIRST_BUTTON = 10;
	private static final int TOP_PANE_MARGIN_HEIGHT = 4;
	private static final int TOP_PANE_MARGIN_WIDTH = 5;
	public static final String ID = "org.eclipse.gmt.modisco.infra.browser.custom.editor"; //$NON-NLS-1$
	private static final int MAIN_FORM_MARGIN_WIDTH = 4;
	private static final int MAIN_FORM_MARGIN_HEIGHT = 6;
	private static final int LEFT_PANE_SASH_WEIGHT = 30;
	private static final int RIGHT_PANE_SASH_WEIGHT = 70;

	private MetamodelView fMetamodelView;
	private MetaclassViewer metaclassViewer;
	private MetaclassViewConfiguration metaclassViewConfiguration;
	private CustomizationViewer customizationViewer;
	private Button btnEditFeatureValue;
	private Button btnResetDefaults;
	private Button btnAdd;
	private Button btnEdit;
	private Button btnRemove;
	private Button btnUp;
	private Button btnDown;
	private Button cbAppliesToSubinstances;
	/** Whether the customization model was modified */
	private boolean dirty;
	private Label headerLabel;
	private FormToolkit toolkit;
	private boolean readOnly;

	/** EMF errors on the resource */
	private final HashSet<Diagnostic> allErrors = new HashSet<Diagnostic>();
	/** Button that appears to open the dialog listing errors */
	private Button errorsButton;
	/** The metamodel that is being customized */
	private Resource fMetamodel;
	private Form fForm;

	public CustomizationEditor() {
		// make sure the facets are initialized
		FacetSetCatalog.getSingleton();
	}

	@Override
	public void init(final IEditorSite site, final IEditorInput input) throws PartInitException {
		if (!(input instanceof IFileEditorInput || input instanceof URIEditorInput)) {
			throw new PartInitException("Invalid Input: Must be IFileEditorInput or URIEditorInput"); //$NON-NLS-1$
		}
		// URIEditorInput => read-only
		this.readOnly = (input instanceof URIEditorInput);

		setSite(site);
		setInput(input);
	}

	@Override
	public void createPartControl(final Composite parent) {
		java.net.URI locationURI;
		String fileName = ""; //$NON-NLS-1$
		final IEditorInput editorInput = getEditorInput();
		if (editorInput instanceof IFileEditorInput) {
			final IFileEditorInput fileEditorInput = (IFileEditorInput) editorInput;
			locationURI = fileEditorInput.getFile().getLocationURI();
			fileName = fileEditorInput.getFile().getName();
		} else if (editorInput instanceof URIEditorInput) {
			final URIEditorInput uriEditorInput = (URIEditorInput) editorInput;
			try {
				locationURI = new java.net.URI(uriEditorInput.getURI().toString());
			} catch (final URISyntaxException e) {
				throw new IllegalArgumentException(e);
			}
			fileName = new Path(locationURI.getPath()).lastSegment();
		} else {
			throw new IllegalStateException("wrong kind of input: " //$NON-NLS-1$
					+ editorInput.getClass().getName());
		}

		loadResource(locationURI);
		setPartName(fileName);

		this.metaclassViewConfiguration = new MetaclassViewConfiguration();
		// load user settings
		this.metaclassViewConfiguration.load();

		// register close listener to save user settings when editor is closed
		getSite().getPage().addPartListener(new SaveSettingsOnCloseListener());

		createContents(parent);

		this.metaclassViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(final SelectionChangedEvent event) {
				final ITreeSelection selection = (ITreeSelection) CustomizationEditor.this.metaclassViewer
						.getSelection();
				CustomizationEditor.this.metaclassSelectionChanged(selection.getFirstElement());
				CustomizationEditor.this.customizationViewer.getViewer().getControl().setRedraw(
						false);
				CustomizationEditor.this.customizationViewer.getViewer().expandAll();
				CustomizationEditor.this.customizationViewer.getViewer().getControl().setRedraw(
						true);
			}
		});

		this.customizationViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(final SelectionChangedEvent event) {
				CustomizationEditor.this.customizationSelectionChanged(getSelectedElement());
			}
		});

		// shortcuts to avoid having to use the buttons
		this.customizationViewer.getViewer().addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				final Object selectedElement = getSelectedElement();
				if (selectedElement instanceof CustomViewFeature) {
					btnEditFeatureValueClicked();
				} else if (selectedElement instanceof FeatureValueCase) {
					btnEditClicked();
				}
			}
		});

		this.customizationViewer.getViewer().getControl().addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.keyCode == SWT.DEL) {
					final Object selectedElement = getSelectedElement();
					if (selectedElement instanceof FeatureValueCase) {
						btnRemoveClicked();
					}
				}
			}
		});

		this.fMetamodel = initMetamodel();
	}

	private Resource initMetamodel() {
		final String metamodelURI = this.fMetamodelView.getMetamodelURI();
		final EPackage ePackage = EPackage.Registry.INSTANCE.getEPackage(metamodelURI);
		if (ePackage != null) {
			final Resource resource = ePackage.eResource();
			if (resource != null) {
				this.metaclassViewer.setInput(resource);
				return resource;
			}
		}
		MoDiscoLogger.logError(NLS.bind(Messages.CustomizationEditor_cannotFindMetamodel,
				metamodelURI), Activator.getDefault());
		return null;
	}

	private void createContents(final Composite parent) {
		this.toolkit = new FormToolkit(parent.getDisplay());
		this.fForm = this.toolkit.createForm(parent);
		this.toolkit.decorateFormHeading(this.fForm);

		final FillLayout fillLayout = new FillLayout();
		fillLayout.marginWidth = CustomizationEditor.MAIN_FORM_MARGIN_WIDTH;
		fillLayout.marginHeight = CustomizationEditor.MAIN_FORM_MARGIN_HEIGHT;
		this.fForm.getBody().setLayout(fillLayout);
		this.fForm.setHeadClient(createTopPane(this.fForm.getHead()));
		createBottomPane(this.fForm.getBody());
		buttonsEnablementForSelection(null);
	}

	/** The white header */
	private Composite createTopPane(final Composite parent) {
		final Composite topPane = new Composite(parent, SWT.NONE);
		final GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		topPane.setLayoutData(gridData);

		final GridLayout gridLayout = new GridLayout(3, false);
		gridLayout.marginWidth = CustomizationEditor.TOP_PANE_MARGIN_WIDTH;
		gridLayout.marginHeight = CustomizationEditor.TOP_PANE_MARGIN_HEIGHT;
		topPane.setLayout(gridLayout);
		this.headerLabel = new Label(topPane, SWT.NONE);
		this.headerLabel.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		refreshHeader();

		this.errorsButton = new Button(topPane, SWT.PUSH);
		this.errorsButton.setToolTipText(Messages.CustomizationEditor_btnTooltipViewErrors);
		this.errorsButton.setImage(ImageProvider.getInstance().getErrorIcon());
		this.errorsButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final ErrorsDialog errorsDialog = new ErrorsDialog(getSite().getShell(),
						CustomizationEditor.this.allErrors);
				errorsDialog.open();
			}
		});
		// only appears when there are errors
		if (this.allErrors.isEmpty()) {
			this.errorsButton.setVisible(false);
		}

		final Button propertiesButton = new Button(topPane, SWT.PUSH);
		propertiesButton.setText(Messages.CustomizationEditor_btnProperties);
		propertiesButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				propertiesClicked();
			}
		});

		// Color backgroundColor =
		// Display.getDefault().getSystemColor(SWT.COLOR_WHITE);
		// topPane.setBackground(backgroundColor);
		// headerLabel.setBackground(backgroundColor);
		// propertiesButton.setBackground(backgroundColor);

		return topPane;
	}

	private void refreshHeader() {
		final String readOnlyText;
		if (this.readOnly) {
			readOnlyText = Messages.CustomizationEditor_readonly;
		} else {
			readOnlyText = ""; //$NON-NLS-1$
		}
		this.headerLabel.setText(readOnlyText
				+ NLS.bind(Messages.CustomizationEditor_customViewFor, getMetamodelName(),
						this.fMetamodelView.getMetamodelURI()));
	}

	protected void propertiesClicked() {
		final CustomizationPropertiesDialog dialog = new CustomizationPropertiesDialog(getSite()
				.getShell(), this.readOnly);
		dialog.setInitialMetamodel(this.fMetamodelView.getMetamodelURI());
		final EList<String> importedQuerySets = this.fMetamodelView.getAvailableQuerySets();
		dialog.setInitialQuerySets(importedQuerySets);
		dialog.setInitialImportAllQuerySets(this.fMetamodelView.isAllQuerySetsAvailable());
		if (dialog.open() == Window.OK && !this.readOnly) {
			// change metamodel
			if (!dialog.getMetamodel().equals(this.fMetamodelView.getMetamodelURI())) {
				this.fMetamodelView.setMetamodelURI(dialog.getMetamodel());
				Resource newMetamodel = initMetamodel();
				promptToCleanCustomization(newMetamodel);
			}

			this.fMetamodelView.setAllQuerySetsAvailable(dialog.isAllQuerySetsAvailable());

			// save imported query sets
			final EList<String> imports = this.fMetamodelView.getAvailableQuerySets();
			imports.clear();
			final String[] selectedQuerySets = dialog.getSelectedQuerySets();
			for (final String selectedQuerySet : selectedQuerySets) {
				imports.add(selectedQuerySet);
			}

			setDirty(true);
			refreshHeader();
			refresh();
		}
	}

	/**
	 * Opens a message box to ask the user whether to keep customizations for
	 * the old metamodels, or clean the customizations.
	 * 
	 * @param newMetamodel
	 */
	private void promptToCleanCustomization(final Resource newMetamodel) {
		String message = Messages.CustomizationEditor_youChangedMetamodel
				+ Messages.CustomizationEditor_promptCleanCustomizations;
		if (MessageDialog.openQuestion(getSite().getShell(),
				Messages.CustomizationEditor_Title_cleanCustomization, message)) {
			final EList<TypeView> typeViews = this.fMetamodelView.getTypes();
			final List<EClass> metaclasses = EMFUtil.getMetaclasses(newMetamodel);

			// remove all type views that don't have a corresponding metaclass
			final ListIterator<TypeView> typeViewsIterator = typeViews.listIterator();
			while (typeViewsIterator.hasNext()) {
				final TypeView typeView = typeViewsIterator.next();
				final String metaclassName = typeView.getMetaclassName();
				final EClass eClass = EMFUtil.findMetaclassWithName(metaclassName, metaclasses);
				if (eClass == null) {
					typeViewsIterator.remove();
				} else {
					// remove attribute views that don't have a corresponding
					// attribute
					// in the metaclass
					final EList<AttributeView> attributeViews = typeView.getAttributes();
					ListIterator<AttributeView> attributeViewsIterator = attributeViews
							.listIterator();
					while (attributeViewsIterator.hasNext()) {
						AttributeView attributeView = attributeViewsIterator.next();
						if (EMFUtil.findElementWithName(attributeView.getAttributeName(), eClass
								.getEAllAttributes()) == null) {
							attributeViewsIterator.remove();
						}
					}

					// remove reference views that don't have a corresponding
					// reference
					// in the metaclass
					final EList<ReferenceView> referenceViews = typeView.getReferences();
					ListIterator<ReferenceView> referenceViewsIterator = referenceViews
							.listIterator();
					while (referenceViewsIterator.hasNext()) {
						ReferenceView referenceView = referenceViewsIterator.next();
						if (EMFUtil.findElementWithName(referenceView.getReferenceName(), eClass
								.getEAllReferences()) == null) {
							referenceViewsIterator.remove();
						}
					}
				}
			}
			this.metaclassViewer.refresh();
		}
	}

	private String getMetamodelName() {
		final String nsURI = this.fMetamodelView.getMetamodelURI();
		String metamodelName = EMFUtil.getMetamodelName(nsURI);
		if (metamodelName == null) {
			return Messages.CustomizationEditor_unknown;
		}
		return metamodelName;
	}

	private Composite createBottomPane(final Composite parent) {
		final SashForm sashForm = new FormStyleSashForm(parent, SWT.HORIZONTAL, this.toolkit) {
			@Override
			protected void createContents() {
				createLeftPane(this);
				createRightPane(this);
			}
		};

		sashForm.setWeights(new int[] { CustomizationEditor.LEFT_PANE_SASH_WEIGHT,
				CustomizationEditor.RIGHT_PANE_SASH_WEIGHT });
		// sashForm.setSashWidth(5);

		return sashForm;
	}

	private Composite createLeftPane(final Composite parent) {
		final Section section = this.toolkit.createSection(parent, ExpandableComposite.TITLE_BAR
				| ExpandableComposite.EXPANDED);
		section.setText(Messages.CustomizationEditor_sectionName_types);

		final Composite leftPaneComposite = this.toolkit.createComposite(section);
		section.setClient(leftPaneComposite);

		final GridLayout gridLayout = new GridLayout(1, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		leftPaneComposite.setLayout(gridLayout);

		final MetaclassViewToolBar toolBar = createToolBar(section);
		section.setTextClient(toolBar.getParentComposite());
		this.metaclassViewer = new MetaclassViewer(leftPaneComposite,
				this.metaclassViewConfiguration, this.fMetamodelView);
		toolBar.setMetaclassViewer(this.metaclassViewer);

		final GridData metaclassViewerGridData = new GridData();
		metaclassViewerGridData.grabExcessHorizontalSpace = true;
		metaclassViewerGridData.grabExcessVerticalSpace = true;
		metaclassViewerGridData.horizontalAlignment = SWT.FILL;
		metaclassViewerGridData.verticalAlignment = SWT.FILL;
		this.metaclassViewer.getViewer().getControl().setLayoutData(metaclassViewerGridData);

		return leftPaneComposite;
	}

	/** Right pane : treeviewer + fixed buttons pane */
	private Composite createRightPane(final Composite parent) {
		final Section section = this.toolkit.createSection(parent, ExpandableComposite.TITLE_BAR
				| ExpandableComposite.EXPANDED);
		section.setText(Messages.CustomizationEditor_sectionName_customizations);

		final Composite rightPaneComposite = this.toolkit.createComposite(section, SWT.NONE);
		section.setClient(rightPaneComposite);

		final GridLayout gridLayout = new GridLayout(2, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		rightPaneComposite.setLayout(gridLayout);

		// tree + bottom options
		final Composite middle = new Composite(rightPaneComposite, SWT.NONE);
		final GridLayout middleLayout = new GridLayout();
		middleLayout.marginHeight = 0;
		middleLayout.marginWidth = 0;
		middle.setLayout(middleLayout);
		middle.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		this.customizationViewer = new CustomizationViewer(middle);
		this.customizationViewer.getViewer().getControl().setLayoutData(
				new GridData(SWT.FILL, SWT.FILL, true, true));
		final Composite optionsPane = this.toolkit.createComposite(middle);
		optionsPane.setLayout(new GridLayout());
		optionsPane.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

		this.cbAppliesToSubinstances = this.toolkit.createButton(optionsPane,
				Messages.CustomizationEditor_appliesToSubInstances, SWT.CHECK);
		this.cbAppliesToSubinstances.setEnabled(false);
		this.cbAppliesToSubinstances.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				appliesToSubinstancesChanged(CustomizationEditor.this.cbAppliesToSubinstances
						.getSelection());
			}
		});

		if (this.readOnly) {
			this.cbAppliesToSubinstances.setEnabled(false);
		}

		final GridData customizationViewerGridData = new GridData();
		customizationViewerGridData.grabExcessHorizontalSpace = true;
		customizationViewerGridData.grabExcessVerticalSpace = true;
		customizationViewerGridData.horizontalAlignment = SWT.FILL;
		customizationViewerGridData.verticalAlignment = SWT.FILL;
		this.customizationViewer.getViewer().getControl()
				.setLayoutData(customizationViewerGridData);

		if (!this.readOnly) {
			createButtonsPane(rightPaneComposite);
		}

		// pushes the checkbox to the bottom
		// Composite spring = new Composite(buttonsPaneComposite, SWT.NONE);
		// GridData springData = new GridData(SWT.NONE, SWT.FILL, false, true);
		// springData.heightHint = 1;
		// spring.setLayoutData(springData);

		return rightPaneComposite;
	}

	private Composite createButtonsPane(final Composite parent) {
		final Composite buttonsPaneComposite = new Composite(parent, SWT.NONE);
		final GridData buttonsPaneGridData = new GridData();
		buttonsPaneGridData.grabExcessHorizontalSpace = false;
		buttonsPaneGridData.grabExcessVerticalSpace = true;
		buttonsPaneGridData.horizontalAlignment = SWT.FILL;
		buttonsPaneGridData.verticalAlignment = SWT.FILL;
		buttonsPaneComposite.setLayoutData(buttonsPaneGridData);

		buttonsPaneComposite.setLayout(new GridLayout());

		final Composite spacer1 = new Composite(buttonsPaneComposite, SWT.NONE);
		final GridData spacer1GridData = new GridData();
		spacer1GridData.heightHint = CustomizationEditor.VERTICAL_SPACE_BEFORE_FIRST_BUTTON;
		spacer1.setLayoutData(spacer1GridData);

		this.btnEditFeatureValue = new Button(buttonsPaneComposite, SWT.PUSH);
		this.btnEditFeatureValue.setText(Messages.CustomizationEditor_editFeatureValue);
		this.btnEditFeatureValue.setLayoutData(buttonsGridData());
		this.btnEditFeatureValue.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				btnEditFeatureValueClicked();
			}
		});

		this.btnResetDefaults = new Button(buttonsPaneComposite, SWT.PUSH);
		this.btnResetDefaults.setText(Messages.CustomizationEditor_resetDefaults);
		this.btnResetDefaults.setLayoutData(buttonsGridData());
		this.btnResetDefaults.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				btnResetDefaultsClicked();
			}
		});

		final Composite spacer2 = new Composite(buttonsPaneComposite, SWT.NONE);
		final GridData spacer2GridData = new GridData();
		spacer2GridData.heightHint = CustomizationEditor.VERTICAL_SPACE_BETWEEN_BUTTON_GROUPS;
		spacer2.setLayoutData(spacer2GridData);

		final Label lblConditions = new Label(buttonsPaneComposite, SWT.NONE);
		lblConditions.setText(Messages.CustomizationEditor_conditions);

		this.btnAdd = new Button(buttonsPaneComposite, SWT.PUSH);
		this.btnAdd.setText(Messages.CustomizationEditor_add);
		this.btnAdd.setLayoutData(buttonsGridData());
		this.btnAdd.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				btnAddClicked();
			}
		});

		this.btnEdit = new Button(buttonsPaneComposite, SWT.PUSH);
		this.btnEdit.setText(Messages.CustomizationEditor_edit);
		this.btnEdit.setLayoutData(buttonsGridData());
		this.btnEdit.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				btnEditClicked();
			}
		});

		this.btnRemove = new Button(buttonsPaneComposite, SWT.PUSH);
		this.btnRemove.setText(Messages.CustomizationEditor_remove);
		this.btnRemove.setLayoutData(buttonsGridData());
		this.btnRemove.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				btnRemoveClicked();
			}
		});

		this.btnUp = new Button(buttonsPaneComposite, SWT.PUSH);
		this.btnUp.setText(Messages.CustomizationEditor_up);
		this.btnUp.setLayoutData(buttonsGridData());
		this.btnUp.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				btnUpClicked();
			}
		});

		this.btnDown = new Button(buttonsPaneComposite, SWT.PUSH);
		this.btnDown.setText(Messages.CustomizationEditor_down);
		this.btnDown.setLayoutData(buttonsGridData());
		this.btnDown.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				btnDownClicked();
			}
		});

		return buttonsPaneComposite;
	}

	private Object buttonsGridData() {
		final GridData buttonsGridData = new GridData();
		buttonsGridData.grabExcessHorizontalSpace = true;
		buttonsGridData.grabExcessVerticalSpace = false;
		buttonsGridData.horizontalAlignment = SWT.FILL;
		buttonsGridData.verticalAlignment = SWT.FILL;
		return buttonsGridData;
	}

	private MetaclassViewToolBar createToolBar(final Composite parent) {
		final Composite toolBarComposite = new Composite(parent, SWT.NONE);
		final GridData toolBarGridData = new GridData();
		toolBarGridData.grabExcessHorizontalSpace = true;
		toolBarGridData.horizontalAlignment = SWT.FILL;
		toolBarComposite.setLayoutData(toolBarGridData);

		return new MetaclassViewToolBar(toolBarComposite, this.metaclassViewConfiguration);
	}

	private void loadResource(final java.net.URI locationURI) {
		final ResourceSet resourceSet = new ResourceSetImpl();
		addResourceErrorsHandler(resourceSet);
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put(
				"uiCustom", new XMIResourceFactoryImpl()); //$NON-NLS-1$
		final URI uri = URI.createURI(locationURI.toString());
		final Resource resource = resourceSet.createResource(uri);
		try {
			resource.load(null);
		} catch (final IOException e) {
			Activator.logException("Error loading resource " + locationURI, e); //$NON-NLS-1$
			MessageDialog.openError(getSite().getShell(), Messages.CustomizationEditor_error,
					Messages.CustomizationEditor_errorLoadingResource);
			throw new IllegalStateException("Model couldn't be loaded", e); //$NON-NLS-1$
		}
		this.fMetamodelView = (MetamodelView) resource.getContents().get(0);
	}

	private void addResourceErrorsHandler(final ResourceSet resourceSet) {
		resourceSet.eAdapters().add(new EContentAdapter() {
			@Override
			public void notifyChanged(final Notification notification) {
				if (notification.getNotifier() instanceof Resource) {
					switch (notification.getFeatureID(Resource.class)) {
					case Resource.RESOURCE__IS_LOADED:
					case Resource.RESOURCE__ERRORS:
					case Resource.RESOURCE__WARNINGS:
						final Resource resource = (Resource) notification.getNotifier();
						final EList<Diagnostic> errors = resource.getErrors();
						if (!errors.isEmpty()) {
							CustomizationEditor.this.allErrors.addAll(errors);
							if (CustomizationEditor.this.errorsButton != null) {
								CustomizationEditor.this.errorsButton.setVisible(true);
							}
						}
						break;
					default:
						// do nothing
					}
				} else {
					super.notifyChanged(notification);
				}
			}

			@Override
			protected void setTarget(final Resource target) {
				basicSetTarget(target);
			}

			@Override
			protected void unsetTarget(final Resource targetResource) {
				basicUnsetTarget(targetResource);
			}
		});
	}

	private void metaclassSelectionChanged(final Object selectedElement) {
		this.cbAppliesToSubinstances.setEnabled(false);
		this.cbAppliesToSubinstances.setSelection(false);
		if (selectedElement == null) {
			this.customizationViewer.setInput(null);
			return;
		}

		CustomView customView = null;
		if (selectedElement instanceof EClass) {
			final EClass eClass = (EClass) selectedElement;
			customView = findOrCreateTypeView(eClass);
			if (!this.readOnly) {
				this.cbAppliesToSubinstances.setEnabled(true);
			}
			this.cbAppliesToSubinstances.setSelection(((TypeView) customView)
					.isAppliesToSubInstances());
		} else if (selectedElement instanceof Attribute) {
			final Attribute attribute = (Attribute) selectedElement;
			customView = findOrCreateAttributeView(attribute.getEClass(), attribute.getEAttribute());
		} else if (selectedElement instanceof Reference) {
			final Reference reference = (Reference) selectedElement;
			customView = findOrCreateReferenceView(reference.getEClass(), reference.getEReference());
		} else {
			throw new IllegalStateException("Unexpected element type: " //$NON-NLS-1$
					+ selectedElement.getClass().getSimpleName());
		}

		// for each possible feature, add an empty feature if it is not yet
		// present (for editing)
		final EList<CustomViewFeature> customizedFeatures = customView.getCustomizedFeatures();
		for (final CustomizableFeatures feature : CustomizableFeatures.values()) {
			if (appliesTo(feature, selectedElement)) {

				boolean present = false;
				for (final CustomViewFeature customViewFeature : customizedFeatures) {
					if (customViewFeature.getCustomizedFeature().equals(feature)) {
						present = true;
						break;
					}
				}
				if (!present) {
					final CustomViewFeature newCustomViewFeature = UicustomFactory.eINSTANCE
							.createCustomViewFeature();
					newCustomViewFeature.setCustomizedFeature(feature);
					customizedFeatures.add(newCustomViewFeature);
				}
			}
		}

		this.customizationViewer.setInput(customView);
	}

	private boolean appliesTo(final CustomizableFeatures feature, final Object selectedElement) {

		// XXX only the icons can be customized for facets (for now)
		if (selectedElement instanceof Facet) {
			switch (feature) {
			case ICON:
			case FACET_MAIN_ICON:
			case FACET_OVERLAY_ICON:
			case METACLASS_VISIBLE:
				return true;
			default:
				return false;
			}
		}

		switch (feature) {
		case HIDE_METACLASS_NAME:
		case METACLASS_VISIBLE:
			return selectedElement instanceof EClass;
		case FACET_MAIN_ICON:
		case FACET_OVERLAY_ICON:
			return false;
		default:
			return true;
		}
	}

	private TypeView findOrCreateTypeView(final EClass eClass) {
		final String className = ModelUtils.getMetaclassQualifiedName(eClass);
		if (className == null) {
			return null;
		}
		final EList<TypeView> types = this.fMetamodelView.getTypes();
		for (final TypeView typeView : types) {
			if (className.equals(typeView.getMetaclassName())) {
				return typeView;
			}
		}
		final TypeView typeView = UicustomFactory.eINSTANCE.createTypeView();
		typeView.setMetaclassName(className);
		this.fMetamodelView.getTypes().add(typeView);
		return typeView;
	}

	private AttributeView findOrCreateAttributeView(final EClass eClass, final EAttribute attribute) {
		final TypeView typeView = findOrCreateTypeView(eClass);

		final String attributeName = attribute.getName();
		final EList<AttributeView> attributeViews = typeView.getAttributes();
		for (final AttributeView attributeView : attributeViews) {
			if (attributeName.equals(attributeView.getAttributeName())) {
				return attributeView;
			}
		}
		final AttributeView attributeView = UicustomFactory.eINSTANCE.createAttributeView();
		attributeView.setAttributeName(attributeName);
		typeView.getAttributes().add(attributeView);
		return attributeView;
	}

	private ReferenceView findOrCreateReferenceView(final EClass eClass, final EReference reference) {
		final TypeView typeView = findOrCreateTypeView(eClass);

		final String referenceName = reference.getName();
		final EList<ReferenceView> referenceViews = typeView.getReferences();
		for (final ReferenceView referenceView : referenceViews) {
			if (referenceName.equals(referenceView.getReferenceName())) {
				return referenceView;
			}
		}
		final ReferenceView referenceView = UicustomFactory.eINSTANCE.createReferenceView();
		referenceView.setReferenceName(referenceName);
		typeView.getReferences().add(referenceView);
		return referenceView;
	}

	protected void customizationSelectionChanged(final Object selectedElement) {
		buttonsEnablementForSelection(selectedElement);
	}

	private void buttonsEnablementForSelection(final Object selectedElement) {
		if (this.readOnly) {
			return;
		}

		if (selectedElement == null) {
			this.btnEditFeatureValue.setEnabled(false);
			this.btnAdd.setEnabled(false);
			this.btnEdit.setEnabled(false);
			this.btnUp.setEnabled(false);
			this.btnDown.setEnabled(false);
			this.btnRemove.setEnabled(false);
		} else if (selectedElement instanceof CustomViewFeature) {
			this.btnEditFeatureValue.setEnabled(true);
			this.btnAdd.setEnabled(true);
			this.btnEdit.setEnabled(false);
			this.btnUp.setEnabled(false);
			this.btnDown.setEnabled(false);
			this.btnRemove.setEnabled(false);
		} else if (selectedElement instanceof FeatureValueCase) {
			final FeatureValueCase featureValueCase = (FeatureValueCase) selectedElement;
			this.btnEditFeatureValue.setEnabled(false);
			this.btnAdd.setEnabled(false);
			this.btnEdit.setEnabled(true);
			this.btnUp.setEnabled(!isFirst(featureValueCase));
			this.btnDown.setEnabled(!isLast(featureValueCase));
			this.btnRemove.setEnabled(true);
		}
	}

	private boolean isFirst(final FeatureValueCase featureValueCase) {
		final EList<FeatureValueCase> valueCases = featureValueCase.getFeature().getValueCases();
		if (valueCases.size() > 0 && valueCases.get(0).equals(featureValueCase)) {
			return true;
		}
		return false;
	}

	private boolean isLast(final FeatureValueCase featureValueCase) {
		final EList<FeatureValueCase> valueCases = featureValueCase.getFeature().getValueCases();
		if (valueCases.size() > 0 && valueCases.get(valueCases.size() - 1).equals(featureValueCase)) {
			return true;
		}
		return false;
	}

	private Object getSelectedElement() {
		final ITreeSelection selection = (ITreeSelection) this.customizationViewer.getSelection();
		return selection.getFirstElement();
	}

	private void btnEditFeatureValueClicked() {
		if (this.readOnly) {
			return;
		}
		final Object selectedElement = getSelectedElement();
		if (selectedElement instanceof CustomViewFeature) {
			final CustomViewFeature customViewFeature = (CustomViewFeature) selectedElement;
			final CustomizableFeatures customizedFeature = customViewFeature.getCustomizedFeature();

			// find the type (to handle scope)
			EClass customizedMetaclass = findCustomizedMetaclass(customViewFeature);

			final EditFeatureValueDialog editFeatureValueDialog = createDialogFor(
					customizedFeature, false, true, customizedMetaclass);

			if (editFeatureValueDialog == null) {
				return;
			}

			// initialize the dialog with current values
			final FeatureValue defaultValue = customViewFeature.getDefaultValue();
			if (defaultValue instanceof StaticFeatureValue) {
				editFeatureValueDialog.setSelectedValueType(ValueType.STATIC);
				final StaticFeatureValue staticFeatureValue = (StaticFeatureValue) defaultValue;
				editFeatureValueDialog.setInitialStaticValue(staticFeatureValue.getValue());
			} else if (defaultValue instanceof DerivedFeatureValue) {
				editFeatureValueDialog.setSelectedValueType(ValueType.DERIVED);
				final DerivedFeatureValue derivedFeatureValue = (DerivedFeatureValue) defaultValue;
				final ModelQuery valueCalculator = derivedFeatureValue.getValueCalculator();
				if (EMFUtil.tryResolve(valueCalculator)) {
					editFeatureValueDialog.setSelectedQuery(valueCalculator.getModelQuerySet()
							.getName(), valueCalculator.getName());
				}
			} else {
				editFeatureValueDialog.setSelectedValueType(ValueType.DEFAULT);
			}

			editFeatureValueDialog.open();
			if (editFeatureValueDialog.getReturnCode() == Window.OK) {
				final ValueType selectedValueType = editFeatureValueDialog.getSelectedValueType();
				if (selectedValueType == ValueType.STATIC) {
					final String staticValue = editFeatureValueDialog.getStaticValue();
					if (staticValue != null) {
						final StaticFeatureValue value = UicustomFactory.eINSTANCE
								.createStaticFeatureValue();
						value.setValue(staticValue);
						customViewFeature.setDefaultValue(value);
						setDirty(true);
					}
				} else if (selectedValueType == ValueType.DEFAULT) {
					customViewFeature.setDefaultValue(null);
					setDirty(true);
				} else if (selectedValueType == ValueType.DERIVED) {

					final String selectedQuerySetName = editFeatureValueDialog
							.getSelectedQuerySetName();
					final String selectedQueryName = editFeatureValueDialog.getSelectedQueryName();

					final ModelQuery modelQuery = findModelQuery(selectedQuerySetName,
							selectedQueryName);
					if (modelQuery != null) {
						final DerivedFeatureValue value = UicustomFactory.eINSTANCE
								.createDerivedFeatureValue();
						value.setValueCalculator(modelQuery);
						customViewFeature.setDefaultValue(value);
						setDirty(true);
					}
				}
			}
		}

		refresh();
	}

	private void btnResetDefaultsClicked() {
		final IStructuredSelection selection = (IStructuredSelection) this.metaclassViewer
				.getSelection();
		final Object selectedElement = selection.getFirstElement();
		if (selectedElement instanceof EClass) {
			final EClass eClass = (EClass) selectedElement;
			final TypeView typeView = findOrCreateTypeView(eClass);
			final String message = NLS.bind(
					Messages.CustomizationEditor_customizationsWillBeDeleted, eClass.getName());
			if (MessageDialog.openConfirm(getSite().getShell(),
					Messages.CustomizationEditor_resetDefaults, message)) {
				EList<CustomViewFeature> customizedFeatures = typeView.getCustomizedFeatures();
				for (CustomViewFeature customViewFeature : customizedFeatures) {
					customViewFeature.getValueCases().clear();
					customViewFeature.setDefaultValue(null);
				}
				setDirty(true);
				refresh();
			}
		}
	}

	private ModelQuery findModelQuery(final String querySetName, final String queryName) {
		final ModelQuerySet modelQuerySet = ModelQuerySetCatalog.getSingleton().getModelQuerySet(
				querySetName);
		if (modelQuerySet != null) {
			final EList<ModelQuery> queries = modelQuerySet.getQueries();
			ModelQuery query = null;
			for (final ModelQuery modelQuery : queries) {
				if (modelQuery.getName().equals(queryName)) {
					query = modelQuery;
					break;
				}
			}
			if (query == null) {
				MessageDialog
						.openError(getSite().getShell(),
								Messages.CustomizationEditor_nonExistingQuery, NLS.bind(
										Messages.CustomizationEditor_missingQuery, queryName,
										querySetName));
				return null;
			}
			return query;
		}
		MessageDialog.openError(getSite().getShell(),
				Messages.CustomizationEditor_nonExistingQuerySet, NLS.bind(
						Messages.CustomizationEditor_missingQuerySet, querySetName));
		return null;
	}

	private EditFeatureValueDialog createDialogFor(final CustomizableFeatures customizedFeature,
			final boolean withCondition, final boolean withDefault, final EClass customizedMetaclass) {
		switch (customizedFeature) {
		case BOLD:
			return createBooleanFeatureDialog(Messages.CustomizationEditor_bold, withCondition,
					withDefault, customizedMetaclass);
		case ITALIC:
			return createBooleanFeatureDialog(Messages.CustomizationEditor_italic, withCondition,
					withDefault, customizedMetaclass);
		case STRUCKTHROUGH:
			return createBooleanFeatureDialog(Messages.CustomizationEditor_strikethrough,
					withCondition, withDefault, customizedMetaclass);
		case UNDERLINED:
			return createBooleanFeatureDialog(Messages.CustomizationEditor_underlined,
					withCondition, withDefault, customizedMetaclass);
		case VISIBLE:
			return createBooleanFeatureDialog(Messages.CustomizationEditor_visible, withCondition,
					withDefault, customizedMetaclass);
		case METACLASS_VISIBLE:
			return createBooleanFeatureDialog(Messages.CustomizationEditor_metaclassVisible,
					withCondition, withDefault, customizedMetaclass);
		case LABEL:
			return createTextFeatureDialog(Messages.CustomizationEditor_text, withCondition,
					withDefault, customizedMetaclass);
		case COLOR:
		case BACKGROUND_COLOR:
			return createColorFeatureDialog(withCondition, withDefault, customizedMetaclass);
		case ICON:
		case FACET_MAIN_ICON:
			return createIconFeatureDialog(withCondition, withDefault, customizedMetaclass);
		case FACET_OVERLAY_ICON:
			return createOverlayIconFeatureDialog(withCondition, withDefault, customizedMetaclass);
		case FONT_NAME:
			return createFontFeatureDialog(withCondition, withDefault, customizedMetaclass);
		case HIDE_METACLASS_NAME:
			return createBooleanFeatureDialog(Messages.CustomizationEditor_hideMetaclassName,
					withCondition, withDefault, customizedMetaclass);
		default:
			Activator.logError("Unhandled CustomizableFeature : " //$NON-NLS-1$
					+ customizedFeature.getName());
			return null;
		}
	}

	private List<String> getAvailableQuerySets() {
		if (this.fMetamodelView.isAllQuerySetsAvailable()) {
			return null;
		}
		return this.fMetamodelView.getAvailableQuerySets();
	}

	private List<Class<?>> returnTypes(final Class<?>... types) {
		List<Class<?>> returnTypes = new ArrayList<Class<?>>();
		for (Class<?> clazz : types) {
			returnTypes.add(clazz);
		}
		return returnTypes;
	}

	private EditFeatureValueDialog createBooleanFeatureDialog(final String text,
			final boolean withCondition, final boolean withDefault, final EClass customizedMetaclass) {

		return new EditFeatureValueDialog(getSite().getShell(), withCondition, withDefault,
				getAvailableQuerySets(), returnTypes(Boolean.class, boolean.class),
				customizedMetaclass) {
			private Button button;

			@Override
			protected void createValueEditor(final Composite parent) {
				this.button = new Button(parent, SWT.CHECK);
				this.button.setText(text);
				this.button.setSelection(Boolean.parseBoolean(getInitialStaticValue()));
			}

			@Override
			protected void okPressed() {
				setStaticValue(Boolean.toString(this.button.getSelection()));
				super.okPressed();
			}
		};
	}

	private EditFeatureValueDialog createTextFeatureDialog(final String labelText,
			final boolean withCondition, final boolean withDefault, final EClass customizedMetaclass) {
		return new EditFeatureValueDialog(getSite().getShell(), withCondition, withDefault,
				getAvailableQuerySets(), returnTypes(String.class), customizedMetaclass) {
			private Text text;

			@Override
			protected void createValueEditor(final Composite parent) {
				final Composite composite = new Composite(parent, SWT.NONE);
				composite.setLayout(new GridLayout(2, false));
				composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
				final Label label = new Label(composite, SWT.NONE);
				label.setText(labelText);
				this.text = new Text(composite, SWT.BORDER);
				this.text.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
				if (getInitialStaticValue() != null) {
					this.text.setText(getInitialStaticValue());
				}
			}

			@Override
			protected void okPressed() {
				setStaticValue(this.text.getText());
				super.okPressed();
			}
		};
	}

	private EditFeatureValueDialog createColorFeatureDialog(final boolean withCondition,
			final boolean withDefault, final EClass customizedMetaclass) {
		return new EditFeatureValueDialog(getSite().getShell(), withCondition, withDefault,
				getAvailableQuerySets(), returnTypes(String.class), customizedMetaclass) {
			private ColorSelector colorSelector;

			@Override
			protected void createValueEditor(final Composite parent) {
				this.colorSelector = new ColorSelector(parent);
				if (getInitialStaticValue() != null) {
					final RGB color = UicustomUtil.decodeColor(getInitialStaticValue());
					if (color != null) {
						this.colorSelector.setColorValue(color);
					} else {
						Activator.logError("Wrong color format : " //$NON-NLS-1$
								+ getInitialStaticValue());
					}
				} else {
					this.colorSelector.setColorValue(new RGB(0, 0, 0));
				}
			}

			@Override
			protected void okPressed() {
				final RGB col = this.colorSelector.getColorValue();
				setStaticValue("(" + col.red + "," + col.green + "," //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						+ col.blue + ")"); //$NON-NLS-1$
				super.okPressed();
			}
		};
	}

	private EditFeatureValueDialog createIconFeatureDialog(final boolean withCondition,
			final boolean withDefault, final EClass customizedMetaclass) {
		return new EditFeatureValueDialog(getSite().getShell(), withCondition, withDefault,
				getAvailableQuerySets(), returnTypes(String.class), customizedMetaclass) {

			private Text text;

			@Override
			protected void createValueEditor(final Composite parent) {
				final Composite composite = new Composite(parent, SWT.NONE);
				composite.setLayout(new GridLayout(2, false));
				composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
				this.text = new Text(composite, SWT.BORDER);
				this.text.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
				if (getInitialStaticValue() != null) {
					this.text.setText(getInitialStaticValue());
				}

				final Button button = new Button(composite, SWT.PUSH);
				button.setText(Messages.CustomizationEditor_btnOpenElementSelectionDialogText);
				button.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(final SelectionEvent e) {
						IFile selectedFile = openIconSelectionDialog(getShell(), getText()
								.getText());
						if (selectedFile != null) {
							getText().setText(selectedFile.getFullPath().toString());
						}
					}
				});
			}

			@Override
			protected void okPressed() {
				setStaticValue(this.text.getText());
				super.okPressed();
			}

			public Text getText() {
				return this.text;
			}
		};
	}

	private EditFeatureValueDialog createOverlayIconFeatureDialog(final boolean withCondition,
			final boolean withDefault, final EClass customizedMetaclass) {
		return new EditFeatureValueDialog(getSite().getShell(), withCondition, withDefault,
				getAvailableQuerySets(), returnTypes(String.class), customizedMetaclass) {

			private Text text;
			private Combo combo;

			@Override
			protected void createValueEditor(final Composite parent) {
				final Composite composite = new Composite(parent, SWT.NONE);
				composite.setLayout(new GridLayout(2, false));
				composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
				this.text = new Text(composite, SWT.BORDER);
				this.text.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

				final Button button = new Button(composite, SWT.PUSH);
				button.setText(Messages.CustomizationEditor_btnOpenElementSelectionDialogText);
				button.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(final SelectionEvent e) {
						IFile selectedFile = openIconSelectionDialog(getShell(), getText()
								.getText());
						if (selectedFile != null) {
							getText().setText(selectedFile.getFullPath().toString());
						}
					}
				});
				this.combo = new Combo(composite, SWT.DROP_DOWN | SWT.READ_ONLY);
				for (OverlayIconPosition overlayIconPosition : OverlayIconPosition.values()) {
					this.combo.add(overlayIconPosition.name());
				}

				if (getInitialStaticValue() != null) {
					try {
						OverlayIconInfo overlayIcon = UicustomUtil
								.decodeOverlayIcon(getInitialStaticValue());
						this.text.setText(overlayIcon.getPath());
						this.combo.setText(overlayIcon.getIconPosition().name());
					} catch (Exception e) {
						MoDiscoLogger.logError(e, "Error decoding overlay icon", Activator //$NON-NLS-1$
								.getDefault());
					}
				}
				if (this.combo.getSelectionIndex() == -1) {
					this.combo.select(0);
				}
			}

			@Override
			protected void okPressed() {
				OverlayIconPosition iconPosition = OverlayIconPosition
						.valueOf(this.combo.getText());
				setStaticValue(UicustomUtil.encodeOverlayIcon(this.text.getText(), iconPosition));
				super.okPressed();
			}

			public Text getText() {
				return this.text;
			}
		};
	}

	private IFile openIconSelectionDialog(final Shell shell, final String initialText) {
		final ElementTreeSelectionDialog iconSelectionDialog = new ElementTreeSelectionDialog(
				shell, new WorkbenchLabelProvider(), new WorkbenchContentProvider());
		iconSelectionDialog.setInput(ResourcesPlugin.getWorkspace().getRoot());
		iconSelectionDialog.setAllowMultiple(false);
		iconSelectionDialog.setMessage(Messages.CustomizationEditor_selectAnIcon);
		iconSelectionDialog.setTitle(Messages.CustomizationEditor_iconSelection);
		final IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(initialText);
		if (resource != null) {
			iconSelectionDialog.setInitialSelection(resource);
		}
		final int result = iconSelectionDialog.open();
		if (result == Window.OK) {
			final Object firstResult = iconSelectionDialog.getFirstResult();
			if (firstResult instanceof IFile) {
				final IFile file = (IFile) firstResult;
				return file;
			}
			MessageDialog.openWarning(CustomizationEditor.this.getSite().getShell(),
					Messages.CustomizationEditor_invalidSelection,
					Messages.CustomizationEditor_notFileSelected);
		}
		return null;
	}

	private EditFeatureValueDialog createFontFeatureDialog(final boolean withCondition,
			final boolean withDefault, final EClass customizedMetaclass) {
		return new EditFeatureValueDialog(getSite().getShell(), withCondition, withDefault,
				getAvailableQuerySets(), returnTypes(String.class), customizedMetaclass) {

			private Text text;

			@Override
			protected void createValueEditor(final Composite parent) {
				final Composite composite = new Composite(parent, SWT.NONE);
				composite.setLayout(new GridLayout(2, false));
				composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
				this.text = new Text(composite, SWT.BORDER);
				this.text.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
				if (getInitialStaticValue() != null) {
					this.text.setText(getInitialStaticValue());
				}

				final Button button = new Button(composite, SWT.PUSH);
				button.setText(Messages.CustomizationEditor_btnOpenFontDialogText);
				button.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(final SelectionEvent e) {
						final FontDialog fontDialog = new FontDialog(CustomizationEditor.this
								.getSite().getShell());
						final int fontHeight = 10;
						fontDialog.setFontList(new FontData[] { new FontData(getText().getText(),
								fontHeight, 0) });
						final FontData font = fontDialog.open();
						if (font != null) {
							getText().setText(font.getName());
						}
					}
				});
			}

			@Override
			protected void okPressed() {
				setStaticValue(this.text.getText());
				super.okPressed();
			}

			public Text getText() {
				return this.text;
			}
		};
	}

	private void btnAddClicked() {
		if (this.readOnly) {
			return;
		}
		final Object selectedElement = getSelectedElement();
		if (selectedElement instanceof CustomViewFeature) {
			final CustomViewFeature customViewFeature = (CustomViewFeature) selectedElement;
			final CustomizableFeatures customizedFeature = customViewFeature.getCustomizedFeature();

			// find the type (to handle scope)
			EClass customizedMetaclass = findCustomizedMetaclass(customViewFeature);

			final EditFeatureValueDialog editFeatureValueDialog = createDialogFor(
					customizedFeature, true, false, customizedMetaclass);

			FeatureValueCase newValueCase = null;
			if (editFeatureValueDialog != null) {
				editFeatureValueDialog.open();
				if (editFeatureValueDialog.getReturnCode() == Window.OK) {
					final ValueType selectedValueType = editFeatureValueDialog
							.getSelectedValueType();
					if (selectedValueType == ValueType.STATIC) {
						final String staticValue = editFeatureValueDialog.getStaticValue();
						if (staticValue != null) {
							final ModelQuery conditionQuery = findModelQuery(editFeatureValueDialog
									.getSelectedConditionQuerySetName(), editFeatureValueDialog
									.getSelectedConditionQueryName());
							if (conditionQuery != null) {
								final StaticFeatureValue value = UicustomFactory.eINSTANCE
										.createStaticFeatureValue();
								value.setValue(staticValue);

								newValueCase = UicustomFactory.eINSTANCE.createFeatureValueCase();
								newValueCase.setValue(value);

								newValueCase.setCondition(conditionQuery);
								customViewFeature.getValueCases().add(newValueCase);
								setDirty(true);
							}
						}
					} else if (selectedValueType == ValueType.DERIVED) {

						final ModelQuery conditionQuery = findModelQuery(editFeatureValueDialog
								.getSelectedConditionQuerySetName(), editFeatureValueDialog
								.getSelectedConditionQueryName());

						final ModelQuery query = findModelQuery(editFeatureValueDialog
								.getSelectedQuerySetName(), editFeatureValueDialog
								.getSelectedQueryName());

						if (conditionQuery != null && query != null) {
							final DerivedFeatureValue value = UicustomFactory.eINSTANCE
									.createDerivedFeatureValue();
							value.setValueCalculator(query);

							newValueCase = UicustomFactory.eINSTANCE.createFeatureValueCase();
							newValueCase.setValue(value);
							newValueCase.setCondition(conditionQuery);

							customViewFeature.getValueCases().add(newValueCase);
							setDirty(true);
						}

					}
				}
			}
			refresh();
			if (newValueCase != null) {
				this.customizationViewer.getViewer().setSelection(
						new StructuredSelection(newValueCase), true);
			}
		}
	}

	private EClass findCustomizedMetaclass(final CustomViewFeature customViewFeature) {
		CustomView customView = customViewFeature.getCustomView();
		List<EClass> metaclasses = EMFUtil.getMetaclasses(this.fMetamodel);

		if (customView instanceof TypeView) {
			TypeView typeView = (TypeView) customView;
			return EMFUtil.findMetaclassWithName(typeView.getMetaclassName(), metaclasses);
		} else if (customView instanceof AttributeView) {
			AttributeView attributeView = (AttributeView) customView;
			return EMFUtil.findMetaclassWithName(attributeView.getType().getMetaclassName(),
					metaclasses);
		} else if (customView instanceof ReferenceView) {
			ReferenceView referenceView = (ReferenceView) customView;
			return EMFUtil.findMetaclassWithName(referenceView.getType().getMetaclassName(),
					metaclasses);
		} else {
			throw new IllegalStateException("Unknown CustomView type: " //$NON-NLS-1$
					+ customView.getClass().getSimpleName());
		}

	}

	private void btnEditClicked() {
		if (this.readOnly) {
			return;
		}
		final Object selectedElement = getSelectedElement();
		if (selectedElement instanceof FeatureValueCase) {
			final FeatureValueCase featureValueCase = (FeatureValueCase) selectedElement;
			final CustomViewFeature customViewFeature = featureValueCase.getFeature();
			final CustomizableFeatures customizedFeature = customViewFeature.getCustomizedFeature();

			// find the type (to handle scope)
			final EClass customizedMetaclass = findCustomizedMetaclass(customViewFeature);

			final EditFeatureValueDialog editFeatureValueDialog = createDialogFor(
					customizedFeature, true, false, customizedMetaclass);

			final FeatureValue featureValue = featureValueCase.getValue();
			if (featureValue instanceof StaticFeatureValue) {
				final StaticFeatureValue staticFeatureValue = (StaticFeatureValue) featureValue;
				editFeatureValueDialog.setSelectedValueType(ValueType.STATIC);
				editFeatureValueDialog.setInitialStaticValue(staticFeatureValue.getValue());
			} else if (featureValue instanceof DerivedFeatureValue) {
				final DerivedFeatureValue derivedFeatureValue = (DerivedFeatureValue) featureValue;
				editFeatureValueDialog.setSelectedValueType(ValueType.DERIVED);
				final ModelQuery valueCalculator = derivedFeatureValue.getValueCalculator();
				if (EMFUtil.tryResolve(valueCalculator)) {
					editFeatureValueDialog.setSelectedQuery(valueCalculator.getModelQuerySet()
							.getName(), valueCalculator.getName());
				}
			}

			final ModelQuery condition = featureValueCase.getCondition();
			if (condition != null && EMFUtil.tryResolve(condition)) {
				editFeatureValueDialog.setSelectedConditionQuery(condition.getModelQuerySet()
						.getName(), condition.getName());
			}

			if (editFeatureValueDialog != null) {
				editFeatureValueDialog.open();
				if (editFeatureValueDialog.getReturnCode() == Window.OK) {
					final ValueType selectedValueType = editFeatureValueDialog
							.getSelectedValueType();
					if (selectedValueType == ValueType.STATIC) {
						final String staticValue = editFeatureValueDialog.getStaticValue();
						if (staticValue != null) {
							final StaticFeatureValue value = UicustomFactory.eINSTANCE
									.createStaticFeatureValue();
							value.setValue(staticValue);

							featureValueCase.setValue(value);
							setDirty(true);
						}
					} else if (selectedValueType == ValueType.DERIVED) {
						final DerivedFeatureValue value = UicustomFactory.eINSTANCE
								.createDerivedFeatureValue();

						final ModelQuery query = findModelQuery(editFeatureValueDialog
								.getSelectedQuerySetName(), editFeatureValueDialog
								.getSelectedQueryName());

						final ModelQuery conditionQuery = findModelQuery(editFeatureValueDialog
								.getSelectedConditionQuerySetName(), editFeatureValueDialog
								.getSelectedConditionQueryName());

						if (query != null && conditionQuery != null) {
							value.setValueCalculator(query);
							featureValueCase.setCondition(conditionQuery);
							featureValueCase.setValue(value);

							setDirty(true);
						}
					}
				}
			}
		}

		refresh();
	}

	private void btnRemoveClicked() {
		if (this.readOnly) {
			return;
		}
		final Object selectedElement = getSelectedElement();
		if (selectedElement instanceof FeatureValueCase) {
			final FeatureValueCase featureValueCase = (FeatureValueCase) selectedElement;
			final EList<FeatureValueCase> list = featureValueCase.getFeature().getValueCases();
			list.remove(featureValueCase);
			setDirty(true);
			refresh();
		}
	}

	private void btnUpClicked() {
		if (this.readOnly) {
			return;
		}
		final Object selectedElement = getSelectedElement();
		if (selectedElement instanceof FeatureValueCase) {
			final FeatureValueCase featureValueCase = (FeatureValueCase) selectedElement;
			final EList<FeatureValueCase> list = featureValueCase.getFeature().getValueCases();
			// swap the element with the one before
			final int index = list.indexOf(featureValueCase);
			if (index - 1 >= 0) {
				list.move(index - 1, index);
				setDirty(true);
			}
			refresh();
		}
	}

	private void btnDownClicked() {
		if (this.readOnly) {
			return;
		}
		final Object selectedElement = getSelectedElement();
		if (selectedElement instanceof FeatureValueCase) {
			final FeatureValueCase featureValueCase = (FeatureValueCase) selectedElement;
			final EList<FeatureValueCase> list = featureValueCase.getFeature().getValueCases();
			// swap the element with the one after
			final int index = list.indexOf(featureValueCase);
			if (index + 1 < list.size()) {
				list.move(index + 1, index);
				setDirty(true);
			}
			refresh();
		}
	}

	private void appliesToSubinstancesChanged(final boolean newValue) {
		final IStructuredSelection selection = (IStructuredSelection) this.metaclassViewer
				.getSelection();
		final Object selectedElement = selection.getFirstElement();
		if (selectedElement instanceof EClass) {
			final EClass eClass = (EClass) selectedElement;
			final TypeView typeView = findOrCreateTypeView(eClass);
			typeView.setAppliesToSubInstances(newValue);
			setDirty(true);
			refresh();
		}
	}

	private void refresh() {
		this.customizationViewer.refresh();
		this.metaclassViewer.refresh();
		buttonsEnablementForSelection(getSelectedElement());
	}

	@Override
	public void doSave(final IProgressMonitor monitor) {
		if (this.readOnly) {
			return;
		}

		final IFileEditorInput fileEditorInput = (IFileEditorInput) getEditorInput();
		final String locationURI = fileEditorInput.getFile().getLocationURI().toString();

		final ResourceSet resourceSet = new ResourceSetImpl();
		resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put(
				"uiCustom", new XMIResourceFactoryImpl()); //$NON-NLS-1$
		final Resource resource = resourceSet.createResource(URI.createURI(locationURI));

		// save a clean copy of the model, in which useless elements are removed
		final MetamodelView metamodelViewCopy = EcoreUtil.copy(this.fMetamodelView);
		resource.getContents().add(cleanModel(metamodelViewCopy));

		try {
			resource.save(null);
			setDirty(false);
			// synchronize changes with Workspace
			fileEditorInput.getFile().refreshLocal(IResource.DEPTH_ZERO, monitor);
		} catch (final Exception e) {
			Activator.logException(e);
		}
	}

	/** Clean the customization model by removing unnecessary model elements */
	private MetamodelView cleanModel(final MetamodelView metamodelView) {
		final ListIterator<TypeView> typesIterator = metamodelView.getTypes().listIterator();
		while (typesIterator.hasNext()) {
			final TypeView typeView = typesIterator.next();

			boolean containsInformation = true;

			if (!clean(typeView)) {
				containsInformation = false;
			}

			final ListIterator<AttributeView> attributesIterator = typeView.getAttributes()
					.listIterator();
			while (attributesIterator.hasNext()) {
				final AttributeView attributeView = attributesIterator.next();
				if (clean(attributeView)) {
					containsInformation = true;
				} else {
					attributesIterator.remove();
				}
			}

			final ListIterator<ReferenceView> referencesIterator = typeView.getReferences()
					.listIterator();
			while (referencesIterator.hasNext()) {
				final ReferenceView referenceView = referencesIterator.next();
				if (clean(referenceView)) {
					containsInformation = true;
				} else {
					referencesIterator.remove();
				}
			}

			if (!containsInformation) {
				typesIterator.remove();
			}
		}
		return metamodelView;
	}

	/**
	 * Clean the custom view
	 * 
	 * @return whether the custom view contains information and thus must be
	 *         kept
	 */
	private boolean clean(final CustomView customView) {
		final EList<CustomViewFeature> customizedFeatures = customView.getCustomizedFeatures();

		final ListIterator<CustomViewFeature> featuresIterator = customizedFeatures.listIterator();
		while (featuresIterator.hasNext()) {
			final CustomViewFeature customViewFeature = featuresIterator.next();
			if (customViewFeature.getDefaultValue() == null
					&& customViewFeature.getValueCases().size() == 0) {
				featuresIterator.remove();
			}
		}

		return customizedFeatures.size() > 0;
	}

	@Override
	public void doSaveAs() {
		// nothing
	}

	@Override
	public boolean isDirty() {
		return this.dirty && !this.readOnly;
	}

	private void setDirty(final boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(IEditorPart.PROP_DIRTY);
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
		// nothing
	}

	/** A listener to save user settings when the editor is closed */
	private class SaveSettingsOnCloseListener implements IPartListener {
		public void partClosed(final IWorkbenchPart part) {
			if (part == CustomizationEditor.this) {
				CustomizationEditor.this.metaclassViewConfiguration.save();
				getSite().getPage().removePartListener(this);
			}
		}

		public void partOpened(final IWorkbenchPart part) {
			// nothing
		}

		public void partDeactivated(final IWorkbenchPart part) {
			// nothing
		}

		public void partBroughtToTop(final IWorkbenchPart part) {
			// nothing
		}

		public void partActivated(final IWorkbenchPart part) {
			// nothing
		}
	}

	@Override
	public void dispose() {
		super.dispose();
		this.toolkit.dispose();
	}
}
