/**
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Gregoire DUPE (Mia-Software) - initial API and implementation
 */
package org.eclipse.gmt.modisco.infra.facet.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.gmt.modisco.infra.common.core.internal.builder.EcoreCatalog;
import org.eclipse.gmt.modisco.infra.facet.FacetSet;
import org.eclipse.gmt.modisco.infra.facet.core.exception.ModiscoFacetException;
import org.eclipse.gmt.modisco.infra.facet.core.internal.Activator;
import org.eclipse.gmt.modisco.infra.query.core.ModelQuerySetCatalog;

public final class FacetSetCatalog extends EcoreCatalog {

	private static final String REGISTRATION_EXTENSION_POINT_ID = "org.eclipse.gmt.modisco.infra.facet.registration"; //$NON-NLS-1$

	private static FacetSetCatalog singleton = null;

	public static FacetSetCatalog getSingleton() {
		if (FacetSetCatalog.singleton == null) {
			EcoreCatalog.getSingleton();
			ModelQuerySetCatalog.getSingleton();
			FacetSetCatalog.singleton = new FacetSetCatalog();
		}
		return FacetSetCatalog.singleton;
	}

	private HashMap<EPackage, HashSet<FacetSet>> ePackageToDefaultFacetSetMap;

	public HashMap<EPackage, HashSet<FacetSet>> getEPackageToDefaultFacetSetMap() {
		if (this.ePackageToDefaultFacetSetMap == null) {
			this.ePackageToDefaultFacetSetMap = new HashMap<EPackage, HashSet<FacetSet>>();
		}
		return this.ePackageToDefaultFacetSetMap;
	}

	private FacetSetCatalog() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRootObjectName(final EObject rootObject) {
		return ((FacetSet) rootObject).getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRootObjectNsUri(final EObject rootObject) {
		return ((FacetSet) rootObject).getNsURI();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Class<?> getRootClass() {
		return FacetSet.class;
	}

	/**
	 * This method returns the facet set of the resource having for name the
	 * "name" parameter value.
	 * 
	 * @param name
	 *            the name of a resource contained in the catalog
	 * @return return a facet set
	 */
	public FacetSet getFacetSet(final String name) {
		return (FacetSet) this.getRootObject(name);
	}

	public HashSet<FacetSet> getDefaultFacetSets(final EPackage ePackage) {
		return this.getEPackageToDefaultFacetSetMap().get(ePackage);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getRegistryExtensionPoint() {
		return FacetSetCatalog.REGISTRATION_EXTENSION_POINT_ID;
	}

	/**
	 * This methods returns the facet sets of all the resources contained in the
	 * catalog.
	 * 
	 * @return returns all the facet sets
	 */
	public Collection<FacetSet> getAllFacetSets() {
		Collection<FacetSet> result = new ArrayList<FacetSet>();
		for (EObject eObject : this.getAllRootObjects()) {
			if (eObject instanceof FacetSet) {
				FacetSet facetSet = (FacetSet) eObject;
				result.add(facetSet);
			}
		}
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getMoDiscoSubProtocol() {
		return "facet"; //$NON-NLS-1$
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws ModiscoFacetException
	 */
	@Override
	protected EObject openResource(final URI uri, final IResource file) {
		EObject result = super.openResource(uri, file);
		if (result instanceof FacetSet) {
			FacetSet facetSet = (FacetSet) result;
			EAnnotation annotation = facetSet.getEAnnotation("default"); //$NON-NLS-1$
			if (annotation != null) {
				if (annotation.getReferences().size() == 1
						&& annotation.getReferences().get(0) instanceof EPackage) {
					EPackage ePackage = (EPackage) annotation.getReferences().get(0);
					HashSet<FacetSet> facetSets = this.getEPackageToDefaultFacetSetMap().get(
							ePackage);
					if (facetSets == null) {
						facetSets = new HashSet<FacetSet>();
						this.getEPackageToDefaultFacetSetMap().put(ePackage, facetSets);
					}
					facetSets.add(facetSet);
				}
			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.gmt.modisco.infra.common.core.internal.builder.EcoreCatalog#getActivator()
	 */
	@Override
	protected Plugin getActivator() {
		return Activator.getDefault();
	}
}