/*******************************************************************************
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.common.core.internal.adapters.instances;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.infra.common.core.internal.CommonModiscoActivator;
import org.eclipse.gmt.modisco.infra.common.core.internal.logging.MoDiscoLogger;

/**
 * An adapter that acts as a cache for lists of instances by EClass. It works by
 * listening to changes in the model and updating the cache accordingly.
 * <p>
 * It must be created on a {@link Resource} through
 * {@link MetaclassInstancesAdapterFactory}, by doing the following: <code>
 * MetaclassInstances instances = (MetaclassInstances) MetaclassInstancesAdapterFactory.getInstance().adapt(resource, MetaclassInstances.class);
 * </code>
 */
public class MetaclassInstancesAdapter extends AdapterImpl implements MetaclassInstances {

	private final Resource resource;

	/** All the model elements of the metaclass */
	private Map<EClass, Set<EObject>> instancesByEClass;
	/** All the model elements of the metaclass or one of its sub-types */
	private Map<EClass, Set<EObject>> instancesByType;

	/** model change listeners */
	private final List<ModelChangeListener> listeners = new ArrayList<ModelChangeListener>();

	protected MetaclassInstancesAdapter(final Resource resource, final boolean clearCache) {
		this.resource = resource;
		// this is to allow sub-classes to do initialization in their
		// constructor before calling clearCache()
		if (clearCache) {
			clearCache();
		}
	}

	public void clearCache() {
		this.instancesByEClass = new HashMap<EClass, Set<EObject>>();
		this.instancesByType = new HashMap<EClass, Set<EObject>>();

		TreeIterator<EObject> allContents = this.resource.getAllContents();
		while (allContents.hasNext()) {
			EObject eObject = allContents.next();
			addModelElement(eObject, false);
		}
	}

	/**
	 * Add a new element to the set of elements corresponding to its metaclass.
	 * 
	 * @param element
	 *            the element to add
	 * @param recursively
	 *            whether to also add elements contained in the given element
	 */
	protected void addModelElement(final EObject element, final boolean recursively) {
		// make sure there is only one occurrence
		element.eAdapters().remove(this);
		// so that this element will notify us when it's changed
		element.eAdapters().add(this);

		List<EClass> eClasses = getEClasses(element);
		for (EClass eClass : eClasses) {
			if (eClass == null) {
				MoDiscoLogger.logWarning(
						"Element has null eClass: " + element, CommonModiscoActivator.getDefault()); //$NON-NLS-1$
				continue;
			}

			// direct instances
			associateToEClass(element, eClass);

			// instances of subclasses (including this class)
			associateToType(element, eClass);
			EList<EClass> allSuperTypes = eClass.getEAllSuperTypes();
			for (EClass superType : allSuperTypes) {
				associateToType(element, superType);
			}

			// fireModelChanged();

			if (recursively) {
				// also add the contained elements recursively
				EList<EObject> contents = element.eContents();
				for (EObject contained : contents) {
					addModelElement(contained, true);
				}
			}
		}
	}

	/**
	 * Remove this instance from the list of instances of the metaclass and its
	 * super-classes
	 */
	protected void removeModelElement(final EObject element) {
		element.eAdapters().remove(this);

		List<EClass> eClasses = getEClasses(element);
		for (EClass eClass : eClasses) {
			if (eClass == null) {
				MoDiscoLogger.logWarning(
						"Element has null eClass: " + element, CommonModiscoActivator.getDefault()); //$NON-NLS-1$
				continue;
			}

			disassociateFromEClass(element, eClass);

			disassociateFromType(element, eClass);
			EList<EClass> allSuperTypes = eClass.getEAllSuperTypes();
			for (EClass superType : allSuperTypes) {
				disassociateFromType(element, superType);
			}
		}
		
		// also remove the contained elements recursively
		EList<EObject> contents = element.eContents();
		for (EObject contained : contents) {
			removeModelElement(contained);
		}
	}

	@Override
	public void notifyChanged(final Notification msg) {
		// handle the change
		handleChanged(msg);
		// pass on the notification to listeners
		notifyModelChanged(msg);
	}

	private void handleChanged(final Notification msg) {
		final int eventType = msg.getEventType();
		final Object feature = msg.getFeature();
		final Object oldValue = msg.getOldValue();
		final Object newValue = msg.getNewValue();

		switch (eventType) {
		case Notification.ADD:
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				if (!reference.isContainment()) {
					return;
				}
			}
			if (newValue instanceof EObject) {
				addModelElement((EObject) newValue, true);
			}
			break;

		case Notification.REMOVE:
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				if (!reference.isContainment()) {
					return;
				}
			}
			if (oldValue instanceof EObject) {
				removeModelElement((EObject) oldValue);
			}
			break;

		case Notification.ADD_MANY:
			if (newValue instanceof EList<?>) {
				EList<?> eList = (EList<?>) newValue;
				for (Object object : eList) {
					if (object instanceof EObject) {
						addModelElement((EObject) object, true);
					}
				}
			}
			break;

		case Notification.REMOVE_MANY:
			if (newValue instanceof EList<?>) {
				EList<?> eList = (EList<?>) newValue;
				for (Object object : eList) {
					if (object instanceof EObject) {
						removeModelElement((EObject) object);
					}
				}
			}
			break;

		// features that have a multiplicity of 1
		case Notification.SET:
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				/*
				 * setting a reference should add or remove elements only if the
				 * reference is a composition link
				 */
				if (!reference.isContainment()) {
					break;
				}

				if (newValue == null) {
					if (oldValue instanceof EObject) {
						removeModelElement((EObject) oldValue);
					}
				} else if (newValue instanceof EObject) {
					addModelElement((EObject) newValue, true);
				}
			}
			break;

		case Notification.UNSET:
			if (feature instanceof EReference) {
				EReference reference = (EReference) feature;
				/*
				 * unsetting a reference should remove the element only if the
				 * reference is a composition link
				 */
				if (!reference.isContainment()) {
					break;
				}
			}
			if (oldValue instanceof EObject) {
				removeModelElement((EObject) oldValue);
			}
			break;
		default:
		}
	}

	/**
	 * Normally, an {@link EObject} only has one {@link EClass}, but this is to
	 * allow handling Facets on a derived class.
	 */
	protected List<EClass> getEClasses(final EObject element) {
		return Collections.singletonList(element.eClass());
	}

	protected void associateToEClass(final EObject element, final EClass eClass) {
		Set<EObject> instancesForEClass = this.instancesByEClass.get(eClass);
		if (instancesForEClass == null) {
			instancesForEClass = new LinkedHashSet<EObject>();
			this.instancesByEClass.put(eClass, instancesForEClass);
		}
		instancesForEClass.add(element);
	}

	protected void associateToType(final EObject element, final EClass eClass) {
		Set<EObject> instancesForType = this.instancesByType.get(eClass);
		if (instancesForType == null) {
			instancesForType = new LinkedHashSet<EObject>();
			this.instancesByType.put(eClass, instancesForType);
		}
		instancesForType.add(element);
	}

	protected void disassociateFromEClass(final EObject element, final EClass eClass) {
		Set<EObject> instancesForEClass = this.instancesByEClass.get(eClass);
		if (instancesForEClass != null) {
			instancesForEClass.remove(element);
			if (instancesForEClass.isEmpty()) {
				// last instance removed
				this.instancesByEClass.remove(eClass);
				// TODO: notify
			}
		}
	}

	protected void disassociateFromType(final EObject element, final EClass eClass) {
		Set<EObject> instancesForType = this.instancesByType.get(eClass);
		if (instancesForType != null) {
			instancesForType.remove(element);
			if (instancesForType.isEmpty()) {
				// last instance removed
				this.instancesByType.remove(eClass);
				// TODO: notify
			}
		}
	}

	@Override
	public boolean isAdapterForType(final Object type) {
		return (type == MetaclassInstances.class);
	}

	public Set<EObject> getInstances(final EClass eClass, final boolean includingSubclasses) {
		final Set<EObject> set;
		if (includingSubclasses) {
			set = this.instancesByType.get(eClass);
		} else {
			set = this.instancesByEClass.get(eClass);
		}
		if (set == null) {
			return Collections.emptySet();
		}
		return Collections.unmodifiableSet(set);
	}

	protected Map<EClass, Set<EObject>> getInstancesByEClass() {
		return this.instancesByEClass;
	}

	protected Map<EClass, Set<EObject>> getInstancesByType() {
		return this.instancesByType;
	}

	protected Resource getResource() {
		return this.resource;
	}

	public void addListener(final ModelChangeListener listener) {
		if (!this.listeners.contains(listener)) {
			this.listeners.add(listener);
		}
	}

	public void removeListener(final ModelChangeListener listener) {
		this.listeners.remove(listener);
	}

	protected void notifyModelChanged(final Notification msg) {
		for (ModelChangeListener listener : this.listeners) {
			listener.modelChanged(msg);
		}
	}
}
