/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Grgoire Dup (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.common.core.internal.builder;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.gmt.modisco.infra.common.core.internal.CommonModiscoActivator;
import org.eclipse.gmt.modisco.infra.common.core.internal.logging.MoDiscoLogger;

/**
 * This class is the MoDisco project builder implementation.
 * 
 * @author Grgoire Dup (Mia-Software)
 */
public class ModiscoProjectBuilder extends IncrementalProjectBuilder {

	private static final String EXTENSION_POINT_ID = "org.eclipse.gmt.modisco.common.core.builder.register"; //$NON-NLS-1$
	/**
	 * The MoDisco builder id
	 */
	public static final String BUILDER_NAME = "org.eclipse.gmt.modisco.common.core.builder"; //$NON-NLS-1$
	private static ArrayList<BuilderDescriptor> modiscoBuilders = null;

	public ModiscoProjectBuilder() {
		super();
		if (ModiscoProjectBuilder.modiscoBuilders == null) {
			ModiscoProjectBuilder.modiscoBuilders = new ArrayList<BuilderDescriptor>();
			IConfigurationElement[] configs = Platform.getExtensionRegistry()
					.getConfigurationElementsFor(ModiscoProjectBuilder.EXTENSION_POINT_ID);
			for (IConfigurationElement config : configs) {
				try {
					String id = config.getDeclaringExtension().getUniqueIdentifier();
					IModiscoBuilder builderInst = (IModiscoBuilder) config
							.createExecutableExtension("class"); //$NON-NLS-1$
					BuilderDescriptor builderDescriptor = new BuilderDescriptor(builderInst, id);
					ModiscoProjectBuilder.modiscoBuilders.add(builderDescriptor);
					for (IConfigurationElement depends : config.getChildren("depends")) { //$NON-NLS-1$
						builderDescriptor.getDependsOn().add(depends.getAttribute("builder")); //$NON-NLS-1$
					}
				} catch (Exception e) {
					MoDiscoLogger.logError(e, CommonModiscoActivator.getDefault());
				}
			}
			Collections.sort(ModiscoProjectBuilder.modiscoBuilders);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void clean(final IProgressMonitor monitor) throws CoreException {
		Iterator<BuilderDescriptor> builders = ModiscoProjectBuilder.modiscoBuilders.iterator();
		while (builders.hasNext()) {
			BuilderDescriptor builder = builders.next();
			builder.getBuilderInst().clean(this, monitor);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected IProject[] build(final int kind, @SuppressWarnings("unchecked") final Map args, final IProgressMonitor monitor)
			throws CoreException {
		Iterator<BuilderDescriptor> builders = ModiscoProjectBuilder.modiscoBuilders.iterator();
		while (builders.hasNext()) {
			BuilderDescriptor builder = builders.next();
			builder.getBuilderInst().build(this, kind, args, monitor);
		}
		return null;
	}

	/**
	 * This class is used to hold the builder id, the builder instance and the
	 * builder dependencies in a single structure.
	 */
	private class BuilderDescriptor implements Comparable<ModiscoProjectBuilder.BuilderDescriptor> {
		private final IModiscoBuilder builderInst;

		private final List<String> dependsOn = new ArrayList<String>();

		private final String id;

		/**
		 * @param builderInst
		 *            an instance of the described builder
		 * @param id
		 *            the id of the described builder
		 */
		public BuilderDescriptor(final IModiscoBuilder builderInst, final String id) {
			this.builderInst = builderInst;
			this.id = id;
		}

		/**
		 * @return the list of dependencies on other MoDisco builders.
		 */
		public List<String> getDependsOn() {
			return this.dependsOn;
		}

		/**
		 * @return the instance of the described builder
		 */
		public IModiscoBuilder getBuilderInst() {
			return this.builderInst;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Comparable#compareTo(java.lang.Object)
		 */
		public int compareTo(final BuilderDescriptor builderDescriptor) {
			int result = 0;
			if (this.dependsOn.contains(builderDescriptor.id)) {
				result = 1;
			}
			if (builderDescriptor.dependsOn.contains(this.id)) {
				result = -1;
			}
			return result;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString() {
			return this.id + "(" + this.builderInst.getClass().getName() + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

}
