/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.query.ui.dialogs;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.gmt.modisco.infra.common.core.internal.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.infra.common.core.internal.utils.ImageUtils;
import org.eclipse.gmt.modisco.infra.common.ui.internal.util.ImageProvider;
import org.eclipse.gmt.modisco.infra.query.ModelQuery;
import org.eclipse.gmt.modisco.infra.query.ModelQuerySet;
import org.eclipse.gmt.modisco.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.gmt.modisco.infra.query.ui.Activator;
import org.eclipse.gmt.modisco.infra.query.ui.Messages;
import org.eclipse.gmt.modisco.infra.query.ui.wizards.CreateQueryWizard;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.FilteredTree;
import org.eclipse.ui.dialogs.PatternFilter;

/** A dialog to select a {@link ModelQuery} */
public class QuerySelectionDialog extends Dialog {

	private Collection<ModelQuerySet> fQuerySets;
	private final ViewerFilter viewerFilter;
	private FilteredTree filteredTree;
	private final boolean multiSelection;
	private List<ModelQuery> selectedQueries;

	/**
	 * @param parentShell
	 *            the parent shell
	 * @param querySets
	 *            the query sets from which queries will be selectable. If
	 *            <code>null</code>, all query sets will be displayed.
	 * @param viewerFilter
	 *            an optional filter for selectable queries (can be
	 *            <code>null</code> if no filtering is required)
	 */
	public QuerySelectionDialog(final Shell parentShell, final boolean multiSelection,
			final Collection<ModelQuerySet> querySets, final ViewerFilter viewerFilter) {
		super(parentShell);
		this.multiSelection = multiSelection;
		if (querySets == null) {
			this.fQuerySets = ModelQuerySetCatalog.getSingleton().getAllModelQuerySets();
		} else {
			this.fQuerySets = querySets;
		}
		this.viewerFilter = viewerFilter;
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);
		// composite.setLayout(new GridLayout());
		this.filteredTree = createFilteredTree(composite, this.fQuerySets, this.viewerFilter,
				this.multiSelection);

		createContextMenu(this.filteredTree.getViewer());

		// double-click or enter to choose the selected query
		this.filteredTree.getViewer().addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				QuerySelectionDialog.this.okPressed();
			}
		});
		return composite;
	}

	private class OpenCreateQueryWizardAction extends Action {
		private final ModelQuerySet modelQuerySet;

		public OpenCreateQueryWizardAction(final ModelQuerySet modelQuerySet) {
			this.modelQuerySet = modelQuerySet;
			setText(Messages.QuerySelectionDialog_createQuery);
		}

		@SuppressWarnings("synthetic-access")
		@Override
		public void run() {
			final CreateQueryWizard createQueryWizard = new CreateQueryWizard(this.modelQuerySet,
					null);
			WizardDialog wizardDialog = new WizardDialog(null, createQueryWizard);
			if (wizardDialog.open() == Window.OK) {
				Job refreshJob = new Job(Messages.QuerySelectionDialog_refreshQuerySetsJob) {
					@Override
					protected IStatus run(final IProgressMonitor monitor) {
						try {
							// rebuild the query catalog
							ResourcesPlugin.getWorkspace().build(
									IncrementalProjectBuilder.FULL_BUILD, monitor);
						} catch (CoreException e) {
							MoDiscoLogger.logError(e, Activator.getDefault());
						}
						// the query sets have been re-created => find by name
						Collection<ModelQuerySet> oldQuerySets = new ArrayList<ModelQuerySet>(
								QuerySelectionDialog.this.fQuerySets);
						final Collection<ModelQuerySet> newQuerySets = new ArrayList<ModelQuerySet>();
						for (ModelQuerySet oldQuerySet : oldQuerySets) {
							ModelQuerySet querySet = ModelQuerySetCatalog.getSingleton()
									.getModelQuerySet(oldQuerySet.getName());
							if (querySet != null) {
								newQuerySets.add(querySet);
							}
						}
						QuerySelectionDialog.this.fQuerySets = newQuerySets;
						Display.getDefault().asyncExec(new Runnable() {
							public void run() {
								changeInput(QuerySelectionDialog.this.filteredTree, newQuerySets);
								// select the created query
								ModelQuery createdModelQuery = createQueryWizard
										.getCreatedModelQuery();
								ModelQuerySet querySet = ModelQuerySetCatalog.getSingleton()
										.getModelQuerySet(
												createdModelQuery.getModelQuerySet().getName());
								ModelQuery query = querySet.getQuery(createdModelQuery.getName());
								QuerySelectionDialog.this.filteredTree.getViewer().setSelection(
										new StructuredSelection(query), true);
							}
						});
						return Status.OK_STATUS;
					}
				};
				refreshJob.setPriority(Job.DECORATE);
				refreshJob.schedule();
			}
		}
	}

	private void createContextMenu(final TreeViewer viewer) {
		final MenuManager contextMenu = new MenuManager("#PopUp"); //$NON-NLS-1$
		contextMenu.setRemoveAllWhenShown(true);
		contextMenu.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(final IMenuManager manager) {
				ISelection selection = viewer.getSelection();
				if (selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					if (structuredSelection.size() == 1) {
						Object firstElement = structuredSelection.getFirstElement();
						if (firstElement instanceof ModelQuerySet) {
							final ModelQuerySet modelQuerySet = (ModelQuerySet) firstElement;
							manager.add(new OpenCreateQueryWizardAction(modelQuerySet));
							return;
						}
					}
				}
				manager.add(new OpenCreateQueryWizardAction(null));
			}
		});
		final Menu menu = contextMenu.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
	}

	@Override
	protected void configureShell(final Shell shell) {
		super.configureShell(shell);
		shell.setText(Messages.QuerySelectionDialog_selectQuery);
		shell.setImage(ImageProvider.getInstance().getModiscoLogo());
	}

	@Override
	protected boolean isResizable() {
		return true;
	}

	@Override
	protected IDialogSettings getDialogBoundsSettings() {
		final String sectionName = this.getClass().getName() + ".settings"; //$NON-NLS-1$
		IDialogSettings settings = Activator.getDefault().getDialogSettings();
		IDialogSettings section = settings.getSection(sectionName);
		if (section == null) {
			section = settings.addNewSection(sectionName);
		}
		return section;
	}

	@Override
	protected void okPressed() {
		this.selectedQueries = new ArrayList<ModelQuery>();
		ISelection selection = this.filteredTree.getViewer().getSelection();
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof ModelQuery) {
					this.selectedQueries.add((ModelQuery) selectedElement);
				}
			}
		}
		super.okPressed();
	}

	public List<ModelQuery> getSelectedQueries() {
		return this.selectedQueries;
	}

	/**
	 * Create a tree with a filter text field to select queries.
	 * 
	 * @param parentComposite
	 *            the parent composite
	 * @param availableQuerySets
	 *            the query sets from which queries are selected.
	 *            <code>null</code> means all.
	 * @param viewerFilter
	 *            a filter on the displayed queries. Can be <code>null</code>.
	 * @return a filtered tree control
	 */
	public static FilteredTree createFilteredTree(final Composite parentComposite,
			final Collection<ModelQuerySet> availableQuerySets, final ViewerFilter viewerFilter,
			final boolean multiSelection) {

		final PatternFilter filter = new PatternFilter() {
			@Override
			protected boolean isLeafMatch(final Viewer viewer, final Object element) {
				String labelText = ((ILabelProvider) ((StructuredViewer) viewer).getLabelProvider())
						.getText(element);

				if (labelText == null) {
					return false;
				}
				return wordMatches(labelText) || parentMatches(viewer, element);
			}

			private boolean parentMatches(final Viewer viewer, final Object element) {
				ITreeContentProvider contentProvider = (ITreeContentProvider) ((StructuredViewer) viewer)
						.getContentProvider();
				Object parent = contentProvider.getParent(element);
				if (parent == null) {
					return false;
				}
				return isLeafMatch(viewer, parent);
			}
		};
		final int selectionStyle;
		if (multiSelection) {
			selectionStyle = SWT.MULTI;
		} else {
			selectionStyle = SWT.SINGLE;
		}
		final FilteredTree tree = new FilteredTree(parentComposite, selectionStyle | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.BORDER, filter, true);

		final TreeViewer treeViewer = tree.getViewer();
		treeViewer.setContentProvider(new ITreeContentProvider() {

			public Object[] getElements(final Object inputElement) {
				@SuppressWarnings("unchecked")
				final java.util.List<ModelQuerySet> querySets2 = (java.util.List<ModelQuerySet>) inputElement;
				return querySets2.toArray();
			}

			public Object[] getChildren(final Object parentElement) {
				if (parentElement instanceof ModelQuerySet) {
					final ModelQuerySet querySet = (ModelQuerySet) parentElement;
					return querySet.getQueries().toArray();
				}
				return null;
			}

			public Object getParent(final Object element) {
				if (element instanceof ModelQuery) {
					final ModelQuery query = (ModelQuery) element;
					return query.getModelQuerySet();
				}
				return null;
			}

			public boolean hasChildren(final Object element) {
				if (element instanceof ModelQuerySet) {
					final ModelQuerySet querySet = (ModelQuerySet) element;
					return !querySet.getQueries().isEmpty();
				}
				return false;
			}

			public void inputChanged(final Viewer viewer, final Object oldInput,
					final Object newInput) {
				// nothing
			}

			public void dispose() {
				// nothing
			}
		});

		treeViewer.setLabelProvider(new LabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof ModelQuerySet) {
					final ModelQuerySet querySet = (ModelQuerySet) element;
					return querySet.getName();
				} else if (element instanceof ModelQuery) {
					final ModelQuery query = (ModelQuery) element;
					return query.getName();
				}
				return super.getText(element);
			}

			@Override
			public Image getImage(final Object element) {
				return ImageUtils.getImage(element);
			}
		});

		if (viewerFilter != null) {
			treeViewer.setFilters(new ViewerFilter[] { viewerFilter });
		}

		changeInput(tree, availableQuerySets);
		return tree;
	}

	/**
	 * Changes the input of the given FilteredTree, created by
	 * {@link #createFilteredTree(Composite, Collection, ViewerFilter, boolean)}
	 * 
	 * @param tree
	 *            the filtered tree control
	 * @param availableQuerySets
	 *            the query sets to set as input. <code>null</code> means all.
	 */
	public static void changeInput(final FilteredTree tree,
			final Collection<ModelQuerySet> availableQuerySets) {
		final Collection<ModelQuerySet> querySets;
		if (availableQuerySets == null) {
			querySets = ModelQuerySetCatalog.getSingleton().getAllModelQuerySets();
		} else {
			querySets = availableQuerySets;
		}

		tree.getViewer().setInput(querySets);
	}
}
