/*
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.java.discoverer.tests;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.junit.Assert;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.infra.common.core.internal.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.infra.common.core.internal.utils.ModelUtils;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameter;
import org.eclipse.gmt.modisco.java.actions.DefaultDiscoverer;
import org.eclipse.gmt.modisco.java.actions.DiscoverJavaModelFromJavaProject;
import org.eclipse.jdt.core.IJavaProject;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

/**
 * @author Gabriel Barbier
 *
 */
public class SimpleBlackBoxDiscovery {

	private final String rootSourcesPath = "/workspace/"; //$NON-NLS-1$
	private final String referencesFolderPath = "/references/"; //$NON-NLS-1$
	private final String javaModelExtension = ".javaxmi"; //$NON-NLS-1$
	/**
	 * @throws java.lang.Exception
	 */
	@Before
	public void setUp() throws Exception {
		// Nothing
	}

	/**
	 * @throws java.lang.Exception
	 */
	@After
	public void tearDown() throws Exception {
		// Nothing
	}

	@Test
	public void testJavaModelDiscoveryFromJavaProject() {
		final String projectName = "test001"; //$NON-NLS-1$
		JavaProjectFactory test = null;
		try {
			test = new JavaProjectFactory(projectName);
			Assert.assertNotNull(test);
			
			IJavaProject javaProject = test.getJavaProject();
			Assert.assertNotNull(javaProject);
			
			String sourceFolderPath = this.rootSourcesPath + projectName + "/src"; //$NON-NLS-1$
			URL src = Activator.getDefault().getBundle().getEntry(sourceFolderPath);
			Assert.assertNotNull(src);
			test.populateSourceFolder(sourceFolderPath);
			
			DiscoverJavaModelFromJavaProject javaDiscoverer = new DiscoverJavaModelFromJavaProject();
			Assert.assertNotNull(javaDiscoverer);
			Map<DiscoveryParameter, Object> parameters = new HashMap<DiscoveryParameter, Object>();
			parameters.put(DefaultDiscoverer.PARAMETER_SILENT_MODE, true);
			javaDiscoverer.discoverElement(javaProject, parameters);
			Resource output = (Resource) parameters.get(DefaultDiscoverer.PARAMETER_TARGET_RESOURCE);
			Assert.assertNotNull(output);

			
		} catch (Exception e) {
			Assert.fail("Transformation of java project into java model shall not produce errors !"); //$NON-NLS-1$
		} finally {
			if (test != null) {
				try {
					test.dispose();
				} catch (CoreException e) {
					MoDiscoLogger.logError(e, Activator.getDefault());
				}
			}
		}

	}
	
	@SuppressWarnings("nls")
	@Test
	@Ignore("absolute model paths prevent the test from working on the build server")
	public void testJavaModelFromJavaProjectWithReferenceModel() {
		final String projectName = "test001";
		JavaProjectFactory test = null;
		try {
			test = new JavaProjectFactory(projectName);
			Assert.assertNotNull(test);
			
			IJavaProject javaProject = test.getJavaProject();
			Assert.assertNotNull(javaProject);
			
			String sourceFolderPath = this.rootSourcesPath + projectName + "/src";
			URL src = Activator.getDefault().getBundle().getEntry(sourceFolderPath);
			Assert.assertNotNull(src);
			test.populateSourceFolder(sourceFolderPath);
			
			DiscoverJavaModelFromJavaProject javaDiscoverer = new DiscoverJavaModelFromJavaProject();
			Assert.assertNotNull(javaDiscoverer);
			Map<DiscoveryParameter, Object> parameters = new HashMap<DiscoveryParameter, Object>();
			parameters.put(DefaultDiscoverer.PARAMETER_SILENT_MODE, true);
			javaDiscoverer.discoverElement(javaProject, parameters);
			Resource output = (Resource) parameters.get(DefaultDiscoverer.PARAMETER_TARGET_RESOURCE);
			Assert.assertNotNull(output);

			/*
			 * Because everything looks great, well we have to test model content.
			 * In details we will compare current resource with a reference (checked manually)
			 */
			String referencePath = "/" + Activator.PLUGIN_ID + this.referencesFolderPath + projectName + this.javaModelExtension;
			URI referenceUri = URI.createPlatformPluginURI(referencePath, true);
			Assert.assertNotNull(referenceUri);
			Resource referenceModel = ModelUtils.load(referenceUri);
			Assert.assertNotNull(referenceModel);
			boolean result = ModelUtils.compareModels(output, referenceModel, true);
			Assert.assertTrue("Comparison of Java models has failed !", result);
		} catch (Exception e) {
			e.printStackTrace();
			Assert.fail("Transformation of java project into java model shall not produce errors !");
		} finally {
			if (test != null) {
				try {
					test.dispose();
				} catch (CoreException e) {
					MoDiscoLogger.logError(e, Activator.getDefault());
				}
			}
		}

	}
}
