/*******************************************************************************
 * Copyright (c) 2009, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.browser.actions;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import org.eclipse.core.runtime.IPath;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.editors.EcoreBrowser;
import org.eclipse.gmt.modisco.infra.browser.util.ImageProvider;
import org.eclipse.gmt.modisco.infra.common.core.internal.utils.ImageUtils;
import org.eclipse.gmt.modisco.infra.facet.Facet;
import org.eclipse.gmt.modisco.infra.facet.FacetPackage;
import org.eclipse.gmt.modisco.infra.facet.FacetSet;
import org.eclipse.gmt.modisco.infra.facet.core.FacetSetCatalog;
import org.eclipse.gmt.modisco.infra.facet.ui.views.FacetSetContentProvider;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.DialogSettings;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.CheckedTreeSelectionDialog;

public class LoadFacetsAction extends Action {

	private final IBrowserActionBarContributor browserActionBarContributor;

	public LoadFacetsAction(final IBrowserActionBarContributor browserActionBarContributor) {
		super(Messages.BrowserActionBarContributor_loadFacets, IAction.AS_PUSH_BUTTON);
		this.browserActionBarContributor = browserActionBarContributor;
		setToolTipText(Messages.BrowserActionBarContributor_loadFacets);
		setImageDescriptor(ImageProvider.getInstance().getFacetIconDescriptor());
	}

	private class LoadFacetsDialog extends CheckedTreeSelectionDialog {
		private static final String LOAD_FACETS_DIALOG = "LoadFacetsDialog"; //$NON-NLS-1$
		private static final String EXPANDED_ELEMENTS = "expandedElements"; //$NON-NLS-1$

		public LoadFacetsDialog(final Shell parent, final ILabelProvider labelProvider,
				final ITreeContentProvider contentProvider, final String metamodelURI) {
			super(parent, labelProvider, contentProvider);
			// show only Facets corresponding to the current metamodel
			addFilter(new ViewerFilter() {
				@Override
				public boolean select(final Viewer viewer, final Object parentElement,
						final Object element) {
					if (element instanceof FacetSet) {
						FacetSet facetSet = (FacetSet) element;
						if (facetSet.getExtendedPackage() == null) {
							return true;
						}
						String nsURI = facetSet.getExtendedPackage().getNsURI();
						return nsURI != null && nsURI.equals(metamodelURI);
					}
					return true;
				}
			});
		}

		@Override
		public int open() {
			restoreExpandedElements();
			return super.open();
		}

		@Override
		public boolean close() {
			saveExpandedElements();
			return super.close();
		}

		public void saveExpandedElements() {
			try {
				String filename = getSettingsFilename();
				DialogSettings settings = new DialogSettings(
						LoadFacetsAction.LoadFacetsDialog.LOAD_FACETS_DIALOG);

				List<String> expandedFacetSets = new ArrayList<String>();

				Object[] expandedElements = getTreeViewer().getExpandedElements();
				for (Object expandedElement : expandedElements) {
					if (expandedElement instanceof FacetSet) {
						FacetSet facetSet = (FacetSet) expandedElement;
						expandedFacetSets.add(facetSet.getName());
					}
				}
				String[] array = expandedFacetSets.toArray(new String[expandedFacetSets.size()]);
				settings.put(LoadFacetsAction.LoadFacetsDialog.EXPANDED_ELEMENTS, array);
				settings.save(filename);
			} catch (Exception e) {
				MoDiscoBrowserPlugin.logException(e);
			}
		}

		private String getSettingsFilename() {
			IPath stateLocation = MoDiscoBrowserPlugin.getPlugin().getStateLocation();
			String filename = stateLocation.append(
					LoadFacetsAction.LoadFacetsDialog.LOAD_FACETS_DIALOG).toOSString();
			return filename;
		}

		public void restoreExpandedElements() {
			try {
				String filename = getSettingsFilename();
				DialogSettings settings = new DialogSettings(
						LoadFacetsAction.LoadFacetsDialog.LOAD_FACETS_DIALOG);
				if (new File(filename).isFile()) {
					settings.load(filename);
					String[] array = settings
							.getArray(LoadFacetsAction.LoadFacetsDialog.EXPANDED_ELEMENTS);

					final Collection<FacetSet> facetSets = FacetSetCatalog.getSingleton()
							.getAllFacetSets();
					final List<FacetSet> expandedFacetSets = new ArrayList<FacetSet>();
					for (String expandedFacetSetName : array) {
						for (FacetSet facetSet : facetSets) {
							if (facetSet.getName() != null
									&& facetSet.getName().equals(expandedFacetSetName)) {
								expandedFacetSets.add(facetSet);
							}
						}
					}
					setExpandedElements(expandedFacetSets.toArray());
				}
			} catch (Exception e) {
				MoDiscoBrowserPlugin.logException(e);
			}

		}
	}

	@Override
	public void run() {
		final EcoreBrowser browser = this.browserActionBarContributor.getBrowser();
		if (browser != null) {

			final ITreeContentProvider treeContentProvider = new FacetSetContentProvider(
					FacetPackage.eINSTANCE.getFacet());

			final LoadFacetsDialog dialog = new LoadFacetsDialog(null, this.labelProvider,
					treeContentProvider, browser.getMetamodelURI());
			dialog.setContainerMode(true);
			dialog.setHelpAvailable(false);
			dialog.setTitle(Messages.BrowserActionBarContributor_dialogTitle_LoadFacets);
			dialog.setMessage(Messages.BrowserActionBarContributor_chooseFacetToLoad);
			final Collection<FacetSet> facetSets = FacetSetCatalog.getSingleton().getAllFacetSets();
			dialog.setInput(facetSets.toArray());

			dialog.setInitialElementSelections(new ArrayList<Facet>(browser
					.getBrowserConfiguration().getLoadedFacets()));

			if (dialog.open() == Window.OK) {
				final Object[] result = dialog.getResult();
				browser.getBrowserConfiguration().clearFacets();
				final HashSet<Facet> facetsToLoad = new HashSet<Facet>();
				for (final Object resultElement : result) {
					if (resultElement instanceof Facet) {
						final Facet facet = (Facet) resultElement;
						facetsToLoad.add(facet);
					} else if (!(resultElement instanceof FacetSet)) {
						throw new RuntimeException("Unexpected element type"); //$NON-NLS-1$
					}
				}

				BusyIndicator.showWhile(null, new Runnable() {
					public void run() {
						for (Facet facetToLoad : facetsToLoad) {
							browser.getBrowserConfiguration().loadFacet(facetToLoad);
						}
						if (facetsToLoad.size() > 0) {
							// since customizations can apply to facets, reload
							// customizations
							browser.getBrowserConfiguration().getCustomizationEngine()
									.loadCustomizations();
						}
						// browser.reloadModel();
						// for modified facets 
						browser.getBrowserConfiguration().getInstancesForMetaclasses().buildDerivationTree();
						browser.getBrowserConfiguration().touch();
						browser.refreshDelayed(true);
					}
				});
			}
		}
	}

	private final LabelProvider labelProvider = new LabelProvider() {
		@Override
		public String getText(final Object element) {
			String label;
			if (element instanceof FacetSet) {
				final FacetSet facetSet = (FacetSet) element;
				label = facetSet.getName();
			} else if (element instanceof Facet) {
				final Facet facet = (Facet) element;
				label = facet.getName();
			} else if (element == null) {
				label = "null"; //$NON-NLS-1$
			} else {
				label = element.getClass().getName();
			}
			return label;
		}

		@Override
		public Image getImage(final Object element) {
			return ImageUtils.getImage(element);
		}
	};
}
