/*******************************************************************************
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.browser.editors;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ContentHandler;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.URIHandler;
import org.eclipse.emf.ecore.resource.impl.ExtensibleURIConverterImpl;
import org.eclipse.emf.ecore.resource.impl.URIHandlerImpl;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.core.AttributeItem;
import org.eclipse.gmt.modisco.infra.browser.core.ITreeElement;
import org.eclipse.gmt.modisco.infra.browser.core.LinkItem;
import org.eclipse.gmt.modisco.infra.browser.core.ModelElementItem;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

public final class BrowserUtils {

	private BrowserUtils() {
		// make this class non-instantiable
	}

	/** Create a URI converter that avoids opening HTTP connections. */
	public static URIConverter createURIConverter() {
		final List<URIHandler> defaultHandlers = URIHandler.DEFAULT_HANDLERS;
		final List<URIHandler> uriHandlers = new ArrayList<URIHandler>();
		for (URIHandler uriHandler : defaultHandlers) {
			if (uriHandler.getClass() != URIHandlerImpl.class) {
				uriHandlers.add(uriHandler);
			}
		}
		uriHandlers.add(new URIHandlerImpl() {
			@Override
			public Map<String, ?> getAttributes(final URI uri, final Map<?, ?> options) {
				// overloaded to avoid opening an HTTP connection
				return new HashMap<String, Object>();
			}
		});
		return new ExtensibleURIConverterImpl(uriHandlers, ContentHandler.Registry.INSTANCE
				.contentHandlers());
	}

	public static int askUserForDepthToLoadReferencedResources(final int initialValue,
			final Shell shell) {
		IInputValidator validator = new IInputValidator() {
			public String isValid(final String inputText) {
				if ("*".equals(inputText)) { //$NON-NLS-1$
					return null;
				}
				try {
					Integer.parseInt(inputText);
				} catch (NumberFormatException e) {
					return NLS.bind(Messages.EcoreBrowser_notAPositiveInteger, inputText);
				}
				return null;
			}
		};
		final String initialValueStr;
		if (initialValue == -1) {
			initialValueStr = "*"; //$NON-NLS-1$
		} else {
			initialValueStr = Integer.toString(initialValue);
		}

		final InputDialog inputDialog = new InputDialog(shell,
				Messages.EcoreBrowser_selectDepthReferencedResources,
				Messages.EcoreBrowser_modelContainsProxies
						+ Messages.EcoreBrowser_chooseLoadProxiesDepth, initialValueStr, validator);

		if (inputDialog.open() == Window.OK) {
			String value = inputDialog.getValue();
			if (value == null) {
				return 0;
			}
			if ("*".equals(value)) { //$NON-NLS-1$
				return -1;
			}
			try {
				return Integer.parseInt(value);
			} catch (NumberFormatException e) {
				MoDiscoBrowserPlugin.logException(e);
				return 0;
			}
		}
		return 0;
	}

	/** Copy the text of the selected element to clipboard */
	public static void handleCopyToClipboard(final ISelection selection) {
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			StringBuilder builder = new StringBuilder();
			Iterator<?> iterator = treeSelection.iterator();
			while (iterator.hasNext()) {
				if (builder.length() > 0) {
					builder.append("\n"); //$NON-NLS-1$
				}
				Object selectedElement = iterator.next();
				if (selectedElement instanceof ModelElementItem) {
					ModelElementItem modelElementItem = (ModelElementItem) selectedElement;
					builder.append(modelElementItem.getName());
				} else if (selectedElement instanceof AttributeItem) {
					AttributeItem attributeItem = (AttributeItem) selectedElement;
					String valueText = attributeItem.getValueText();
					final String strEquals = " = "; //$NON-NLS-1$
					if (valueText.startsWith(strEquals)) {
						builder.append(valueText.substring(strEquals.length()));
					} else {
						builder.append(valueText);
					}
				} else if (selectedElement instanceof LinkItem) {
					LinkItem linkItem = (LinkItem) selectedElement;
					builder.append(linkItem.getReference().getName());
				} else if (selectedElement instanceof ITreeElement) {
					ITreeElement treeElement = (ITreeElement) selectedElement;
					builder.append(treeElement.getText());
				} else {
					builder.append(selectedElement.toString());
				}
			}
			Clipboard clipboard = new Clipboard(Display.getDefault());
			clipboard.setContents(new Object[] { builder.toString() },
					new Transfer[] { TextTransfer.getInstance() });
			clipboard.dispose();
		}
	}

	/**
	 * Replace proxy objects by their underlying Ecore objects in the given
	 * selection
	 * 
	 * @return the unwrapped version of the selection
	 */
	public static ISelection unwrapSelection(final ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			final List<Object> elements = new ArrayList<Object>();
			final Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				final Object element = iterator.next();
				if (element instanceof ModelElementItem) {
					final ModelElementItem modelElementItem = (ModelElementItem) element;
					elements.add(modelElementItem.getEObject());
				} else if (element instanceof LinkItem) {
					final LinkItem linkItem = (LinkItem) element;
					elements.add(linkItem.getReference());
				} else if (element instanceof AttributeItem) {
					final AttributeItem attributeItem = (AttributeItem) element;
					elements.add(attributeItem.getAttribute());
				} else {
					elements.add(element);
				}
			}
			return new StructuredSelection(elements);
		}
		return selection;
	}

}
