/*******************************************************************************
 * Copyright (c) 2008, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    Nicolas Payneau (Mia-Software) - Enable editability
 *******************************************************************************/
package org.eclipse.gmt.modisco.infra.browser.editors;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EventObject;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.command.BasicCommandStack;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.ui.MarkerHelper;
import org.eclipse.emf.common.ui.viewer.IViewerProvider;
import org.eclipse.emf.common.util.BasicDiagnostic;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EValidator;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.ReflectiveItemProviderAdapterFactory;
import org.eclipse.emf.edit.provider.resource.ResourceItemProviderAdapterFactory;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.emf.edit.ui.provider.PropertyDescriptor;
import org.eclipse.emf.edit.ui.provider.PropertySource;
import org.eclipse.emf.edit.ui.util.EditUIMarkerHelper;
import org.eclipse.emf.edit.ui.util.EditUIUtil;
import org.eclipse.emf.edit.ui.view.ExtendedPropertySheetPage;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.core.InstancesForMetaclass;
import org.eclipse.gmt.modisco.infra.browser.core.InstancesForMetaclasses;
import org.eclipse.gmt.modisco.infra.browser.core.InstancesForMetaclasses.MetaclassesChangeListener;
import org.eclipse.gmt.modisco.infra.browser.core.MetaclassList;
import org.eclipse.gmt.modisco.infra.browser.core.ModelElementItemEx;
import org.eclipse.gmt.modisco.infra.browser.core.QueryItem;
import org.eclipse.gmt.modisco.infra.browser.core.SearchResults;
import org.eclipse.gmt.modisco.infra.browser.custom.MetamodelView;
import org.eclipse.gmt.modisco.infra.browser.custom.core.CustomizationsCatalog;
import org.eclipse.gmt.modisco.infra.browser.preferences.PreferenceConstants;
import org.eclipse.gmt.modisco.infra.browser.queries.SelectedQueriesManager;
import org.eclipse.gmt.modisco.infra.browser.queries.SelectedQuery;
import org.eclipse.gmt.modisco.infra.browser.uicore.ChangeListener;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.AppearanceConfiguration;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.CustomTreePainter;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.customization.CustomizationEngine;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.model.AttributeItem;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.model.BigListItem;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.model.ITreeElement;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.model.LinkItem;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.model.ModelElementItem;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.common.core.internal.adapters.instances.MetaclassInstances;
import org.eclipse.gmt.modisco.infra.common.core.internal.builder.AbstractMoDiscoCatalog.ModiscoCatalogChangeListener;
import org.eclipse.gmt.modisco.infra.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.infra.common.ui.internal.controls.FormStyleSashForm;
import org.eclipse.gmt.modisco.infra.common.ui.internal.editorInputs.IResourceEditorInput;
import org.eclipse.gmt.modisco.infra.common.ui.internal.util.DialogUtils;
import org.eclipse.gmt.modisco.infra.facet.Facet;
import org.eclipse.gmt.modisco.infra.facet.FacetSet;
import org.eclipse.gmt.modisco.infra.facet.core.FacetSetCatalog;
import org.eclipse.gmt.modisco.infra.facet.core.adapters.instances.MetaclassInstancesAdapterFactoryWithFacet;
import org.eclipse.gmt.modisco.infra.facet.core.adapters.instances.MetaclassInstancesAdapterWithFacet;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.INavigationLocation;
import org.eclipse.ui.INavigationLocationProvider;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.ide.IGotoMarker;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.IPropertySource;
import org.eclipse.ui.views.properties.PropertySheet;
import org.eclipse.ui.views.properties.PropertySheetPage;

/**
 * MoDisco model editor: a two pane editor. The left pane shows the list of
 * metaclasses and the right pane shows the list of instances of the metaclass
 * selected in the left pane.
 * <p>
 * The following pop-up menus can be contributed to this part:
 * <ul>
 * <li>{@link EcoreBrowser#MENU_ID}
 * <li>{@link MetaclassViewer#MENU_ID}
 * </ul>
 */
public class EcoreBrowser extends EditorPart implements ISelectionProvider, IMenuListener,
		IViewerProvider, INavigationLocationProvider, MetaclassesChangeListener,
		IEditingDomainProvider, IGotoMarker {

	public static final String MENU_ID = EcoreBrowser.EDITOR_ID + ".menu"; //$NON-NLS-1$

	private static final int MAIN_FORM_MARGIN_HEIGHT = 6;
	private static final int MAIN_FORM_MARGIN_WIDTH = 4;
	private static final int RIGHT_PANE_SASH_WEIGHT = 65;
	private static final int LEFT_PANE_SASH_WEIGHT = 35;
	public static final String EDITOR_ID = "org.eclipse.gmt.modisco.infra.browser.editorID"; //$NON-NLS-1$
	private static final String PROPERTY_VIEW_ID = "org.eclipse.ui.views.PropertySheet"; //$NON-NLS-1$

	private ResourceSet fResourceSet;

	private Resource model;

	/** An adapter factory that uses the registry */
	private ComposedAdapterFactory adapterFactoryWithRegistry;

	/** The property sheet page. */
	private PropertySheetPage propertySheetPage;

	/** The tree part of the model editor. */
	private TreeViewer treeViewer;

	/** The tree component (instances pane) */
	private Tree fTree;

	/** The list of metaclasses that appears in the left pane of the editor */
	private MetaclassViewer metaclassViewer;

	/** This listens to the tree viewer's selection. */
	private ISelectionChangedListener selectionChangedListener;

	/**
	 * This keeps track of all the
	 * {@link org.eclipse.jface.viewers.ISelectionChangedListener}s that are
	 * listening to this editor.
	 */
	private final Collection<ISelectionChangedListener> selectionChangedListeners = new ArrayList<ISelectionChangedListener>();

	/** This keeps track of the selection of the editor as a whole. */
	private ISelection editorSelection = StructuredSelection.EMPTY;

	/**
	 * Resources that have been removed since last activation.
	 */
	private final Collection<Resource> removedResources = new ArrayList<Resource>();

	/**
	 * Resources that have been changed since last activation.
	 */
	private final Collection<Resource> changedResources = new ArrayList<Resource>();

	private final Collection<Resource> savedResources = new ArrayList<Resource>();

	/** This listens for workspace changes. */
	private final IResourceChangeListener resourceChangeListener;

	/**
	 * Controls whether the problem indication should be updated.
	 */
	private boolean updateProblemIndication = true;

	/**
	 * Map to store the diagnostic associated with a resource.
	 */
	private final Map<Resource, Diagnostic> resourceToDiagnosticMap = new LinkedHashMap<Resource, Diagnostic>();

	/**
	 * The MarkerHelper is responsible for creating workspace resource markers
	 * presented in Eclipse's Problems View.
	 */
	private final MarkerHelper markerHelper = new EditUIMarkerHelper();

	/**
	 * This listens for when the outline becomes active
	 */
	private final IPartListener partListener = new IPartListener() {
		public void partActivated(final IWorkbenchPart p) {
			if (p instanceof PropertySheet) {
				if (((PropertySheet) p).getCurrentPage() == EcoreBrowser.this.propertySheetPage) {
					getActionBarContributor().setActiveEditor(EcoreBrowser.this);
					handleActivate();
				}
			} else if (p == EcoreBrowser.this) {
				handleActivate();
			}
		}

		public void partBroughtToTop(final IWorkbenchPart p) {
			// Ignore.
		}

		public void partClosed(final IWorkbenchPart p) {
			// Ignore.
		}

		public void partDeactivated(final IWorkbenchPart p) {
			// Ignore.
		}

		public void partOpened(final IWorkbenchPart p) {
			// Ignore.
		}
	};

	/**
	 * This keeps track of the editing domain that is used to track all changes
	 * to the model.
	 */
	private AdapterFactoryEditingDomain editingDomain;
	/**
	 * The configuration of this editor, which is passed to the adapters, so
	 * that they can use it to change their behavior accordingly
	 */
	private final BrowserConfiguration browserConfiguration;

	/** The composite in which the editor is displayed */
	private Composite parentComposite;

	/** The search box */
	private Text searchBox;

	/** The search job, to allow cancellation in case of a big model */
	private Job searchJob = null;

	/**
	 * The input of the tree viewer, saved before filtering by search, and
	 * restored afterwards
	 */
	private Object savedInput;

	/** A listener on changes to fonts settings */
	private IPropertyChangeListener fontChangeListener;

	/** Listens to changes in the loaded customizations, and updates the viewers */
	private ChangeListener fCustomizationEngineChangeListener = null;

	/** Listens to changes in the Facet catalog, and updates the viewers */
	private ModiscoCatalogChangeListener fFacetsChangeListener = null;

	/** Allows optimized refreshes of the viewer */
	private Job refreshJob = null;

	/**
	 * Whether the resource being viewed comes from the package registry, in
	 * which case it must not be unloaded
	 */
	private boolean readOnly;
	private FormStyleSashForm sashForm;
	private CustomTreePainter customTreePainter;

	private FormToolkit formToolkit;
	private Form form;
	private LeftPane leftPane;
	private TreeToolBar instancesToolBar;

	private BrowserSettingsPersistence settingsPersistence;

	/** Indicates the editor is being disposed */
	private boolean disposing = false;

	/** Create the Mia Ecore model editor. */
	public EcoreBrowser() {
		super();
		this.browserConfiguration = new BrowserConfiguration(this);

		// load user settings
		this.browserConfiguration.getAppearanceConfiguration().load();

		this.resourceChangeListener = new WorkspaceChangeListener(this);
	}

	/** Set the selection into the tree viewer */
	public void setSelectionToViewer(final Collection<?> selection) {
		final Collection<?> theSelection = selection;

		if (theSelection != null && !theSelection.isEmpty()) {
			Runnable runnable = new Runnable() {
				public void run() {
					if (EcoreBrowser.this.treeViewer != null) {
						EcoreBrowser.this.treeViewer.setSelection(new StructuredSelection(
								theSelection.toArray()), true);
					}
				}
			};
			getSite().getShell().getDisplay().asyncExec(runnable);
		}
	}

	/** @return the viewer as required by the {@link IViewerProvider} interface. */
	public Viewer getViewer() {
		return this.treeViewer;
	}

	public MetaclassViewer getMetaclassViewer() {
		return this.metaclassViewer;
	}

	/**
	 * Create a context menu for the viewer, add a listener, and register the
	 * menu for extension.
	 */
	protected void createContextMenuFor(final StructuredViewer viewer) {
		final MenuManager contextMenu = new MenuManager("#PopUp"); //$NON-NLS-1$
		contextMenu.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		contextMenu.setRemoveAllWhenShown(true);
		contextMenu.addMenuListener(this);
		final Menu menu = contextMenu.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(EcoreBrowser.MENU_ID, contextMenu,
				new UnwrappingSelectionProvider(viewer));
	}

	/** Load a resource based on the editor's input. */
	protected Resource createModel() {
		Resource result = null;
		// this.fResourceSet.setURIConverter(createURIConverter());
		this.fResourceSet.getURIConverter().getURIMap().putAll(EcorePlugin.computePlatformURIMap());
		this.readOnly = false;
		final URI resourceURI = EditUIUtil.getURI(getEditorInput());
		try {
			final IEditorInput editorInput = getEditorInput();
			if (editorInput instanceof FileEditorInput) {
				// Load the resource
				result = this.editingDomain.getResourceSet().getResource(resourceURI, true);
				this.fResourceSet = this.editingDomain.getResourceSet();
			} else if (editorInput instanceof IResourceEditorInput) {
				IResourceEditorInput resourceEditorInput = (IResourceEditorInput) editorInput;
				result = resourceEditorInput.getResource();
				// this.fResourceSet.getResources().add(result);
				ResourceSet resourceSet = result.getResourceSet();
				if (resourceSet != null) {
					this.fResourceSet = resourceSet;
				}
				this.readOnly = true;
				setPartName(resourceEditorInput.getName());
			} else if (editorInput instanceof IStorageEditorInput) {
				IStorageEditorInput storageEditorInput = (IStorageEditorInput) editorInput;
				IStorage storage = storageEditorInput.getStorage();
				InputStream inputStream = storage.getContents();
				Resource resource = this.fResourceSet.createResource(URI.createURI(resourceURI
						.toString()));
				resource.load(inputStream, Collections.emptyMap());
				this.readOnly = true;
				result = resource;
			} else {
				final String resolveURI = resourceURI.toString();
				final EPackage ePackage = this.fResourceSet.getPackageRegistry().getEPackage(
						resolveURI);
				if (ePackage != null) {
					result = ePackage.eResource();
					if (result != null) {
						this.fResourceSet.getResources().add(result);
						this.readOnly = true;
						setPartName(ePackage.getName());
					}
				} else {
					MessageDialog.openWarning(getSite().getShell(),
							Messages.EcoreBrowser_cannotOpenModelFromURI, NLS
									.bind(Messages.EcoreBrowser_cannotFindEPackageWithThatURI,
											resolveURI));
				}
			}
		} catch (final Exception e) {
			MoDiscoBrowserPlugin.logException(e);
			MessageDialog.openError(this.parentComposite.getShell(),
					Messages.EcoreBrowser_errorLoadingModel, e.toString());
			this.fResourceSet.getResource(resourceURI, false);
		}
		this.fResourceSet.setURIConverter(BrowserUtils.createURIConverter());
		return result;
	}

	/** Create the editor composite. */
	@Override
	public void createPartControl(final Composite parent) {

		this.formToolkit = new FormToolkit(parent.getDisplay());
		this.form = this.formToolkit.createForm(parent);

		this.parentComposite = parent;
		parent.setLayout(new FillLayout());

		initializeEditingDomain();

		final FillLayout fillLayout = new FillLayout();
		fillLayout.marginWidth = EcoreBrowser.MAIN_FORM_MARGIN_WIDTH;
		fillLayout.marginHeight = EcoreBrowser.MAIN_FORM_MARGIN_HEIGHT;
		this.form.getBody().setLayout(fillLayout);

		// use an adapter factory that looks in the registry
		this.browserConfiguration.getAppearanceConfiguration().setAdapterFactory(
				this.adapterFactoryWithRegistry);

		initResourceSet();

		this.settingsPersistence = new BrowserSettingsPersistence(getMetamodelURI(), this,
				this.browserConfiguration);

		this.browserConfiguration.setSelectedQueriesManager(new SelectedQueriesManager());
		this.sashForm = new FormStyleSashForm(this.form.getBody(), SWT.HORIZONTAL, this.formToolkit) {
			@Override
			protected void createContents() {
				createLeftPane(this);
				createTreePane(this);
			}
		};

		setupInitialTreeFont();

		this.sashForm.setWeights(new int[] { EcoreBrowser.LEFT_PANE_SASH_WEIGHT,
				EcoreBrowser.RIGHT_PANE_SASH_WEIGHT });

		// keep track of the selection, to be able to restore it when using the
		// "back" button
		addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(final SelectionChangedEvent event) {
				markNavigationLocation();
			}
		});

		/*
		 * Select the metaclass corresponding to the 'root' element (that is,
		 * the 'model' element). This in turn sets the input of the treeview.
		 */
		this.metaclassViewer.selectRootElement();

		createContextMenuFor(this.treeViewer);
		DragAdapter.create(this.treeViewer);

		loadDefaultFacetSets(this.model);

		IDialogSettings settingsForCurrentMetamodel = this.settingsPersistence
				.loadLastSettingsForCurrentMetamodel();

		List<Facet> lastFacets = this.settingsPersistence
				.getLastFacets(settingsForCurrentMetamodel);
		for (Facet facet : lastFacets) {
			this.browserConfiguration.getAppearanceConfiguration().loadFacet(facet);
		}

		List<MetamodelView> lastCustomizations = this.settingsPersistence
				.getLastCustomizations(settingsForCurrentMetamodel);
		if (lastCustomizations.size() > 0) {
			loadCustomizations(lastCustomizations);
		} else {
			loadDefaultCustomizations();
		}

		// note: must be done after adding or removing facets too
		this.browserConfiguration.getInstancesForMetaclasses().buildDerivationTree();

		// add a listener to reload when Facets are modified
		reloadOnFacetChange();
	}

	/**
	 * @return the URI of the resource corresponding to the EClass of the main
	 *         resource's root element
	 */
	public String getMetamodelURI() {
		try {
			EList<EObject> contents = this.model.getContents();
			if (contents.size() > 0) {
				EObject eObject = contents.get(0);
				EClass eClass = eObject.eClass();
				if (eClass != null) {
					return eClass.getEPackage().getNsURI();
				}
			}
		} catch (Exception e) {
			MoDiscoLogger.logError(e, MoDiscoBrowserPlugin.getPlugin());
		}
		return ""; //$NON-NLS-1$
	}

	protected void initResourceSet() {
		// create an empty resource set
		this.fResourceSet = createResourceSet();

		HashSet<Resource> browsedResources = new HashSet<Resource>();
		Resource mainResource;
		this.model = createModel();
		mainResource = this.model;
		browsedResources.add(mainResource);
		this.browserConfiguration.setResourceSet(this.fResourceSet);

		addReferencedResources(mainResource, browsedResources);
		this.browserConfiguration.setBrowsedResources(browsedResources);
		this.browserConfiguration.getAppearanceConfiguration().getFacetContext().setResources(
				browsedResources);

		InstancesForMetaclasses instancesForMetaclasses = new InstancesForMetaclasses(
				this.browserConfiguration, browsedResources);
		this.browserConfiguration.setInstancesForMetaclasses(instancesForMetaclasses);

		instancesForMetaclasses.addListener(this);
	}

	/**
	 * Create the resourceSet. This method can be overridden by subclasses.
	 * 
	 * @return The ResourceSet.
	 */
	protected ResourceSet createResourceSet() {
		return new ResourceSetImpl();
	}

	/**
	 * Load FacetSet defined as default for the root EPackage of root elements
	 * in the given resource
	 */
	private void loadDefaultFacetSets(final Resource resource) {
		for (EObject root : resource.getContents()) {
			EPackage rootEPackage = root.eClass().getEPackage();
			Set<FacetSet> defaultFacetSets = FacetSetCatalog.getSingleton().getDefaultFacetSets(
					rootEPackage);
			if (defaultFacetSets != null) {
				for (FacetSet facetSet : defaultFacetSets) {
					if (facetSet != null) {
						this.browserConfiguration.getAppearanceConfiguration().getFacetContext()
								.addFacetSet(facetSet);
					}
				}
			}
		}
	}

	/**
	 * If proxies are found, ask the user whether to load resources referenced
	 * by proxies found in the given resource, and up to which depth.
	 * 
	 * @param resource
	 *            the resource in which to look for proxies to resolve
	 * @param allResources
	 *            the set of resources to which to add resources referenced by
	 *            proxies
	 */
	private void addReferencedResources(final Resource resource, final Set<Resource> allResources) {
		IPreferenceStore preferenceStore = MoDiscoBrowserPlugin.getPlugin().getPreferenceStore();
		// only walk through the resource contents if necessary
		if (preferenceStore.getInt(PreferenceConstants.P_BROWSER_LOADING_DEPTH) != 0
				|| preferenceStore.getBoolean(PreferenceConstants.P_BROWSER_ASK_LOADING_DEPTH)) {
			addReferencedResourcesInternal(resource, allResources, 1);
		}
	}

	/**
	 * Adds resources referenced by proxies found in the given resources to the
	 * set <code>allResources</code>
	 * 
	 * @param resource
	 *            the resource in which to look for proxies to resolve
	 * @param allResources
	 *            the set of resources to which to add resources referenced by
	 *            proxies
	 * @param currentDepth
	 *            the current depth
	 * 
	 * @see EcoreBrowser#addReferencedResources(Resource, Set)
	 */
	private void addReferencedResourcesInternal(final Resource resource,
			final Set<Resource> allResources, final int currentDepth) {

		final Set<Resource> directReferencedResources = new HashSet<Resource>();

		for (EObject root : resource.getContents()) {
			Map<EObject, Collection<EStructuralFeature.Setting>> externalReferences = EcoreUtil.ExternalCrossReferencer
					.find(root);
			if (!externalReferences.isEmpty()) {
				int loadingDepth = this.browserConfiguration.getReferencedResourcesLoadingDepth();
				if (loadingDepth == BrowserConfiguration.NOT_YET_ASKED) {
					final int defaultLoadingDepth = MoDiscoBrowserPlugin.getPlugin()
							.getPreferenceStore().getInt(
									PreferenceConstants.P_BROWSER_LOADING_DEPTH);
					if (MoDiscoBrowserPlugin.getPlugin().getPreferenceStore().getBoolean(
							PreferenceConstants.P_BROWSER_ASK_LOADING_DEPTH)) {
						loadingDepth = BrowserUtils.askUserForDepthToLoadReferencedResources(
								defaultLoadingDepth, getSite().getShell());
					} else {
						loadingDepth = defaultLoadingDepth;
						if (loadingDepth < -1) {
							loadingDepth = 0;
						}
					}
					this.browserConfiguration.setReferencedResourcesLoadingDepth(loadingDepth);
				}
				if (loadingDepth != -1 && loadingDepth < currentDepth) {
					return;
				}
				for (EObject external : externalReferences.keySet()) {
					Resource externalResource;
					if (external.eIsProxy()) {
						externalResource = EcoreUtil.resolve(external, root).eResource();
					} else {
						externalResource = external.eResource();
					}
					if (externalResource != null) {
						directReferencedResources.add(externalResource);
					} else {
						MoDiscoLogger
								.logError(
										"Failed to load an external element: " + external.eClass().getName() //$NON-NLS-1$
												+ " " + EcoreUtil.getURI(external), //$NON-NLS-1$
										MoDiscoBrowserPlugin.getPlugin());
					}
				}
			}
		}

		int loadingDepth = this.browserConfiguration.getReferencedResourcesLoadingDepth();

		// recurse on sub-resources
		for (Resource directlyReferencedResource : directReferencedResources) {
			// avoid infinite recursion in the case of mutually referencing
			// resources
			if (!allResources.contains(directlyReferencedResource)) {
				allResources.add(directlyReferencedResource);
				if (loadingDepth == -1 || currentDepth + 1 <= loadingDepth) {
					addReferencedResourcesInternal(directlyReferencedResource, allResources,
							currentDepth + 1);
				}
			}
		}
	}

	/**
	 * Create the left pane, containing the metaclass viewer.
	 * 
	 * @param parent
	 *            the parent composite
	 */
	protected void createLeftPane(final Composite parent) {
		final Section section = this.formToolkit.createSection(parent,
				ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setText(Messages.EcoreBrowser_metaclasses);
		final Composite listPaneComposite = this.formToolkit.createComposite(section);
		section.setClient(listPaneComposite);
		this.leftPane = new LeftPane(section, listPaneComposite, this.browserConfiguration,
				this.formToolkit, getSite());
		final Composite toolBar = this.leftPane.getToolBarComposite();
		section.setTextClient(toolBar);
		listPaneComposite.setLayout(new FillLayout());
		this.metaclassViewer = this.leftPane.getMetaclassViewer();
		this.metaclassViewer
				.addSelectionChangedListener(new MetaclassListViewerSelectionChangedListener());
	}

	/**
	 * Create the tree and its viewer
	 * 
	 * @param parent
	 *            the parent composite of the tree
	 */
	protected void createTreePane(final Composite parent) {

		final Section section = this.formToolkit.createSection(parent,
				ExpandableComposite.TITLE_BAR | ExpandableComposite.EXPANDED);
		section.setText(Messages.EcoreBrowser_instances);
		final Composite treePane = this.formToolkit.createComposite(section, SWT.NONE);
		section.setClient(treePane);

		final GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		treePane.setLayout(layout);

		final Composite treeToolBar = createTreeToolBar(section);
		section.setTextClient(treeToolBar);

		createSearchPanel(treePane);

		this.fTree = this.formToolkit.createTree(treePane, SWT.MULTI | SWT.BORDER);
		final GridData treeGridData = new GridData();
		treeGridData.grabExcessHorizontalSpace = true;
		treeGridData.grabExcessVerticalSpace = true;
		treeGridData.horizontalAlignment = SWT.FILL;
		treeGridData.verticalAlignment = SWT.FILL;
		this.fTree.setLayoutData(treeGridData);

		this.fTree.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.keyCode == SWT.ESC) {
					// Escape on the tree: restore view (after search)
					EcoreBrowser.this.searchBox.setText(""); //$NON-NLS-1$
					search();
				} else if (e.keyCode == SWT.CR) {
					// Enter: browse selected item
					handleEnter();
				} else if (e.keyCode == 'c' && e.stateMask == SWT.CTRL) {
					// Ctrl+C : copy element text to clipboard
					BrowserUtils.handleCopyToClipboard(getEditorSelection());
				} else if (e.keyCode == SWT.DEL) {
					handleDel();
				}
			}
		});

		this.customTreePainter = new CustomTreePainter(this.fTree, this.browserConfiguration
				.getAppearanceConfiguration());
		// create a tooltip to show facet names
		FacetTooltip.create(this.fTree, this.customTreePainter);

		this.treeViewer = new TreeViewer(this.fTree);
		// speeds up all operations on a big model
		this.treeViewer.setUseHashlookup(true);

		hookViewerToSelection(this.treeViewer);

		this.treeViewer.setContentProvider(new BrowserContentProvider());
		this.treeViewer.setLabelProvider(new BrowserLabelProvider());

		registerShowPropertyViewOnDblClick(this.fTree);
	}

	/**
	 * The user pressed the "Enter" key : if selected element is a model
	 * element, then browse to it. If it is an attribute, then open properties
	 * view.
	 */
	protected void handleEnter() {
		final ISelection selection = getEditorSelection();
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			final Object selectedElement = treeSelection.getFirstElement();
			if (selectedElement instanceof ModelElementItem) {
				browse();
			} else if (selectedElement instanceof AttributeItem
					|| selectedElement instanceof LinkItem) {
				openPropertiesView();
			} else if (selectedElement instanceof QueryItem) {
				QueryItem queryItem = (QueryItem) selectedElement;
				SelectedQuery query = queryItem.getQuery();
				query.executeQuery();
			}
		}
	}

	/** The user pressed the "Delete" key */
	protected void handleDel() {
		final ISelection selection = getEditorSelection();
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			Iterator<?> iterator = treeSelection.iterator();
			while (iterator.hasNext()) {
				Object selectedElement = iterator.next();
				if (selectedElement instanceof QueryItem) {
					QueryItem queryItem = (QueryItem) selectedElement;
					this.browserConfiguration.getSelectedQueriesManager().remove(
							queryItem.getQuery());
				}
			}
		}
	}

	/**
	 * Set the initial font using saved user preferences, and add a listener to
	 * be notified of changes in the preferences
	 */
	private void setupInitialTreeFont() {
		setupPreferredFont();

		this.fontChangeListener = new IPropertyChangeListener() {
			public void propertyChange(final PropertyChangeEvent event) {
				if (event.getProperty().equals(JFaceResources.DIALOG_FONT)) {
					setupPreferredFont();
				}
			}
		};
		JFaceResources.getFontRegistry().addListener(this.fontChangeListener);
	}

	/** Set the user preferred font on the editor */
	private void setupPreferredFont() {
		FontDescriptor descriptor = FontDescriptor.createFrom(JFaceResources.getDialogFont());
		descriptor = descriptor.increaseHeight(this.browserConfiguration
				.getAppearanceConfiguration().getFontSizeDelta());
		final Font customFont = descriptor.createFont(Display.getDefault());
		descriptor = descriptor.setStyle(SWT.ITALIC);
		final Font customItalicFont = descriptor.createFont(Display.getDefault());
		this.browserConfiguration.getAppearanceConfiguration().setCustomFont(customFont);
		this.browserConfiguration.getAppearanceConfiguration()
				.setCustomItalicFont(customItalicFont);
		this.fTree.setFont(customFont);
		this.treeViewer.refresh();
		this.metaclassViewer.setFont(customFont);
		this.metaclassViewer.refresh();
	}

	private Composite createTreeToolBar(final Composite parent) {
		final Composite treeToolBarComposite = new Composite(parent, SWT.WRAP);
		this.instancesToolBar = TreeToolBar.create(treeToolBarComposite, this);
		return treeToolBarComposite;
	}

	/**
	 * Register a listener to show the property view or execute a query when the
	 * tree is double-clicked
	 */
	private void registerShowPropertyViewOnDblClick(final Tree tree) {
		tree.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDoubleClick(final MouseEvent event) {
				boolean handled = false;
				if (event.button == 1) {
					TreeItem[] selection = tree.getSelection();
					for (TreeItem treeItem : selection) {
						Object element = treeItem.getData();
						if (element instanceof QueryItem) {
							QueryItem queryItem = (QueryItem) element;
							SelectedQuery query = queryItem.getQuery();
							query.executeQuery();
							handled = true;
						}
					}
					if (!handled) {
						openPropertiesView();
					}
				}
			}
		});
	}

	/**
	 * Create the search panel, containing a label and search field.
	 * 
	 * @param parent
	 *            the parent composite, which must have a GridLayout
	 */
	protected void createSearchPanel(final Composite parent) {
		final Composite searchPanel = new Composite(parent, SWT.NONE);
		searchPanel.setLayout(new GridLayout());

		searchPanel.setLayout(new FillLayout());
		final GridData searchPanelGridData = new GridData();
		searchPanelGridData.grabExcessHorizontalSpace = true;
		searchPanelGridData.horizontalAlignment = SWT.FILL;
		searchPanel.setLayoutData(searchPanelGridData);

		// Label searchLabel = new Label(searchPanel, SWT.NONE);
		// searchLabel.setText("&Search:");

		this.searchBox = new Text(searchPanel, SWT.SEARCH);
		this.searchBox.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
					// ENTER: start a search
					search();
				} else if (e.keyCode == SWT.ESC) {
					// ESCAPE: restore the view after a search
					EcoreBrowser.this.searchBox.setText(""); //$NON-NLS-1$
					search();
				}
			}
		});

		this.searchBox.setMessage(Messages.EcoreBrowser_typeFilterText);
		this.searchBox.setData("org.eclipse.swtbot.widget.key", "EcoreBrowser.searchBox"); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Called when the user initiates a search. Find the list of all the
	 * instances of the selected metaclass that have a name containing the
	 * search string (entered by the user in the search box), in a case
	 * insensitive way. Set this list as input of the tree viewer.
	 * <p>
	 * Do the search in a Job that can be canceled by the user.
	 * <p>
	 * If the search box is empty, then restore the previous tree viewer input.
	 */
	protected void search() {
		final String searchText = this.searchBox.getText().toLowerCase();

		if (searchText.trim().length() == 0) {
			if (this.savedInput != null && this.savedInput != this.treeViewer.getInput()) {
				this.treeViewer.setInput(this.savedInput);
			}
			return;
		}

		final List<EClass> selectedMetaclasses = findSelectedEClasses();
		if (selectedMetaclasses.isEmpty()) {
			return;
		}

		this.searchJob = new Job(Messages.EcoreBrowser_jobName_SearchInModelEditor) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {

				final ResourceSet resourceSet = EcoreBrowser.this.fResourceSet;
				final EList<Resource> resources = resourceSet.getResources();
				if (resources.size() < 1) {
					return Status.OK_STATUS;
				}

				// the search results : will become the input of the tree viewer
				final SearchResults searchResults = new SearchResults(
						EcoreBrowser.this.browserConfiguration);

				final Resource resource = resources.get(0);
				final TreeIterator<EObject> allContents = resource.getAllContents();
				while (allContents.hasNext()) {
					final EObject eObject = allContents.next();
					if (selectedMetaclasses.contains(eObject.eClass())) {
						final ModelElementItem modelElementItem = new ModelElementItemEx(eObject,
								null, EcoreBrowser.this.browserConfiguration);
						final String objectText = modelElementItem.getName().toLowerCase();
						if (objectText.contains(searchText)) {
							searchResults.add(eObject);
						}
					}

					// if the user canceled the search
					if (monitor.isCanceled()) {
						return Status.CANCEL_STATUS;
					}
				}

				// do only the minimum amount of work in the UI thread
				Display.getDefault().syncExec(new Runnable() {
					public void run() {
						// do not try to restore selection
						EcoreBrowser.this.treeViewer.setSelection(null);
						EcoreBrowser.this.treeViewer.setInput(searchResults);
					}
				});
				return Status.OK_STATUS;
			}
		};
		this.searchJob.setUser(true);
		this.searchJob.setPriority(Job.LONG);
		this.searchJob.schedule();
	}

	/**
	 * @return the {@link EClass}es corresponding to the metaclasses selected in
	 *         the metaclass list (empty list if none)
	 */
	private List<EClass> findSelectedEClasses() {

		final ArrayList<EClass> selectedMetaclasses = new ArrayList<EClass>();

		final ISelection selection = this.metaclassViewer.getSelection();
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;

			final Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				final Object element = iterator.next();
				if (element instanceof EClass) {
					final EClass eClass = (EClass) element;
					addMetaclassToSelection(eClass, selectedMetaclasses, this.browserConfiguration
							.getAppearanceConfiguration().isDisplayInstancesOfSubclasses());
				}
			}
		}
		return selectedMetaclasses;
	}

	/**
	 * Adds the given metaclass to the given selection.
	 * 
	 * @param addSubclasses
	 *            also adds subclasses
	 */
	private void addMetaclassToSelection(final EClass eClass,
			final ArrayList<EClass> selectedMetaclasses, final boolean addSubclasses) {
		selectedMetaclasses.add(eClass);
		// add subclasses recursively
		if (addSubclasses) {
			InstancesForMetaclasses instancesForMetaclasses = this.browserConfiguration
					.getInstancesForMetaclasses();
			InstancesForMetaclass instancesForMetaclass = instancesForMetaclasses
					.getInstancesForMetaclass(eClass);
			if (instancesForMetaclass != null) {
				InstancesForMetaclass[] subclasses = instancesForMetaclass.getSubclasses();
				for (InstancesForMetaclass subclass : subclasses) {
					addMetaclassToSelection(subclass.getEClass(), selectedMetaclasses,
							addSubclasses);
				}
			} else {
				MoDiscoLogger.logError("Could not find subclasses of '" + eClass.getName() + "'", //$NON-NLS-1$ //$NON-NLS-2$
						MoDiscoBrowserPlugin.getPlugin());
			}

		}
	}

	/**
	 * Listen to selection changes in the metaclass list viewer, and update the
	 * tree viewer accordingly.
	 */
	private final class MetaclassListViewerSelectionChangedListener implements
			ISelectionChangedListener {

		/** Memorize the tree selection for each metaclass */
		private final HashMap<EClass, ITreeSelection> selections = new HashMap<EClass, ITreeSelection>();
		/** The metaclass that was selected before the selection changed */
		private EClass lastSelectedMetaclass = null;

		/**
		 * Get the selected entries corresponding to the given selection in the
		 * metaclass viewer (may be empty)
		 */
		private InstancesForMetaclass[] getEntriesForSelection(final ISelection selection) {
			final ArrayList<InstancesForMetaclass> instancesForMetaclasses = new ArrayList<InstancesForMetaclass>();

			if (selection instanceof IStructuredSelection) {
				final IStructuredSelection structuredSelection = (IStructuredSelection) selection;

				for (final Iterator<?> iterator = structuredSelection.iterator(); iterator
						.hasNext();) {
					final Object element = iterator.next();
					if (element instanceof InstancesForMetaclass) {
						final InstancesForMetaclass instancesForMetaclass = (InstancesForMetaclass) element;
						instancesForMetaclasses.add(instancesForMetaclass);
					}
				}
			}

			return instancesForMetaclasses
					.toArray(new InstancesForMetaclass[instancesForMetaclasses.size()]);
		}

		/**
		 * Updates the tree viewer when the selection changes in the metaclass
		 * viewer. Keeps track of the tree selection for each metaclass, and
		 * restores it when possible.
		 */
		public void selectionChanged(final SelectionChangedEvent event) {

			// save the selection
			if (this.lastSelectedMetaclass != null) {
				this.selections.put(this.lastSelectedMetaclass,
						(ITreeSelection) EcoreBrowser.this.treeViewer.getSelection());
			}

			final ISelection metaclassSelection = event.getSelection();
			final InstancesForMetaclass[] entries = getEntriesForSelection(metaclassSelection);

			ITreeSelection savedSelection = null;
			Object input = null;
			if (entries.length > 0) {
				savedSelection = this.selections.get(entries[0].getEClass());

				final EClass[] selectedMetaclasses = new EClass[entries.length];
				int i = 0;
				for (final InstancesForMetaclass entry : entries) {
					selectedMetaclasses[i++] = entry.getEClass();
				}

				input = new MetaclassList(EcoreBrowser.this.browserConfiguration,
						selectedMetaclasses);
				this.lastSelectedMetaclass = entries[0].getEClass();
			} else {
				this.lastSelectedMetaclass = null;
			}

			try {
				// speedup and avoid flickering
				EcoreBrowser.this.treeViewer.getTree().setRedraw(false);

				EcoreBrowser.this.searchBox.setText(""); //$NON-NLS-1$

				// do not restore selection twice
				EcoreBrowser.this.treeViewer.setSelection(null);

				// set the list of model elements as the input of the treeview
				EcoreBrowser.this.treeViewer.setInput(input);
				EcoreBrowser.this.savedInput = input;
				EcoreBrowser.this.treeViewer.setSelection(savedSelection, true);
			} finally {
				EcoreBrowser.this.treeViewer.getTree().setRedraw(true);
			}

			markNavigationLocation();
		}
	}

	public void setShowEmptyLinks(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowEmptyLinks(value);
		internalRefreshTree();
	}

	public boolean isShowEmptyLinks() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowEmptyLinks();
	}

	public void setSortInstances(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setSortInstances(value);
		internalRefreshTree();
	}

	public boolean isSortInstances() {
		return this.browserConfiguration.getAppearanceConfiguration().isSortInstances();
	}

	public void setShowDerivedLinks(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowDerivedLinks(value);
		internalRefreshTree();
	}

	public boolean isShowDerivedLinks() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowDerivedLinks();
	}

	public void setShowFullQualifiedNames(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowFullQualifiedNames(value);
		internalRefreshTree();
		this.metaclassViewer.refresh();
	}

	public boolean isShowFullQualifiedNames() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowFullQualifiedNames();
	}

	public void setShowMultiplicity(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowMultiplicity(value);
		internalRefreshTree();
	}

	public boolean isShowMultiplicity() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowMultiplicity();
	}

	public void setShowOppositeLinks(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowOppositeLinks(value);
		internalRefreshTree();
	}

	public boolean isShowOppositeLinks() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowOppositeLinks();
	}

	public void setShowContainer(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowContainer(value);
		internalRefreshTree();
	}

	public boolean isShowContainer() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowContainer();
	}

	public void setSortLinks(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setSortLinks(value);
		internalRefreshTree();
	}

	public boolean isSortLinks() {
		return this.browserConfiguration.getAppearanceConfiguration().isSortLinks();
	}

	public void setSortLinksByType(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setSortLinksByType(value);
		internalRefreshTree();
	}

	public boolean isSortLinksByType() {
		return this.browserConfiguration.getAppearanceConfiguration().isSortLinksByType();
	}

	public void setShowAttributes(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowAttributes(value);
		internalRefreshTree();
	}

	public boolean isShowAttributes() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowAttributes();
	}

	public void setShowEmptyAttributes(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowEmptyAttributes(value);
		internalRefreshTree();
	}

	public boolean isShowEmptyAttributes() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowEmptyAttributes();
	}

	public void setFontSizeDelta(final int value) {
		this.browserConfiguration.getAppearanceConfiguration().setFontSizeDelta(value);
		setupPreferredFont();
	}

	public int getFontSizeDelta() {
		return this.browserConfiguration.getAppearanceConfiguration().getFontSizeDelta();
	}

	public void setShowOrdering(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowOrdering(value);
		internalRefreshTree();
	}

	public boolean isShowOrdering() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowOrdering();
	}

	public void setShowTypeOfLinks(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowTypeOfLinks(value);
		internalRefreshTree();
	}

	public boolean isShowTypeOfLinks() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowTypeOfLinks();
	}

	public void setShowElementIDs(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowElementIDs(value);
		internalRefreshTree();
	}

	public boolean isShowElementIDs() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowElementIDs();
	}

	public void setShowURI(final boolean value) {
		this.browserConfiguration.getAppearanceConfiguration().setShowURI(value);
		internalRefreshTree();
	}

	public boolean isShowURI() {
		return this.browserConfiguration.getAppearanceConfiguration().isShowURI();
	}

	/** This is how the framework determines which interfaces we implement. */
	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(final Class key) {
		if (key.equals(IPropertySheetPage.class)) {
			return getPropertySheetPage();
		}
		return super.getAdapter(key);
	}

	/** This accesses a cached version of the property sheet page. */
	public IPropertySheetPage getPropertySheetPage() {
		if (this.propertySheetPage == null) {
			this.propertySheetPage = new ExtendedPropertySheetPage(this.editingDomain) {
				@Override
				public void setSelectionToViewer(final List<?> selection) {
					EcoreBrowser.this.setSelectionToViewer(selection);
					EcoreBrowser.this.setFocus();
				}

				@Override
				public void setActionBars(final IActionBars actionBars) {
					super.setActionBars(actionBars);
					getActionBarContributor().shareGlobalActions(this, actionBars);
				}

				@Override
				public void selectionChanged(final IWorkbenchPart part, final ISelection selection0) {
					ISelection selection = EcoreBrowser.this.getEditorSelection();

					boolean selectionNotified = false;
					if (selection instanceof IStructuredSelection) {
						if (((IStructuredSelection) selection).toArray().length == 1) {
							Object objSelected = ((IStructuredSelection) selection)
									.getFirstElement();
							if (objSelected instanceof ModelElementItem) {
								ModelElementItem modelElementItem = (ModelElementItem) objSelected;
								super.selectionChanged(part, new StructuredSelection(
										modelElementItem.getEObject()));
								selectionNotified = true;
							} else if (objSelected instanceof AttributeItem) {
								super.selectionChanged(part, new StructuredSelection(
										((AttributeItem) objSelected).getParent()));
								selectionNotified = true;
							} else if (objSelected instanceof LinkItem) {
								super.selectionChanged(part, new StructuredSelection(
										((LinkItem) objSelected).getParent()));
								selectionNotified = true;
							}
						}
					}
					if (!selectionNotified) {
						super.selectionChanged(part, StructuredSelection.EMPTY);
					}
				}
			};
			this.propertySheetPage.setPropertySourceProvider(new AdapterFactoryContentProvider(
					this.adapterFactoryWithRegistry) {
				@Override
				protected IPropertySource createPropertySource(final Object object,
						final IItemPropertySource itemPropertySource) {
					return new PropertySource(object, itemPropertySource) {
						@Override
						protected IPropertyDescriptor createPropertyDescriptor(
								final IItemPropertyDescriptor itemPropertyDescriptor) {
							return new PropertyDescriptor(this.object, itemPropertyDescriptor);
						}
					};
				}
			});
		}
		return this.propertySheetPage;
	}

	@Override
	public boolean isDirty() {
		if (isReadOnly()) {
			return false;
		}
		return ((BasicCommandStack) this.editingDomain.getCommandStack()).isSaveNeeded();
	}

	/**
	 * This is for implementing {@link org.eclipse.ui.IEditorPart} : saves the
	 * model file.
	 */
	@Override
	public void doSave(final IProgressMonitor progressMonitor) {
		if (isReadOnly()) {
			return;
		}

		// Save only resources that have actually changed.
		final Map<Object, Object> saveOptions = new HashMap<Object, Object>();
		saveOptions.put(Resource.OPTION_SAVE_ONLY_IF_CHANGED,
				Resource.OPTION_SAVE_ONLY_IF_CHANGED_MEMORY_BUFFER);

		// Do the work within an operation because this is a long running
		// activity that modifies the workbench.
		WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {
			// This is the method that gets invoked when the operation runs.
			@Override
			public void execute(final IProgressMonitor monitor) {
				// Save the resources to the file system.
				boolean first = true;
				for (Resource resource : EcoreBrowser.this.editingDomain.getResourceSet()
						.getResources()) {
					if ((first || !resource.getContents().isEmpty() || isPersisted(resource))
							&& !EcoreBrowser.this.editingDomain.isReadOnly(resource)) {
						try {
							long timeStamp = resource.getTimeStamp();
							resource.save(saveOptions);
							if (resource.getTimeStamp() != timeStamp) {
								EcoreBrowser.this.savedResources.add(resource);
							}
						} catch (final Exception e) {
							MoDiscoLogger.logError(e, MoDiscoBrowserPlugin.getPlugin());
							DialogUtils.openErrorDialog(getSite().getShell(), e,
									Messages.EcoreBrowser_errorSavingResource);
							EcoreBrowser.this.resourceToDiagnosticMap.put(resource,
									analyzeResourceProblems(resource, e));
						}
						first = false;
					}
				}
			}
		};

		this.updateProblemIndication = false;
		try {
			// This runs the options, and shows progress.
			new ProgressMonitorDialog(getSite().getShell()).run(true, false, operation);

			// Refresh the necessary state.
			((BasicCommandStack) this.editingDomain.getCommandStack()).saveIsDone();
			firePropertyChange(IEditorPart.PROP_DIRTY);
		} catch (Exception exception) {
			// Something went wrong that shouldn't.
			MoDiscoBrowserPlugin.getPlugin().log(exception);
		}
		this.updateProblemIndication = true;
		updateProblemIndication();
	}

	@Override
	public void doSaveAs() {
		SaveAsDialog saveAsDialog = new SaveAsDialog(getSite().getShell());
		saveAsDialog.open();
		IPath path = saveAsDialog.getResult();
		if (path != null) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
			if (file != null) {
				doSaveAs(URI.createPlatformResourceURI(file.getFullPath().toString(), true),
						new FileEditorInput(file));
			}
		}
	}

	protected void doSaveAs(final URI uri, final IEditorInput editorInput) {
		this.editingDomain.getResourceSet().getResources().get(0).setURI(uri);
		setInputWithNotify(editorInput);
		setPartName(editorInput.getName());
		IProgressMonitor progressMonitor;
		IStatusLineManager statusLineManager = getActionBars().getStatusLineManager();
		if (statusLineManager != null) {
			progressMonitor = statusLineManager.getProgressMonitor();
		} else {
			progressMonitor = new NullProgressMonitor();
		}
		doSave(progressMonitor);
	}

	@Override
	public boolean isSaveAsAllowed() {
		return !isReadOnly();
	}

	/**
	 * This returns whether something has been persisted to the URI of the
	 * specified resource. The implementation uses the URI converter from the
	 * editor's resource set to try to open an input stream.
	 */
	protected boolean isPersisted(final Resource resource) {
		boolean result = false;
		try {
			InputStream stream = this.editingDomain.getResourceSet().getURIConverter()
					.createInputStream(resource.getURI());
			if (stream != null) {
				result = true;
				stream.close();
			}
		} catch (IOException e) {
			result = false;
		}
		return result;
	}

	/**
	 * This is called during startup to initialize the editor with its site and
	 * input.
	 */
	@Override
	public void init(final IEditorSite site, final IEditorInput editorInput) {
		// TODO: check input is of expected type
		try {
			setSite(site);
			setInputWithNotify(editorInput);
			setPartName(editorInput.getName());
			site.setSelectionProvider(this);
			site.getPage().addPartListener(this.partListener);
			ResourcesPlugin.getWorkspace().addResourceChangeListener(this.resourceChangeListener,
					IResourceChangeEvent.POST_CHANGE);
		} catch (final Exception e) {
			MoDiscoLogger.logError(e, MoDiscoBrowserPlugin.getPlugin());
		}
	}

	/** Focus the tree viewer when the editor is focused */
	@Override
	public void setFocus() {
		if (this.treeViewer != null) {
			this.treeViewer.getTree().setFocus();
		}
	}

	/** This implements {@link org.eclipse.jface.viewers.ISelectionProvider}. */
	public void addSelectionChangedListener(final ISelectionChangedListener listener) {
		this.selectionChangedListeners.add(listener);
	}

	/** This implements {@link org.eclipse.jface.viewers.ISelectionProvider}. */
	public void removeSelectionChangedListener(final ISelectionChangedListener listener) {
		this.selectionChangedListeners.remove(listener);
	}

	/**
	 * Returns the "unwrapped" selection of the editor, with Ecore elements
	 * instead of the proxies used internally
	 */
	public ISelection getSelection() {
		return BrowserUtils.unwrapSelection(this.editorSelection);
	}

	/** @return the "real" selection, with proxy objects (ModelElementItem, etc.) */
	public ISelection getEditorSelection() {
		return this.editorSelection;
	}

	/**
	 * This implements {@link org.eclipse.jface.viewers.ISelectionProvider} to
	 * set this editor's overall selection. Calling this result will notify the
	 * listeners.
	 */
	public void setSelection(final ISelection selection) {
		this.editorSelection = selection;

		for (final ISelectionChangedListener listener : this.selectionChangedListeners) {
			// depending on the listener type, unwrap the selection
			if (listener instanceof IBrowserSelectionChangedListener) {
				// expose the proxy objects
				listener.selectionChanged(new SelectionChangedEvent(this, selection));
			} else {
				// expose the underlying Ecore elements
				listener.selectionChanged(new SelectionChangedEvent(this, BrowserUtils
						.unwrapSelection(selection)));
			}
		}
		setStatusLineMessage(selection);
	}

	/** Connect the given viewer to the editor's selection */
	private void hookViewerToSelection(final Viewer viewer) {
		if (viewer == null) {
			return;
		}

		if (this.selectionChangedListener == null) {
			// Create the listener on demand
			this.selectionChangedListener = new ISelectionChangedListener() {
				// This just notifies those things that are affected by the
				// selection
				public void selectionChanged(final SelectionChangedEvent selectionChangedEvent) {
					setSelection(selectionChangedEvent.getSelection());
				}
			};
		}

		viewer.addSelectionChangedListener(this.selectionChangedListener);

		// Set the editor's selection based on the viewer's selection
		setSelection(viewer.getSelection());
	}

	/** Set the status line message depending on the given selection */
	public void setStatusLineMessage(final ISelection selection) {
		// TODO: maybe display more useful information?
		final IStatusLineManager statusLineManager = getActionBars().getStatusLineManager();

		if (statusLineManager != null) {
			if (selection instanceof IStructuredSelection) {
				final Collection<?> collection = ((IStructuredSelection) selection).toList();
				switch (collection.size()) {
				case 0:
					statusLineManager.setMessage(Messages.EcoreBrowser_noObjectSelected);
					break;
				case 1:
					final Object object = collection.iterator().next();
					if (object instanceof ITreeElement) {
						final ITreeElement treeElement = (ITreeElement) object;
						statusLineManager.setMessage(NLS.bind(
								Messages.EcoreBrowser_singleObjectSelected, treeElement.getText()));
					}
					break;
				default:
					statusLineManager.setMessage(NLS.bind(
							Messages.EcoreBrowser_multiObjectSelected, Integer.toString(collection
									.size())));
					break;
				}
			} else {
				statusLineManager.setMessage(""); //$NON-NLS-1$
			}
		}
	}

	/**
	 * This implements {@link org.eclipse.jface.action.IMenuListener} to help
	 * fill the context menus with contributions from the Edit menu.
	 */
	public void menuAboutToShow(final IMenuManager menuManager) {
		((IMenuListener) getEditorSite().getActionBarContributor()).menuAboutToShow(menuManager);
	}

	public BrowserActionBarContributor getActionBarContributor() {
		return (BrowserActionBarContributor) getEditorSite().getActionBarContributor();
	}

	public IActionBars getActionBars() {
		return getActionBarContributor().getActionBars();
	}

	/** Clean up */
	@Override
	public void dispose() {
		this.disposing = true;
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this.resourceChangeListener);
		getSite().getPage().removePartListener(this.partListener);

		this.adapterFactoryWithRegistry.dispose();
		this.sashForm.dispose();

		if (getActionBarContributor().getActiveEditor() == this) {
			getActionBarContributor().setActiveEditor(null);
		}

		if (this.propertySheetPage != null) {
			this.propertySheetPage.dispose();
		}

		if (this.treeViewer != null && this.selectionChangedListener != null) {
			this.treeViewer.removeSelectionChangedListener(this.selectionChangedListener);
			this.selectionChangedListener = null;
		}

		// dispose fonts
		this.browserConfiguration.getAppearanceConfiguration().setCustomFont(null);
		this.browserConfiguration.getAppearanceConfiguration().setCustomItalicFont(null);

		// avoid leaking editor
		this.browserConfiguration.dispose();

		JFaceResources.getFontRegistry().removeListener(this.fontChangeListener);

		this.customTreePainter.dispose();
		// dispose the customization engine, which removes listeners too
		this.browserConfiguration.getAppearanceConfiguration().getCustomizationEngine().dispose();
		this.fCustomizationEngineChangeListener = null;

		FacetSetCatalog.getSingleton().removeChangeListener(this.fFacetsChangeListener);

		this.formToolkit.dispose();

		// avoid memory leak for EMF registry resources since a new
		// adapter is created each time the same resource is opened
		// in a different editor
		Set<Resource> browsedResources = this.browserConfiguration.getBrowsedResources();
		for (Resource resource : browsedResources) {
			ListIterator<Adapter> listIterator = resource.eAdapters().listIterator();
			while (listIterator.hasNext()) {
				Adapter adapter = listIterator.next();
				if (adapter instanceof MetaclassInstancesAdapterWithFacet) {
					MetaclassInstancesAdapterWithFacet adapterWithFacet = (MetaclassInstancesAdapterWithFacet) adapter;
					if (adapterWithFacet.getFacetContext() == getBrowserConfiguration()
							.getAppearanceConfiguration().getFacetContext()) {
						listIterator.remove();
					}
				}
			}
		}
		// let the FacetContext be GCed
		MetaclassInstancesAdapterFactoryWithFacet.getInstance().setFacetContext(null);

		if (!this.readOnly) {
			// unload all resources
			final EList<Resource> resources = this.fResourceSet.getResources();
			final Job cleanupJob = new Job(Messages.EcoreBrowser_jobName_UnloadResources) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					for (final Resource resource : resources) {
						resource.unload();
					}
					return Status.OK_STATUS;
				}
			};
			cleanupJob.setPriority(Job.DECORATE);
			cleanupJob.setSystem(true);
			cleanupJob.schedule();
		}

		super.dispose();
	}

	/** Open the selected element in the list of elements of the same metaclass */
	public void browse() {
		final ISelection selection = getEditorSelection();
		if (selection instanceof ITreeSelection) {
			final ITreeSelection treeSelection = (ITreeSelection) selection;
			final Object selectedElement = treeSelection.getFirstElement();
			if (selectedElement instanceof ModelElementItem) {
				final ModelElementItem modelElementItem = (ModelElementItem) selectedElement;
				browseTo(modelElementItem);
			}
		}
	}

	/** Open the given element in the list of elements of the same metaclass */
	public void browseTo(final ModelElementItem modelElementItem) {
		final EObject modelElement = modelElementItem.getEObject();
		browseTo(modelElement);
	}

	/** Open the given element in the list of elements of the same metaclass */
	public void browseTo(final EObject modelElement) {
		markNavigationLocation();
		// Select the metaclass in the metaclass list viewer
		this.metaclassViewer.selectMetaclass(modelElement.eClass());
		TreePath treePath = findTreePathForElement(modelElement);
		if (treePath != null) {
			// deselect any previously selected elements
			this.treeViewer.setSelection(null);
			this.treeViewer.setSelection(new TreeSelection(treePath), true);
			markNavigationLocation();
		}
	}

	public void browseToByURI(final String uriFragment) {
		EObject eObject = EMFUtil.findElementByURI(uriFragment, this.fResourceSet);
		if (eObject != null) {
			browseTo(eObject);
		}
	}

	/**
	 * Finds the path of the given element in the tree viewer. The element is
	 * looked for at the root and under BigListItemProviders in the case the
	 * elements are split.
	 * 
	 * @return the path or <code>null</code> if the element was not found
	 */
	private TreePath findTreePathForElement(final EObject eObject) {
		final Object input = this.treeViewer.getInput();
		if (input instanceof MetaclassList) {
			final MetaclassList metaclassListItemProvider = (MetaclassList) input;
			final Collection<?> elements = metaclassListItemProvider.getElements();
			for (final Object element : elements) {
				// if the elements are split
				if (element instanceof BigListItem) {
					final BigListItem bigListItemProvider = (BigListItem) element;
					final Collection<?> subElements = bigListItemProvider.getChildren();
					for (final Object subElement : subElements) {
						if (subElement instanceof ModelElementItem) {
							final ModelElementItem other = (ModelElementItem) subElement;
							if (other.getEObject().equals(eObject)) {
								return new TreePath(new Object[] { subElement });
							}
						}
					}
				}

				if (element instanceof ModelElementItem) {
					final ModelElementItem other = (ModelElementItem) element;
					if (other.getEObject().equals(eObject)) {
						return new TreePath(new Object[] { element });
					}
				}
			}
		}
		return null;
	}

	/** Implements {@link INavigationLocationProvider} */
	public INavigationLocation createEmptyNavigationLocation() {
		return new BrowserNavigationLocation(this);
	}

	/**
	 * Implements {@link INavigationLocationProvider}. Create a location that
	 * can be navigated back to, using the "back" button
	 */
	public INavigationLocation createNavigationLocation() {
		// navigation to the first element of the selection
		final EClass firstSelectedMetaclass = this.metaclassViewer.getFirstSelectedMetaclass();
		return new BrowserNavigationLocation(this, firstSelectedMetaclass,
				(ITreeSelection) this.treeViewer.getSelection());
	}

	/** Mark a location that can be navigated back to, using the "back" button */
	protected void markNavigationLocation() {
		getSite().getPage().getNavigationHistory().markLocation(this);
	}

	/**
	 * Restore a location (used for implementing history).
	 * 
	 * @param eClass
	 *            the metaclass viewer's selection
	 * @param selection
	 *            the tree viewer's selection
	 */
	public void restoreLocation(final EClass eClass, final ITreeSelection selection) {
		this.metaclassViewer.selectMetaclass(eClass);
		this.treeViewer.setSelection(selection);
	}

	/**
	 * Refresh the editor
	 * 
	 * @param metaclassViewerToo
	 *            whether the metaclass view must be refreshed too, or only the
	 *            tree viewer
	 */
	public void refreshDelayed(final boolean metaclassViewerToo) {
		if (this.disposing) {
			return;
		}
		if (this.refreshJob == null) {
			this.refreshJob = new Job(Messages.EcoreBrowser_refreshingModelViewers) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							internalRefreshTree();
							if (metaclassViewerToo) {
								getMetaclassViewer().refresh();
							}
						}
					});
					return Status.OK_STATUS;
				}
			};
		} else {
			/*
			 * If refreshTreeDelayed is called again before the job delay
			 * passed, then cancel the job and re-schedule a new one.
			 */
			this.refreshJob.cancel();
		}
		this.refreshJob.setPriority(Job.INTERACTIVE);
		this.refreshJob.schedule();
	}

	public void clearInstancesCache() {
		AppearanceConfiguration appearanceConfiguration = this.browserConfiguration
				.getAppearanceConfiguration();
		MetaclassInstancesAdapterFactoryWithFacet.getInstance().setFacetContext(
				appearanceConfiguration.getFacetContext());
		Set<Resource> browsedResources = this.browserConfiguration.getBrowsedResources();
		for (Resource resource : browsedResources) {
			MetaclassInstances instances = (MetaclassInstances) MetaclassInstancesAdapterFactoryWithFacet
					.getInstance().adapt(resource, MetaclassInstances.class);
			instances.clearCache();
		}
	}

	private void internalRefreshTree() {
		if (!this.treeViewer.getTree().isDisposed()) {
			try {
				this.treeViewer.getTree().setRedraw(false);
				this.treeViewer.refresh();
			} finally {
				this.treeViewer.getTree().setRedraw(true);
			}
		}
	}

	public BrowserConfiguration getBrowserConfiguration() {
		return this.browserConfiguration;
	}

	private void openPropertiesView() {
		try {
			getEditorSite().getPage().showView(EcoreBrowser.PROPERTY_VIEW_ID);
		} catch (final PartInitException e) {
			MoDiscoBrowserPlugin.logException(e);
		}
	}

	public ResourceSet getResourceSet() {
		return this.fResourceSet;
	}

	public void loadCustomizations(final List<MetamodelView> customizationsToLoad) {
		final CustomizationEngine customizationEngine = this.browserConfiguration
				.getAppearanceConfiguration().getCustomizationEngine();

		if (this.fCustomizationEngineChangeListener == null) {
			this.fCustomizationEngineChangeListener = new ChangeListener() {
				public void changed() {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							refreshDelayed(true);
						}
					});
				}
			};
			customizationEngine.addChangeListener(this.fCustomizationEngineChangeListener);
		}

		customizationEngine.clear();
		for (final MetamodelView customizationToLoad : customizationsToLoad) {
			customizationEngine.registerCustomization(customizationToLoad);
		}
		customizationEngine.loadCustomizations();
	}

	private void loadDefaultCustomizations() {
		loadCustomizations(CustomizationsCatalog.getInstance().getRegistryDefaultCustomizations());
	}

	/** used by unit tests */
	protected TreeToolBar getInstancesToolBar() {
		return this.instancesToolBar;
	}

	protected void reloadOnFacetChange() {
		if (this.fFacetsChangeListener == null) {
			this.fFacetsChangeListener = new ModiscoCatalogChangeListener() {
				private void changed() {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							// reloading Facets by name, since the elements are
							// different after a FacetSet model reload
							List<String> facetsToReload = new ArrayList<String>();
							Set<Facet> loadedFacets = getBrowserConfiguration()
									.getAppearanceConfiguration().getLoadedFacets();
							for (Facet facet : loadedFacets) {
								FacetSet facetSet = facet.getFacetSet();
								facetsToReload.add(facetSet.getName() + "#" + facet.getName()); //$NON-NLS-1$
							}
							getBrowserConfiguration().getAppearanceConfiguration().clearFacets();

							Collection<FacetSet> allFacetSets = FacetSetCatalog.getSingleton()
									.getAllFacetSets();

							for (FacetSet facetSet : allFacetSets) {
								EList<Facet> facets = facetSet.getFacets();
								for (Facet facet : facets) {
									for (String fullname : facetsToReload) {
										if (fullname.equals(facetSet.getName() + "#" //$NON-NLS-1$
												+ facet.getName())) {
											getBrowserConfiguration().getAppearanceConfiguration()
													.loadFacet(facet);
											break;
										}
									}
								}
							}
							// for modified facets
							getBrowserConfiguration().getInstancesForMetaclasses()
									.buildDerivationTree();
						}
					});
				}

				public void removed(final IFile file) {
					changed();
				}

				public void changed(final EObject eObject, final IFile file) {
					changed();
				}

				public void added(final EObject eObject, final IFile file) {
					changed();
				}
			};
			FacetSetCatalog.getSingleton().addChangeListener(this.fFacetsChangeListener);
		}
	}

	public void modelChanged() {
		getBrowserConfiguration().getAppearanceConfiguration().touch();
		refreshDelayed(true);
	}

	/**
	 * Handles activation of the editor or it's associated views.
	 */
	protected void handleActivate() {
		// Recompute the read only state.
		if (this.editingDomain.getResourceToReadOnlyMap() != null) {
			this.editingDomain.getResourceToReadOnlyMap().clear();

			// Refresh any actions that may become enabled or disabled.
			setSelection(getEditorSelection());
		}

		if (!this.removedResources.isEmpty()) {
			if (handleDirtyConflict()) {
				getSite().getPage().closeEditor(EcoreBrowser.this, false);
			} else {
				this.removedResources.clear();
				this.changedResources.clear();
				this.savedResources.clear();
			}
		} else if (!this.changedResources.isEmpty()) {
			this.changedResources.removeAll(this.savedResources);
			handleChangedResources();
			this.changedResources.clear();
			this.savedResources.clear();
		}
	}

	/**
	 * Shows a dialog that asks if conflicting changes should be discarded.
	 */
	protected boolean handleDirtyConflict() {
		return MessageDialog.openQuestion(getSite().getShell(), Messages.UI_FileConflict_label,
				Messages.WARN_FileConflict);
	}

	/**
	 * Handles what to do with changed resources on activation.
	 */
	protected void handleChangedResources() {
		if (!this.changedResources.isEmpty() && (!isDirty() || handleDirtyConflict())) {
			if (isDirty()) {
				this.changedResources.addAll(this.editingDomain.getResourceSet().getResources());
			}
			this.editingDomain.getCommandStack().flush();

			this.updateProblemIndication = false;
			for (Resource resource : this.changedResources) {
				if (resource.isLoaded()) {
					resource.unload();
					try {
						resource.load(Collections.EMPTY_MAP);
					} catch (IOException e) {
						MoDiscoLogger.logError(e, MoDiscoBrowserPlugin.getPlugin());
						DialogUtils.openErrorDialog(getSite().getShell(), e,
								Messages.EcoreBrowser_errorLoadingResource);
						if (!this.resourceToDiagnosticMap.containsKey(resource)) {
							this.resourceToDiagnosticMap.put(resource, analyzeResourceProblems(
									resource, e));
						}
					}
				}
			}

			if (AdapterFactoryEditingDomain.isStale(this.editorSelection)) {
				setSelection(StructuredSelection.EMPTY);
			}

			this.updateProblemIndication = true;
			updateProblemIndication();
		}
	}

	/**
	 * Updates the problems indication with the information described in the
	 * specified diagnostic.
	 */
	protected void updateProblemIndication() {
		if (this.updateProblemIndication) {
			BasicDiagnostic diagnostic = new BasicDiagnostic(Diagnostic.OK,
					MoDiscoBrowserPlugin.PLUGIN_ID, 0, null, new Object[] { this.editingDomain
							.getResourceSet() });
			for (Diagnostic childDiagnostic : this.resourceToDiagnosticMap.values()) {
				if (childDiagnostic.getSeverity() != Diagnostic.OK) {
					diagnostic.add(childDiagnostic);
				}
			}

			// FIXME: diagnostic is not displayed to the user

			if (this.markerHelper.hasMarkers(this.editingDomain.getResourceSet())) {
				this.markerHelper.deleteMarkers(this.editingDomain.getResourceSet());
				if (diagnostic.getSeverity() != Diagnostic.OK) {
					try {
						this.markerHelper.createMarkers(diagnostic);
					} catch (CoreException exception) {
						MoDiscoBrowserPlugin.getPlugin().log(exception);
					}
				}
			}
		}
	}

	/**
	 * Returns a diagnostic describing the errors and warnings listed in the
	 * resource and the specified exception (if any).
	 */
	public Diagnostic analyzeResourceProblems(final Resource resource, final Exception exception) {
		if (!resource.getErrors().isEmpty() || !resource.getWarnings().isEmpty()) {
			Object errorData;
			if (exception == null) {
				errorData = resource;
			} else {
				errorData = exception;
			}
			BasicDiagnostic basicDiagnostic = new BasicDiagnostic(Diagnostic.ERROR,
					MoDiscoBrowserPlugin.PLUGIN_ID, 0, NLS.bind(Messages.UI_ErrorFile_message,
							resource.getURI()), new Object[] { errorData });
			basicDiagnostic.merge(EcoreUtil.computeDiagnostic(resource, true));
			return basicDiagnostic;
		} else if (exception != null) {
			return new BasicDiagnostic(Diagnostic.ERROR, MoDiscoBrowserPlugin.PLUGIN_ID, 0, NLS
					.bind(Messages.UI_ErrorFile_message, resource.getURI()),
					new Object[] { exception });
		} else {
			return Diagnostic.OK_INSTANCE;
		}
	}

	/**
	 * This sets up the editing domain for the model editor
	 */
	protected void initializeEditingDomain() {
		// Create an adapter factory that yields item providers.
		this.adapterFactoryWithRegistry = new ComposedAdapterFactory(
				ComposedAdapterFactory.Descriptor.Registry.INSTANCE);

		this.adapterFactoryWithRegistry.addAdapterFactory(new ResourceItemProviderAdapterFactory());
		this.adapterFactoryWithRegistry
				.addAdapterFactory(new ReflectiveItemProviderAdapterFactory());

		// Create the command stack that will notify this editor as commands are
		// executed.
		BasicCommandStack commandStack = new BasicCommandStack();

		commandStack.addCommandStackListener(new CommandStackListener() {
			public void commandStackChanged(final EventObject event) {
				getParentComposite().getDisplay().asyncExec(new Runnable() {
					public void run() {
						firePropertyChange(IEditorPart.PROP_DIRTY);
						if (EcoreBrowser.this.propertySheetPage != null
								&& !EcoreBrowser.this.propertySheetPage.getControl().isDisposed()) {
							EcoreBrowser.this.propertySheetPage.refresh();
						}
					}
				});
			}
		});

		// Create the editing domain with a special command stack.
		setEditingDomain(new AdapterFactoryEditingDomain(this.adapterFactoryWithRegistry,
				commandStack, new HashMap<Resource, Boolean>()));
	}

	/** added for integration into Papyrus */
	protected void setEditingDomain(final AdapterFactoryEditingDomain editingDomain) {
		this.editingDomain = editingDomain;
	}

	/** This is here for the listener to be able to call it. */
	@Override
	protected void firePropertyChange(final int action) {
		super.firePropertyChange(action);
	}

	/**
	 * Implements {@link IGotoMarker}. Select the EObject corresponding to the
	 * given marker
	 */
	public void gotoMarker(final IMarker marker) {
		try {
			if (marker.getType().equals(EValidator.MARKER)) {
				String uriAttribute = marker.getAttribute(EValidator.URI_ATTRIBUTE, null);
				if (uriAttribute != null) {
					URI uri = URI.createURI(uriAttribute);
					EObject eObject = this.fResourceSet.getEObject(uri, true);
					if (eObject != null) {
						browseTo(eObject);
					}
				}
			}
		} catch (Exception e) {
			MoDiscoLogger.logError(e, MoDiscoBrowserPlugin.getPlugin());
		}
	}

	public Composite getParentComposite() {
		return this.parentComposite;
	}

	public EditingDomain getEditingDomain() {
		return this.editingDomain;
	}

	public Collection<Resource> getSavedResources() {
		return this.savedResources;
	}

	public Collection<Resource> getChangedResources() {
		return this.changedResources;
	}

	public Collection<Resource> getRemovedResources() {
		return this.removedResources;
	}

	public boolean isReadOnly() {
		return this.readOnly;
	}
}
