/*******************************************************************************
 * Copyright (c) 2008, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    Nicolas Payneau (Mia-Software) - Enable editability
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.ui.action.CreateChildAction;
import org.eclipse.emf.edit.ui.action.CreateSiblingAction;
import org.eclipse.emf.edit.ui.action.EditingDomainActionBarContributor;
import org.eclipse.gmt.modisco.infra.browser.Messages;
import org.eclipse.gmt.modisco.infra.browser.actions.BrowseAction;
import org.eclipse.gmt.modisco.infra.browser.actions.DecreaseFontSizeAction;
import org.eclipse.gmt.modisco.infra.browser.actions.ExecuteSelectedQueryAction;
import org.eclipse.gmt.modisco.infra.browser.actions.IBrowserActionBarContributor;
import org.eclipse.gmt.modisco.infra.browser.actions.IncreaseFontSizeAction;
import org.eclipse.gmt.modisco.infra.browser.actions.LoadCustomizationsAction;
import org.eclipse.gmt.modisco.infra.browser.actions.LoadFacetsAction;
import org.eclipse.gmt.modisco.infra.browser.actions.OpenTableEditorAction;
import org.eclipse.gmt.modisco.infra.browser.actions.RefreshViewerAction;
import org.eclipse.gmt.modisco.infra.browser.actions.RemoveQueryAction;
import org.eclipse.gmt.modisco.infra.browser.actions.RestoreFontSizeAction;
import org.eclipse.gmt.modisco.infra.browser.actions.SelectQueryAction;
import org.eclipse.gmt.modisco.infra.browser.actions.ShowPropertiesViewAction;
import org.eclipse.gmt.modisco.infra.browser.core.QueryItem;
import org.eclipse.gmt.modisco.infra.browser.editors.table.TableEditor;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.model.LinkItem;
import org.eclipse.gmt.modisco.infra.browser.uicore.internal.model.ModelElementItem;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IContributionManager;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.SubContributionItem;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IEditorPart;

/** Action bar contributor for the MoDisco Ecore model browser. */
public class BrowserActionBarContributor extends EditingDomainActionBarContributor implements
		IBrowserSelectionChangedListener, IBrowserActionBarContributor {

	private static final String EDIT_MENU_ID = "org.eclipse.gmt.modisco.infra.browser.editMenu"; //$NON-NLS-1$

	/** This keeps track of the active editor. */
	private IEditorPart fActiveEditor;

	/** This keeps track of the current selection provider. */
	private ISelectionProvider selectionProvider;

	/** This action opens the "Properties" view. */
	private final IAction showPropertiesViewAction = new ShowPropertiesViewAction();

	public EcoreBrowser getBrowser() {
		return (EcoreBrowser) this.fActiveEditor;
	}

	/** This action opens a table editor on the current selection. */
	private final IAction openTableEditorAction = new OpenTableEditorAction(this);

	/**
	 * This action refreshes the viewer of the current editor if the editor
	 * implements {@link org.eclipse.emf.common.ui.viewer.IViewerProvider}.
	 */
	private final IAction refreshViewerAction = new RefreshViewerAction(this);

	/**
	 * This action makes it possible to select a query to be shown on each
	 * occurrence of the selected model element
	 */
	private final IAction selectQueryAction = new SelectQueryAction(this);

	/** Executes the selected query(-ies) */
	private final IAction executeSelectedQueryAction = new ExecuteSelectedQueryAction(this);

	/** Removes the selected query(-ies) */
	private final IAction removeQueryAction = new RemoveQueryAction(this);

	/**
	 * The current selection, updated when an event is received from the
	 * selection provider
	 */
	private ISelection fSelection;

	public ISelection getSelection() {
		return this.fSelection;
	}

	/**
	 * An action to show the selected element in the list of instances of its
	 * metaclass
	 */
	private final Action browseAction = new BrowseAction(this);

	/** An action to increase font size */
	private final Action increaseFontSizeAction = new IncreaseFontSizeAction(this);

	/** An action to decrease font size */
	private final Action decreaseFontSizeAction = new DecreaseFontSizeAction(this);

	/** An action to restore font size */
	private final Action restoreFontSizeAction = new RestoreFontSizeAction(this);

	private final Action loadCustomizationsAction = new LoadCustomizationsAction(this);

	private final Action loadFacetsAction = new LoadFacetsAction(this);

	/**
	 * This will contain one
	 * {@link org.eclipse.emf.edit.ui.action.CreateChildAction} corresponding to
	 * each descriptor generated for the current selection by the item provider.
	 */
	private Collection<IAction> createChildActions;

	/**
	 * This is the menu manager into which menu contribution items should be
	 * added for CreateChild actions.
	 */
	private IMenuManager createChildMenuManager;

	private IMenuManager editorMenuManager;

	// /**
	// * This will contain one
	// * {@link org.eclipse.emf.edit.ui.action.CreateSiblingAction}
	// corresponding
	// * to each descriptor generated for the current selection by the item
	// * provider.
	// */
	// private Collection<IAction> createSiblingActions;
	//
	// /**
	// * This is the menu manager into which menu contribution items should be
	// * added for CreateSibling actions.
	// */
	// private IMenuManager createSiblingMenuManager;

	public BrowserActionBarContributor() {
		super(EditingDomainActionBarContributor.ADDITIONS_LAST_STYLE);
	}

	/** This adds {@link Action}s to the tool bar. */
	@Override
	public void contributeToToolBar(final IToolBarManager toolBarManager) {
		toolBarManager.add(this.increaseFontSizeAction);
		toolBarManager.add(this.decreaseFontSizeAction);
		toolBarManager.add(this.restoreFontSizeAction);
		toolBarManager.add(this.loadCustomizationsAction);
		toolBarManager.add(this.loadFacetsAction);
	}

	/**
	 * When the active editor changes, this remembers the change and registers
	 * with it as a selection provider.
	 */
	@Override
	public void setActiveEditor(final IEditorPart part) {
		super.setActiveEditor(part);
		this.fActiveEditor = part;

		// Switch to the new selection provider.
		if (this.selectionProvider != null) {
			this.selectionProvider.removeSelectionChangedListener(this);
		}

		if (part == null) {
			this.selectionProvider = null;
		} else {
			if (part instanceof EcoreBrowser) {
				final EcoreBrowser browser = (EcoreBrowser) part;
				this.selectionProvider = browser.getSite().getSelectionProvider();
				this.selectionProvider.addSelectionChangedListener(this);

				// Fake a selection changed event to update the menus.
				if (browser.getEditorSelection() != null) {
					selectionChanged(new SelectionChangedEvent(this.selectionProvider, browser
							.getEditorSelection()));
				}
			}
		}
	}

	/**
	 * This implements
	 * {@link org.eclipse.jface.viewers.ISelectionChangedListener}, memorizing
	 * {@link org.eclipse.jface.viewers.SelectionChangedEvent}s to be able to
	 * build the menu when they are about to be shown (see
	 * {@link BrowserActionBarContributor#menuAboutToShow(IMenuManager)}).
	 */
	public void selectionChanged(final SelectionChangedEvent event) {
		this.fSelection = event.getSelection();
	}

	/** Update actions state depending on the selection in the editor */
	private void updateActions(final ISelection selection) {
		this.browseAction.setEnabled(enableBrowse(selection));
		this.openTableEditorAction.setEnabled(TableEditor.canBeOpenedOnSelection(selection));
	}

	/**
	 * Whether to enable the browse action. The browse action is enabled only if
	 * the selected element corresponds to an EObject.
	 */
	private boolean enableBrowse(final ISelection selection) {
		if (selection instanceof IStructuredSelection) {
			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			if (structuredSelection.size() != 1) {
				return false;
			}
			final Object element = structuredSelection.getFirstElement();
			if (element instanceof ModelElementItem) {
				return true;
				// return EMFUtil.isInFirstResource(((ModelElementItem)
				// element).getEObject());
			}
		}
		return false;
	}

	/** This populates the pop-up menu before it appears. */
	@Override
	public void menuAboutToShow(final IMenuManager menuManager) {

		if (!getBrowser().isReadOnly()) {
			updateMenu();
			MenuManager submenuManager = null;

			submenuManager = new MenuManager(Messages.UI_CreateChild_menu_item);
			populateManager(submenuManager, this.createChildActions, null);
			menuManager.add(submenuManager);

			// submenuManager = new
			// MenuManager(Messages.UI_CreateSibling_menu_item);
			// populateManager(submenuManager, this.createSiblingActions, null);
			//		menuManager.insertBefore("edit", submenuManager); //$NON-NLS-1$
		}

		menuManager.add(this.browseAction);
		menuManager.add(this.openTableEditorAction);

		int nSelectedModelElementItems = selectionCount(ModelElementItem.class);
		if (nSelectedModelElementItems > 0) {
			menuManager.add(this.selectQueryAction);
			if (nSelectedModelElementItems > 1) {
				this.selectQueryAction.setText(Messages.SelectQueryAction_addQueryOnElements);
			} else {
				this.selectQueryAction.setText(Messages.SelectQueryAction_addQueryOnElement);
			}
		}

		int nSelectedQueries = selectionCount(QueryItem.class);
		if (nSelectedQueries > 0) {
			menuManager.add(this.executeSelectedQueryAction);
			menuManager.add(this.removeQueryAction);
			if (nSelectedQueries > 1) {
				this.executeSelectedQueryAction.setText(Messages.ExecuteQueryAction_executeQueries);
				this.removeQueryAction.setText(Messages.RemoveQueryAction_removeQueries);
			} else {
				this.executeSelectedQueryAction
						.setText(Messages.ExecuteQueryAction_executeSelectedQuery);
				this.removeQueryAction.setText(Messages.RemoveQueryAction_removeQuery);
			}
		}

		if (!getBrowser().isReadOnly()) {
			super.menuAboutToShow(menuManager);
		}

		updateActions(getSelection());

		menuManager.add(new Separator());
		menuManager.add(this.showPropertiesViewAction);
		menuManager.add(this.refreshViewerAction);
	}

	/**
	 * De-populate and re-populate the menu based on the current selection.
	 * Query for the children and siblings that can be added to the selected
	 * object and update the menus accordingly.
	 */
	private void updateMenu() {
		// Remove any menu items for old selection
		if (this.createChildMenuManager != null) {
			depopulateManager(this.createChildMenuManager, this.createChildActions);
		}
		// if (this.createSiblingMenuManager != null) {
		// depopulateManager(this.createSiblingMenuManager,
		// this.createSiblingActions);
		// }

		Collection<?> newChildDescriptors = null;
		// Collection<?> newSiblingDescriptors = null;
		ISelection selectionOnBrowser = null;

		if (this.fSelection instanceof IStructuredSelection
				&& ((IStructuredSelection) this.fSelection).size() == 1) {
			if (this.fActiveEditor instanceof EcoreBrowser) {
				selectionOnBrowser = ((EcoreBrowser) this.fActiveEditor).getSelection();
				Object selectedObject = ((IStructuredSelection) this.fSelection).getFirstElement();

				EditingDomain domain = getBrowser().getEditingDomain();
				newChildDescriptors = domain.getNewChildDescriptors(selectedObject, null);
				// newSiblingDescriptors = domain.getNewChildDescriptors(null,
				// selectedObject);

				if (selectedObject instanceof LinkItem) {
					LinkItem linkItem = (LinkItem) selectedObject;
					selectionOnBrowser = new StructuredSelection(((ModelElementItem) linkItem
							.getTreeParent()).getEObject());
				}
			}
		}

		// Generate actions for selection; populate and redraw the menus
		this.createChildActions = generateCreateChildActions(newChildDescriptors,
				selectionOnBrowser);
		// this.createSiblingActions =
		// generateCreateSiblingActions(newSiblingDescriptors,
		// selectionOnBrowser);

		if (this.createChildMenuManager != null) {
			populateManager(this.createChildMenuManager, this.createChildActions, null);
			this.createChildMenuManager.update(true);
		}
		// if (this.createSiblingMenuManager != null) {
		// populateManager(this.createSiblingMenuManager,
		// this.createSiblingActions, null);
		// this.createSiblingMenuManager.update(true);
		// }
	}

	private int selectionCount(final Class<?> clazz) {
		ISelection selection = getSelection();
		int count = 0;
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Iterator<?> iterator = structuredSelection.iterator();
			while (iterator.hasNext()) {
				Object element = iterator.next();
				if (clazz.isAssignableFrom(element.getClass())) {
					count++;
				}
			}
		}
		return count;
	}

	@Override
	public IEditorPart getActiveEditor() {
		return this.fActiveEditor;
	}

	@Override
	public void contributeToMenu(final IMenuManager menuManager) {
		super.contributeToMenu(menuManager);

		this.editorMenuManager = new MenuManager(Messages.UI_modiscoEditor_menu,
				BrowserActionBarContributor.EDIT_MENU_ID);

		menuManager.insertAfter("additions", this.editorMenuManager); //$NON-NLS-1$

		this.editorMenuManager.add(this.increaseFontSizeAction);
		this.editorMenuManager.add(this.decreaseFontSizeAction);
		this.editorMenuManager.add(this.restoreFontSizeAction);
		this.editorMenuManager.add(this.loadCustomizationsAction);
		this.editorMenuManager.add(this.loadFacetsAction);
		this.editorMenuManager.add(new Separator());
		this.editorMenuManager.add(this.showPropertiesViewAction);
		this.editorMenuManager.add(this.refreshViewerAction);

		this.editorMenuManager.add(new Separator("settings")); //$NON-NLS-1$
		this.editorMenuManager.add(new Separator("actions")); //$NON-NLS-1$
		this.editorMenuManager.add(new Separator("additions")); //$NON-NLS-1$
		this.editorMenuManager.add(new Separator("additions-end")); //$NON-NLS-1$

		// Prepare for CreateChild item addition or removal.
		this.createChildMenuManager = new MenuManager(Messages.UI_CreateChild_menu_item);
		this.editorMenuManager.insertBefore("additions", this.createChildMenuManager); //$NON-NLS-1$

		// Prepare for CreateSibling item addition or removal.
		// this.createSiblingMenuManager = new
		// MenuManager(Messages.UI_CreateSibling_menu_item);
		//		submenuManager.insertBefore("additions", this.createSiblingMenuManager); //$NON-NLS-1$

		// Force an update because Eclipse hides empty menus now.
		this.editorMenuManager.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(final IMenuManager menuManag) {
				menuManag.updateAll(true);
			}
		});

		addGlobalActions(this.editorMenuManager);
	}

	/**
	 * This generates a {@link org.eclipse.emf.edit.ui.action.CreateChildAction}
	 * for each object in <code>descriptors</code>, and returns the collection
	 * of these actions.
	 */
	protected Collection<IAction> generateCreateChildActions(final Collection<?> descriptors,
			final ISelection selection) {
		Collection<IAction> actions = new ArrayList<IAction>();
		if (descriptors != null) {
			for (Object descriptor : descriptors) {
				actions.add(new CreateChildAction(this.fActiveEditor, selection, descriptor));
			}
		}
		return actions;
	}

	/**
	 * This generates a
	 * {@link org.eclipse.emf.edit.ui.action.CreateSiblingAction} for each
	 * object in <code>descriptors</code>, and returns the collection of these
	 * actions.
	 */
	protected Collection<IAction> generateCreateSiblingActions(final Collection<?> descriptors,
			final ISelection selection) {
		Collection<IAction> actions = new ArrayList<IAction>();
		if (descriptors != null) {
			for (Object descriptor : descriptors) {
				actions.add(new CreateSiblingAction(this.fActiveEditor, selection, descriptor));
			}
		}
		return actions;
	}

	/**
	 * This populates the specified <code>manager</code> with
	 * {@link org.eclipse.jface.action.ActionContributionItem}s based on the
	 * {@link org.eclipse.jface.action.IAction}s contained in the
	 * <code>actions</code> collection, by inserting them before the specified
	 * contribution item <code>contributionID</code>. If
	 * <code>contributionID</code> is <code>null</code>, they are simply added.
	 */
	protected void populateManager(final IContributionManager manager,
			final Collection<? extends IAction> actions, final String contributionID) {
		if (actions != null) {
			for (IAction action : actions) {
				if (contributionID != null) {
					manager.insertBefore(contributionID, action);
				} else {
					manager.add(action);
				}
			}
		}
	}

	/**
	 * This removes from the specified <code>manager</code> all
	 * {@link org.eclipse.jface.action.ActionContributionItem}s based on the
	 * {@link org.eclipse.jface.action.IAction}s contained in the
	 * <code>actions</code> collection.
	 */
	protected void depopulateManager(final IContributionManager manager,
			final Collection<? extends IAction> actions) {
		if (actions != null) {
			IContributionItem[] items = manager.getItems();
			for (IContributionItem contributionItem : items) {
				// Look into SubContributionItems
				while (contributionItem instanceof SubContributionItem) {
					contributionItem = ((SubContributionItem) contributionItem).getInnerItem();
				}

				// Delete the ActionContributionItems with matching action.
				if (contributionItem instanceof ActionContributionItem) {
					IAction action = ((ActionContributionItem) contributionItem).getAction();
					if (actions.contains(action)) {
						manager.remove(contributionItem);
					}
				}
			}
		}
	}

	@Override
	protected boolean removeAllReferencesOnDelete() {
		return true;
	}
}
