/*******************************************************************************
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Grgoire Dup (Mia-Software)
 *    Fabien Giquel (Mia-Software)
 *******************************************************************************/
package org.eclipse.gmt.modisco.common.core.utils;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.gmt.modisco.common.core.CommonModiscoActivator;
import org.eclipse.gmt.modisco.common.core.Messages;
import org.eclipse.gmt.modisco.common.core.MoDiscoProject;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.launching.JavaRuntime;
import org.osgi.framework.Bundle;

/**
 * @author Gregoire Dupe (Mia-Software), Fabien Giquel (Mia-Software)
 * 
 */
public final class ProjectUtils {

	private ProjectUtils() {
		// Nothing
	}

	private static final String JAVA_VERSION = "J2SE-1.5"; //$NON-NLS-1$

	public static void addPdeClassPath(final IProject project) throws JavaModelException {
		IJavaProject javaProject = JavaCore.create(project);
		IClasspathEntry[] oldClassPath = javaProject.getRawClasspath();
		IClasspathEntry[] newClassPath = new IClasspathEntry[oldClassPath.length + 1];
		System.arraycopy(oldClassPath, 0, newClassPath, 0, oldClassPath.length);
		newClassPath[oldClassPath.length] = JavaCore.newContainerEntry(new Path(
				"org.eclipse.pde.core.requiredPlugins")); //$NON-NLS-1$
		javaProject.setRawClasspath(newClassPath, new NullProgressMonitor());
	}

	/**
	 * @author Grgoire Dup (Mia-Software) - Removing "Require-Bundle"
	 *         statement
	 */
	public static void createManifest(final IProject project) throws CoreException {
		IFolder folder = project.getFolder("META-INF"); //$NON-NLS-1$
		if (!folder.exists()) {
			folder.create(true, true, new NullProgressMonitor());
		}
		IFile manifestFile = folder.getFile("MANIFEST.MF"); //$NON-NLS-1$
		if (!manifestFile.exists()) {
			StringBuffer manifestSB = new StringBuffer();
			manifestSB.append("Manifest-Version: 1.0\n"); //$NON-NLS-1$
			manifestSB.append("Bundle-ManifestVersion: 2\n"); //$NON-NLS-1$
			manifestSB.append("Bundle-Name: " + project.getName() + "\n"); //$NON-NLS-1$ //$NON-NLS-2$
			manifestSB.append("Bundle-SymbolicName: " + project.getName() //$NON-NLS-1$
					+ ";singleton:=true\n"); //$NON-NLS-1$
			manifestSB.append("Bundle-Version: 0.0.1.qualifier\n"); //$NON-NLS-1$
			manifestSB.append("Bundle-Vendor: \n"); //$NON-NLS-1$
			// manifestSB
			// .append("Require-Bundle: org.eclipse.gmt.modisco.infra.query.manager\n");
			manifestSB.append("Bundle-RequiredExecutionEnvironment: " //$NON-NLS-1$
					+ ProjectUtils.JAVA_VERSION + "\n"); //$NON-NLS-1$
			manifestSB.append("Bundle-ActivationPolicy: lazy\n"); //$NON-NLS-1$
			manifestSB.append("Bundle-ClassPath: .,bin\n"); //$NON-NLS-1$
			InputStream source = new ByteArrayInputStream(manifestSB.toString().getBytes());
			manifestFile.create(source, true, new NullProgressMonitor());
		}
	}

	public static void addPdeNature(final IProject project) throws CoreException {
		IProjectDescription description = project.getDescription();
		String[] natures = description.getNatureIds();
		String[] newNatures = new String[natures.length + 1];
		System.arraycopy(natures, 0, newNatures, 0, natures.length);
		newNatures[natures.length] = "org.eclipse.pde.PluginNature"; //$NON-NLS-1$
		description.setNatureIds(newNatures);
		project.setDescription(description, new NullProgressMonitor());
	}

	public static void addPdeBuilder(final IProject project) throws CoreException {
		IProjectDescription projectDescription = project.getDescription();
		ICommand[] oldBuildSpec = project.getDescription().getBuildSpec();
		ICommand[] newBuildSpec = new ICommand[oldBuildSpec.length + 2];
		System.arraycopy(oldBuildSpec, 0, newBuildSpec, 0, oldBuildSpec.length);
		ICommand command1 = project.getDescription().newCommand();
		command1.setBuilderName("org.eclipse.pde.ManifestBuilder"); //$NON-NLS-1$
		ICommand command2 = project.getDescription().newCommand();
		command2.setBuilderName("org.eclipse.pde.SchemaBuilder"); //$NON-NLS-1$
		newBuildSpec[oldBuildSpec.length] = command1;
		newBuildSpec[oldBuildSpec.length + 1] = command2;
		projectDescription.setBuildSpec(newBuildSpec);
		project.setDescription(projectDescription, new NullProgressMonitor());
	}

	/**
	 * @author Grgoire Dup (Mia-Software) - classpath entries modification
	 */
	public static void configureAsJavaProject(final IProject project, final IProgressMonitor monitor)
			throws CoreException {
		addNature(project, monitor, JavaCore.NATURE_ID);
		IJavaProject javaProject = JavaCore.create(project);
		// Set output folder
		IPath path = project.getFullPath().append("bin"); //$NON-NLS-1$
		javaProject.setOutputLocation(path, null);
		List<IClasspathEntry> classpathEntries = new ArrayList<IClasspathEntry>();
		// Set source folder
		IFolder sourceFolder = project.getFolder("src"); //$NON-NLS-1$
		sourceFolder.create(false, true, monitor);
		classpathEntries
				.add(JavaCore.newSourceEntry(javaProject.getPath().append(new Path("src")))); //$NON-NLS-1$
		// add the jre api to the classpath
		classpathEntries.add(JavaCore.newContainerEntry(new Path(JavaRuntime.JRE_CONTAINER
				+ "/org.eclipse.jdt.internal.debug.ui.launcher.StandardVMType/" //$NON-NLS-1$
				+ ProjectUtils.JAVA_VERSION)));
		javaProject.setRawClasspath(classpathEntries.toArray(new IClasspathEntry[0]), monitor);
	}

	public static void configureAsPluginProject(final IProject project) throws CoreException {
		// TODO PDE Operations would be useful here but they are internal in PDE
		addPdeNature(project);
		addPdeBuilder(project);
		addPdeClassPath(project);
		createManifest(project);
	}

	public static void addNature(final IProject project, final IProgressMonitor monitor,
			final String natureId) throws CoreException {
		IProjectDescription description = project.getDescription();
		String[] natures = description.getNatureIds();
		String[] newNatures = new String[natures.length + 1];
		System.arraycopy(natures, 0, newNatures, 0, natures.length);
		newNatures[natures.length] = natureId;
		description.setNatureIds(newNatures);
		project.setDescription(description, monitor);
	}

	/**
	 * @author Grgore Dup (Mia-Software) - initial implementation
	 */
	public static void createBuildProperties(final IProject project) throws CoreException {
		IFile manifestFile = project.getFile("build.properties"); //$NON-NLS-1$
		if (!manifestFile.exists()) {
			StringBuffer manifestSB = new StringBuffer();
			manifestSB.append("source.. = src/\n"); //$NON-NLS-1$
			manifestSB.append("output.. = bin/\n"); //$NON-NLS-1$
			manifestSB.append("bin.includes = META-INF/,\\\n"); //$NON-NLS-1$
			manifestSB.append("               .\n"); //$NON-NLS-1$
			InputStream source = new ByteArrayInputStream(manifestSB.toString().getBytes());
			manifestFile.create(source, true, new NullProgressMonitor());
		}
	}

	public static void create(final IProject project, final IProgressMonitor monitor)
			throws CoreException {
		IPath projectLocation = project.getLocation();
		monitor.beginTask(Messages.ProjectUtils_0, IProgressMonitor.UNKNOWN);
		if (!project.exists()) {
			IProjectDescription description = project.getWorkspace().newProjectDescription(
					project.getName());
			if (!Platform.getLocation().equals(projectLocation)) {
				description.setLocation(projectLocation);
			}
			project.create(monitor);
			project.open(monitor);
			monitor.subTask(Messages.ProjectUtils_1);
			ProjectUtils.addNature(project, monitor, MoDiscoProject.NATURE_ID);
			ProjectUtils.configureAsJavaProject(project, monitor);
			monitor.subTask(Messages.ProjectUtils_2);
			ProjectUtils.configureAsPluginProject(project);
			ProjectUtils.createBuildProperties(project);
		} else {
			MoDiscoLogger.logWarning("Project creation aborted : the project already exists", //$NON-NLS-1$
					CommonModiscoActivator.getDefault());
		}
		monitor.done();
	}

	public static IProject createTestProject(final String projectName,
			final Bundle bundleContainingResources, final String resourceFolder) throws Exception {
		IWorkspace ws = ResourcesPlugin.getWorkspace();
		ws.getRoot().refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
		IProject projectToCreate = ws.getRoot().getProject(projectName);
		if (projectToCreate.exists()) {
			projectToCreate.delete(true, true, new NullProgressMonitor());
		}
		if (!projectToCreate.exists()) {
			ProjectUtils.create(projectToCreate, new NullProgressMonitor());
		} else {
			throw new Exception(projectName + " project already exists"); //$NON-NLS-1$
		}
		String manifestLocation = "resources/" + resourceFolder //$NON-NLS-1$
				+ "/MANIFEST.MF_" + projectName; //$NON-NLS-1$
		if (bundleContainingResources.getResource(manifestLocation) != null) {
			FileUtils.copyFileFromBundle(manifestLocation, projectToCreate,
					"META-INF/MANIFEST.MF", bundleContainingResources); //$NON-NLS-1$
		}
		return projectToCreate;
	}

	/**
	 * @param folder
	 * @throws CoreException
	 * @throws CoreException
	 * @throws InterruptedException
	 * @throws OperationCanceledException
	 */
	public static void refresh(final IProject project) throws CoreException, InterruptedException {
		project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
		Job.getJobManager().join(ResourcesPlugin.FAMILY_MANUAL_BUILD, null);
		Job.getJobManager().join(ResourcesPlugin.FAMILY_AUTO_BUILD, null);
		Job.getJobManager().join(ResourcesPlugin.FAMILY_MANUAL_REFRESH, null);
		Job.getJobManager().join(ResourcesPlugin.FAMILY_MANUAL_BUILD, null);
	}
}
