/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.custom.core;

import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.gmt.modisco.common.core.builder.IModiscoBuilder;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;

/**
 * A builder that feeds and updates the {@link CustomizationsCatalog} with the
 * customization files it finds
 */
public class CustomizationsBuilder implements IModiscoBuilder {

	protected static final String FILE_EXTENSION = "uiCustom"; //$NON-NLS-1$

	public IProject[] build(final IncrementalProjectBuilder builder, final int kind,
			final Map<?, ?> args, final IProgressMonitor monitor) throws CoreException {
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
			fullBuild(builder);
		} else {
			final IProject project = builder.getProject();
			final IResourceDelta delta = builder.getDelta(project);
			if (delta == null) {
				fullBuild(builder);
			} else {
				incrementalBuild(builder, delta);
			}
		}
		return null;
	}

	private void incrementalBuild(final IncrementalProjectBuilder builder,
			final IResourceDelta resourceDelta) {
		try {
			resourceDelta.accept(new IResourceDeltaVisitor() {
				public boolean visit(final IResourceDelta delta) {

					final IResource resource = delta.getResource();
					final String fileExtension = resource.getFileExtension();
					if (resource instanceof IFile && fileExtension != null
							&& fileExtension.equals(CustomizationsBuilder.FILE_EXTENSION)) {
						final IFile file = (IFile) resource;

						switch (delta.getKind()) {
						case IResourceDelta.ADDED:
						case IResourceDelta.ADDED_PHANTOM:
							CustomizationsCatalog.getInstance().addCustomization(file);
							break;
						case IResourceDelta.REMOVED:
						case IResourceDelta.REMOVED_PHANTOM:
							CustomizationsCatalog.getInstance().removeCustomization(file);
							break;
						case IResourceDelta.CHANGED:
							CustomizationsCatalog.getInstance().customizationChanged(file);
							break;
						default:
						}
					}
					return true;
				}
			});
		} catch (final CoreException e) {
			MoDiscoLogger.logError(e, "Failed to build: " + builder.getProject(), Activator //$NON-NLS-1$
					.getDefault());
		}
	}

	private void fullBuild(final IncrementalProjectBuilder builder) {
		final IProject project = builder.getProject();
		handleProject(project, false);
	}

	public void clean(final IncrementalProjectBuilder builder, final IProgressMonitor monitor)
			throws CoreException {
		final IProject project = builder.getProject();
		handleProject(project, true);
	}

	private void handleProject(final IProject project, final boolean clean) {
		try {
			project.accept(new IResourceVisitor() {
				public boolean visit(final IResource resource) throws CoreException {
					if (resource instanceof IFile) {
						final IFile file = (IFile) resource;
						final String fileExtension = file.getFileExtension();
						if (fileExtension != null
								&& fileExtension.equals(CustomizationsBuilder.FILE_EXTENSION)) {
							if (clean) {
								CustomizationsCatalog.getInstance().removeCustomization(file);
							} else {
								CustomizationsCatalog.getInstance().addCustomization(file);
							}
						}
					}
					return true;
				}
			});
		} catch (final CoreException e) {
			final String operation;
			if (clean) {
				operation = "clean"; //$NON-NLS-1$
			} else {
				operation = "build"; //$NON-NLS-1$
			}
			MoDiscoLogger.logError(e, "Failed to " + operation + " " + project, Activator //$NON-NLS-1$ //$NON-NLS-2$
					.getDefault());
		}
	}
}
