/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.custom.core;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.EventListener;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.gmt.modisco.common.core.resource.MoDiscoResourceSet;
import org.eclipse.gmt.modisco.common.core.utils.FileUtils;
import org.eclipse.gmt.modisco.common.core.utils.ValidationUtils;
import org.eclipse.gmt.modisco.infra.browser.custom.MetamodelView;

/**
 * A catalog of customization files found in the Workspace. It is updated by the
 * {@link CustomizationsBuilder}, which is responsible for adding and removing
 * customization files in response to changes in the Workspace.
 */
public class CustomizationsCatalog {

	public static final String FILE_EXTENSION = "uiCustom"; //$NON-NLS-1$
	private static final String PERSISTENCE_FILE_NAME = "uiCustomCatalog"; //$NON-NLS-1$
	private static CustomizationsCatalog instance = null;

	/** Customizations found in the Workspace */
	private final HashSet<IFile> workspaceCustomizations = new HashSet<IFile>();

	public static CustomizationsCatalog getInstance() {
		if (CustomizationsCatalog.instance == null) {
			CustomizationsCatalog.instance = new CustomizationsCatalog();
		}
		return CustomizationsCatalog.instance;
	}

	public CustomizationsCatalog() {
		load();

		// validation
		addChangeListener(new CustomizationChangeListener() {
			public void removed(final IFile customizationFile) {
			}

			public void added(final MetamodelView addedCustomization, final IFile file) {
				try {
					ValidationUtils.validate(addedCustomization.eResource(), file);
				} catch (final CoreException e) {
					Activator.logException(e);
				}
			}

			public void changed(final MetamodelView customization, final IFile file) {
				try {
					ValidationUtils.validate(customization.eResource(), file);
				} catch (final CoreException e) {
					Activator.logException(e);
				}
			}
		});
	}

	/** Loads and returns a list of customizations found in the Workspace. */
	public List<MetamodelView> getWorkspaceCustomizations() {
		final List<MetamodelView> metamodelViews = new ArrayList<MetamodelView>();
		for (final IFile file : this.workspaceCustomizations) {
			final java.net.URI locationURI = file.getLocationURI();
			final MetamodelView customization = loadCustomization(locationURI);
			if (customization != null) {
				customization.setLocation(locationURI.toString());
				metamodelViews.add(customization);
			}
		}
		return metamodelViews;
	}

	/**
	 * Loads and returns a list of customizations registered through the
	 * "registry" extension point.
	 */
	public List<MetamodelView> getRegistryCustomizations() {
		final List<MetamodelView> metamodelViews = new ArrayList<MetamodelView>();
		final List<URL> uiCustomURLS = CustomizationsRegistry.getInstance().getUiCustomURLS();
		for (final URL url : uiCustomURLS) {
			try {
				final MetamodelView customization = loadCustomization(url.toURI());
				if (customization != null) {
					metamodelViews.add(customization);
					customization.setLocation(url.toString());
				}
			} catch (final Exception e) {
				Activator.logException(e);
			}
		}
		return metamodelViews;
	}
	
	/**
	 * Loads and returns a list of default customizations registered through the
	 * "registry" extension point.
	 */
	public List<MetamodelView> getRegistryDefaultCustomizations() {
		final List<MetamodelView> metamodelViews = new ArrayList<MetamodelView>();
		final List<URL> uiCustomURLS = CustomizationsRegistry.getInstance().getUiCustomDefaultURLS();
		for (final URL url : uiCustomURLS) {
			try {
				final MetamodelView customization = loadCustomization(url.toURI());
				if (customization != null) {
					metamodelViews.add(customization);
					customization.setLocation(url.toString());
				}
			} catch (final Exception e) {
				Activator.logException(e);
			}
		}
		return metamodelViews;
	}

	public List<MetamodelView> getAllCustomizations() {
		final List<MetamodelView> metamodelViews = new ArrayList<MetamodelView>();
		metamodelViews.addAll(getWorkspaceCustomizations());
		metamodelViews.addAll(getRegistryCustomizations());
		return metamodelViews;
	}

	private MetamodelView loadCustomization(final java.net.URI locationURI) {
		try {
			final ResourceSet resourceSet = MoDiscoResourceSet
					.getResourceSetSingleton();
			resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put(
					CustomizationsCatalog.FILE_EXTENSION, new XMIResourceFactoryImpl());
			final URI uri = URI.createURI(locationURI.toString());
			final Resource resource = resourceSet.createResource(uri);
			resource.load(null);
			// if (Tools.validate(resource, file)) {
			final MetamodelView metamodelView = (MetamodelView) resource.getContents().get(0);
			return metamodelView;
			// } else {
			// Activator.logError("Resource '" + locationURI +
			// "' wasn't loaded because"
			// + "it contains errors (see Problems view)");
			// }
		} catch (final Exception e) {
			Activator.logException("Error loading resource " + locationURI, e); //$NON-NLS-1$
		}
		return null;
	}

	/** Adds a customization to the catalog */
	public void addCustomization(final IFile customizationFile) {
		this.workspaceCustomizations.add(customizationFile);
		customizationAdded(customizationFile);
		save();
	}

	/** Persists the list of customizations to a file in the bundle state area */
	private void save() {
		final File file = getPersistenceFile();
		try {
			final FileWriter fileWriter = new FileWriter(file);
			final Iterator<IFile> iterator = this.workspaceCustomizations.iterator();
			while (iterator.hasNext()) {
				final IFile customizationFile = iterator.next();
				fileWriter.write(customizationFile.getFullPath().toString() + "\n"); //$NON-NLS-1$
			}
			fileWriter.close();
		} catch (final IOException e) {
			Activator.logException(e);
		}
	}

	/**
	 * Loads a persisted list of customizations from a file in the bundle state
	 * area
	 */
	private void load() {
		final IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();

		final File persistenceFile = getPersistenceFile();
		if (persistenceFile.isFile()) {
			try {
				final String contents = FileUtils.getContents(persistenceFile);
				final String[] files = contents.split("\\n"); //$NON-NLS-1$
				for (final String file : files) {
					final IResource resource = workspaceRoot.findMember(new Path(file));
					if (resource instanceof IFile) {
						final IFile customizationFile = (IFile) resource;
						this.workspaceCustomizations.add(customizationFile);
					}
				}
			} catch (final IOException e) {
				Activator.logException(e);
			}
		}
	}

	private File getPersistenceFile() {
		return Activator.getDefault().getStateLocation().append(
				CustomizationsCatalog.PERSISTENCE_FILE_NAME).toFile();
	}

	/** Removes a customization from the catalog */
	public void removeCustomization(final IFile customizationFile) {
		this.workspaceCustomizations.remove(customizationFile);
		customizationRemoved(customizationFile);
		save();
	}

	/** Notifies a change to the customization */
	public void customizationChanged(final IFile customizationFile) {
		final MetamodelView customization = loadCustomization(customizationFile.getLocationURI());
		// copy the list to permit removing an element while iterating
		final List<CustomizationChangeListener> tempListeners = new ArrayList<CustomizationChangeListener>(
				this.changeListeners);
		for (final CustomizationChangeListener changeListener : tempListeners) {
			changeListener.changed(customization, customizationFile);
		}
	}

	private void customizationRemoved(final IFile customizationFile) {
		if (customizationFile != null) {
			// copy the list to permit removing an element while iterating
			final List<CustomizationChangeListener> tempListeners = new ArrayList<CustomizationChangeListener>(
					this.changeListeners);
			for (final CustomizationChangeListener changeListener : tempListeners) {
				changeListener.removed(customizationFile);
			}
		}
	}

	private void customizationAdded(final IFile customizationFile) {
		final MetamodelView customization = loadCustomization(customizationFile.getLocationURI());
		if (customization != null) {
			// copy the list to permit removing an element while iterating
			final List<CustomizationChangeListener> tempListeners = new ArrayList<CustomizationChangeListener>(
					this.changeListeners);
			for (final CustomizationChangeListener changeListener : tempListeners) {
				changeListener.added(customization, customizationFile);
			}
		}
	}

	/** A listener for change notifications about customizations */
	public interface CustomizationChangeListener extends EventListener {
		void changed(MetamodelView customization, IFile file);

		void added(MetamodelView addedCustomization, IFile file);

		void removed(IFile customizationFile);
	}

	private final List<CustomizationsCatalog.CustomizationChangeListener> changeListeners = new ArrayList<CustomizationsCatalog.CustomizationChangeListener>();

	public void addChangeListener(final CustomizationChangeListener listener) {
		if (!this.changeListeners.contains(listener)) {
			this.changeListeners.add(listener);
		}
	}

	public void removeChangeListener(final CustomizationChangeListener listener) {
		this.changeListeners.remove(listener);
	}
}
