/*******************************************************************************
 * Copyright (c) 2008, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.infra.browser.editors;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EventListener;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.gmt.modisco.infra.browser.MoDiscoBrowserPlugin;
import org.eclipse.gmt.modisco.infra.browser.core.InstancesForMetaclasses;
import org.eclipse.gmt.modisco.infra.browser.customization.CustomizationEngine;
import org.eclipse.gmt.modisco.infra.browser.queries.SelectedQueriesManager;
import org.eclipse.gmt.modisco.infra.browser.util.EMFUtil;
import org.eclipse.gmt.modisco.infra.facet.Facet;
import org.eclipse.gmt.modisco.infra.facet.FacetSet;
import org.eclipse.gmt.modisco.infra.facet.core.FacetContext;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Display;

/** The configuration of a model editor, with access to the underlying data */
public class BrowserConfiguration {

	/** Property name for the fontSizeDelta property */
	public static final String FONT_SIZE_DELTA_PROP = "fontSizeDelta"; //$NON-NLS-1$
	public static final int NOT_YET_ASKED = -2;

	public enum MetaclassesSortMode {
		ByName, ByCount
	}

	private static final String SHOW_EMPTY_LINKS_SETTING = "showEmptyLinks"; //$NON-NLS-1$
	private static final String SHOW_DERIVED_LINKS_SETTING = "showDerivedLinks"; //$NON-NLS-1$
	private static final String SORT_INSTANCES_SETTING = "showInstances"; //$NON-NLS-1$
	private static final String SORT_LINKS_SETTING = "sortLinks"; //$NON-NLS-1$
	private static final String SORT_LINKS_BY_TYPE_SETTING = "sortLinksByType"; //$NON-NLS-1$
	private static final String SHOW_FULL_QUALIFIED_NAMES_SETTING = "showFullQualifiedNames"; //$NON-NLS-1$
	private static final String SHOW_MULTIPLICITY_SETTING = "showMultiplicity"; //$NON-NLS-1$
	private static final String SHOW_OPPOSITE_LINKS_SETTING = "showOppositeLinks"; //$NON-NLS-1$
	private static final String SHOW_CONTAINER_SETTING = "showContainer"; //$NON-NLS-1$
	private static final String METACLASSES_SORT_MODE = "metaclassesSortMode"; //$NON-NLS-1$
	private static final String SHOW_METACLASSES_FULL_QUALIFIED_NAMES_SETTING = "showMetaclassesFullQualifiedNames"; //$NON-NLS-1$
	private static final String SHOW_EMPTY_METACLASSES = "showEmptyMetaclasses"; //$NON-NLS-1$
	private static final String GROUP_BY_PACKAGE = "groupByPackage"; //$NON-NLS-1$
	private static final String DISPLAY_INSTANCES_OF_SUBCLASSES = "displayInstancesOfSubclasses"; //$NON-NLS-1$
	private static final String SHOW_DERIVATION_TREE = "showDerivationTree"; //$NON-NLS-1$
	private static final String SHOW_ATTRIBUTES = "showAttributes"; //$NON-NLS-1$
	private static final String SHOW_EMPTY_ATTRIBUTES = "showEmptyAttributes"; //$NON-NLS-1$
	private static final String SHOW_ORDERING = "showOrdering"; //$NON-NLS-1$
	private static final String FONT_SIZE_DELTA = "fontSizeDelta"; //$NON-NLS-1$
	private static final String SHOW_TYPE_OF_LINKS = "showTypeOfLinks"; //$NON-NLS-1$
	private static final String SHOW_ELEMENT_IDS = "showElementIDs"; //$NON-NLS-1$

	/** A listener for change notifications */
	public interface ChangeListener extends EventListener {
		void changed();
	}

	/** show the type declared for the link element **/
	private boolean showTypeOfLinks = false;
	/** Show empty links? */
	private boolean showEmptyLinks = false;
	/** Sort model elements by name in the tree? */
	private boolean sortInstances = false;
	/** Sort links by name in the tree? */
	private boolean sortLinks = false;
	/** Sort links by type in the tree? */
	private boolean sortLinksByType = false;
	/** Show derived links (links computed from other links) */
	private boolean showDerivedLinks = false;
	/** Show full qualified names or short names? */
	private boolean showFullQualifiedNames = false;
	/** Show full qualified names or short names of metaclasses? */
	private boolean showMetaclassesFullQualifiedNames = false;
	/** Show multiplicity? */
	private boolean showMultiplicity = false;
	/** Show opposite links? */
	private boolean showOppositeLinks = false;
	/** Show EMF eContainer as a virtual link? */
	private boolean showContainer = false;
	/** Show metaclasses which don't have any instance in the model? */
	private boolean showEmptyMetaclasses = false;
	/** Whether to display metaclasses grouped by package */
	private boolean groupByPackage = false;
	/**
	 * Whether to display instances on a given metaclass when the element has
	 * the type of a subclass
	 */
	private boolean displayInstancesOfSubclasses;
	/** Whether to show metaclasses as a derivation tree */
	private boolean showDerivationTree;
	/** How to sort metaclasses (in the metaclass pane) */
	private MetaclassesSortMode fMetaclassesSortMode = BrowserConfiguration.MetaclassesSortMode.ByName;
	/** Whether to show attributes as links */
	private boolean showAttributes;
	/** Whether to show empty attributes (null or empty list) */
	private boolean showEmptyAttributes;
	/** Whether to show ordering information */
	private boolean showOrdering;
	/** Whether to show the EMF URI for each element */
	private boolean showElementIDs;
	/**
	 * The size difference in points between the default font size and the
	 * user's preferred size
	 */
	private int fontSizeDelta = 0;

	/**
	 * A font derived from the JFace default font, with a custom size defined by
	 * the user
	 */
	private Font customFont = null;
	/**
	 * An italic font derived from the JFace default font, with a custom size
	 * defined by the user
	 */
	private Font customItalicFont = null;

	/** All the classes in the resource set used by the editor */
	private Collection<EClass> allClasses;

	private ResourceSet resourceSet;
	/** All the model elements by metaclass */
	private InstancesForMetaclasses instancesForMetaclasses;
	private ComposedAdapterFactory adapterFactoryWithRegistry;
	private EcoreBrowser editor;

	private final List<PropertyChangeListener> listeners = new ArrayList<PropertyChangeListener>();
	private CustomizationEngine customizationEngine;
	private SelectedQueriesManager selectedQueriesManager;

	/**
	 * A counter for modifications to the browser configuration. Allows clients
	 * of this class to recompute their cached data when the counter has changed
	 */
	private int modificationCount = 0;

	/**
	 * The depth of referenced resources the user wants to load (
	 * <code>-1</code> means infinite, and <code>-2</code> means not yet asked)
	 */
	private int referencedResourcesLoadingDepth = BrowserConfiguration.NOT_YET_ASKED;

	public BrowserConfiguration(final EcoreBrowser editor) {
		this();
		this.editor = editor;
	}

	public BrowserConfiguration() {
		createCustomizationEngine();
		this.adapterFactoryWithRegistry = new ComposedAdapterFactory();
	}

	public EcoreBrowser getEditor() {
		return this.editor;
	}

	public boolean isShowEmptyLinks() {
		return this.showEmptyLinks;
	}

	public void setShowEmptyLinks(final boolean showEmptyLinks) {
		this.showEmptyLinks = showEmptyLinks;
		this.modificationCount++;
	}

	public boolean isSortInstances() {
		return this.sortInstances;
	}

	public void setSortInstances(final boolean sortInstances) {
		this.sortInstances = sortInstances;
		this.modificationCount++;
	}

	public boolean isShowDerivedLinks() {
		return this.showDerivedLinks;
	}

	public void setShowDerivedLinks(final boolean showDerivedLinks) {
		this.showDerivedLinks = showDerivedLinks;
		this.modificationCount++;
	}

	public boolean isShowFullQualifiedNames() {
		return this.showFullQualifiedNames;
	}

	public void setShowFullQualifiedNames(final boolean showFullQualifiedNames) {
		this.showFullQualifiedNames = showFullQualifiedNames;
		this.modificationCount++;
	}

	public boolean isShowMultiplicity() {
		return this.showMultiplicity;
	}

	public void setShowMultiplicity(final boolean showMultiplicity) {
		this.showMultiplicity = showMultiplicity;
		this.modificationCount++;
	}

	public boolean isShowOppositeLinks() {
		return this.showOppositeLinks;
	}

	public void setShowOppositeLinks(final boolean showOppositeLinks) {
		this.showOppositeLinks = showOppositeLinks;
		this.modificationCount++;
	}

	public boolean isShowContainer() {
		return this.showContainer;
	}

	public void setShowContainer(final boolean showContainer) {
		this.showContainer = showContainer;
		this.modificationCount++;
	}

	public MetaclassesSortMode getMetaclassesSortMode() {
		return this.fMetaclassesSortMode;
	}

	public void setMetaclassesSortMode(final MetaclassesSortMode mode) {
		this.fMetaclassesSortMode = mode;
		this.modificationCount++;
	}

	public boolean isShowMetaclassesFullQualifiedNames() {
		return this.showMetaclassesFullQualifiedNames;
	}

	public void setShowMetaclassesFullQualifiedNames(final boolean value) {
		this.showMetaclassesFullQualifiedNames = value;
		this.modificationCount++;
	}

	public void setSortLinks(final boolean value) {
		this.sortLinks = value;
		this.modificationCount++;
	}

	public boolean isSortLinks() {
		return this.sortLinks;
	}

	public void setSortLinksByType(final boolean value) {
		this.sortLinksByType = value;
		this.modificationCount++;
	}

	public boolean isSortLinksByType() {
		return this.sortLinksByType;
	}

	public void setShowEmptyMetaclasses(final boolean value) {
		this.showEmptyMetaclasses = value;
		this.modificationCount++;
	}

	public boolean isShowEmptyMetaclasses() {
		return this.showEmptyMetaclasses;
	}

	public void setGroupByPackage(final boolean value) {
		this.groupByPackage = value;
		this.modificationCount++;
	}

	public boolean isGroupByPackage() {
		return this.groupByPackage;
	}

	public void setDisplayInstancesOfSubclasses(final boolean value) {
		this.displayInstancesOfSubclasses = value;
		this.modificationCount++;
	}

	public boolean isDisplayInstancesOfSubclasses() {
		return this.displayInstancesOfSubclasses;
	}

	public void setShowDerivationTree(final boolean value) {
		this.showDerivationTree = value;
		this.modificationCount++;
	}

	public boolean isShowDerivationTree() {
		return this.showDerivationTree;
	}

	public void setShowAttributes(final boolean value) {
		this.showAttributes = value;
		this.modificationCount++;
	}

	public boolean isShowAttributes() {
		return this.showAttributes;
	}

	public void setShowEmptyAttributes(final boolean value) {
		this.showEmptyAttributes = value;
		this.modificationCount++;
	}

	public boolean isShowEmptyAttributes() {
		return this.showEmptyAttributes;
	}

	public void setShowOrdering(final boolean value) {
		this.showOrdering = value;
		this.modificationCount++;
	}

	public boolean isShowOrdering() {
		return this.showOrdering;
	}

	public void setShowTypeOfLinks(final boolean value) {
		this.showTypeOfLinks = value;
		this.modificationCount++;
	}

	public boolean isShowTypeOfLinks() {
		return this.showTypeOfLinks;
	}

	public void setShowElementIDs(final boolean showElementIDs) {
		this.showElementIDs = showElementIDs;
		this.modificationCount++;
	}

	public boolean isShowElementIDs() {
		return this.showElementIDs;
	}

	public void setFontSizeDelta(final int value) {
		firePropertyChanged(BrowserConfiguration.FONT_SIZE_DELTA_PROP, this.fontSizeDelta, value);
		this.fontSizeDelta = value;
		this.modificationCount++;
	}

	public int getFontSizeDelta() {
		return this.fontSizeDelta;
	}

	public void setCustomFont(final Font customFont) {
		if (this.customFont != null && this.customFont != JFaceResources.getDialogFont()
				&& this.customFont != customFont) {
			this.customFont.dispose();
		}
		this.customFont = customFont;
		this.modificationCount++;
	}

	public Font getCustomFont() {
		return this.customFont;
	}

	public void setCustomItalicFont(final Font customItalicFont) {
		if (this.customItalicFont != null
				&& this.customItalicFont != JFaceResources.getDialogFont()
				&& this.customItalicFont != customItalicFont) {
			this.customItalicFont.dispose();
		}
		this.customItalicFont = customItalicFont;
		this.modificationCount++;
	}

	public Font getCustomItalicFont() {
		return this.customItalicFont;
	}

	public void setResourceSet(final ResourceSet resourceSet) {
		this.resourceSet = resourceSet;
	}

	public ResourceSet getResourceSet() {
		return this.resourceSet;
	}

	/** Returns all the classes in the resource set used by the editor */
	public Collection<EClass> getAllClasses() {
		final List<EClass> classes = new ArrayList<EClass>();
		classes.addAll(this.facetContext.getMetaClasses());
		classes.addAll(this.allClasses);
		return classes;
	}

	public void initInstancesForMetaclasses() {
		this.instancesForMetaclasses = new InstancesForMetaclasses(this);
	}

	public InstancesForMetaclasses getInstancesForMetaclasses() {
		return this.instancesForMetaclasses;
	}

	public void setAdapterFactoryWithRegistry(
			final ComposedAdapterFactory adapterFactoryWithRegistry) {
		this.adapterFactoryWithRegistry = adapterFactoryWithRegistry;
		this.modificationCount++;
	}

	public ComposedAdapterFactory getAdapterFactoryWithRegistry() {
		return this.adapterFactoryWithRegistry;
	}

	public void createCustomizationEngine() {
		this.customizationEngine = new CustomizationEngine();
		this.modificationCount++;
		this.customizationEngine.addChangeListener(new ChangeListener() {
			public void changed() {
				BrowserConfiguration.this.modificationCount++;
			}
		});
	}

	public CustomizationEngine getCustomizationEngine() {
		return this.customizationEngine;
	}

	/** Load user settings */
	public void load() {
		final IDialogSettings dialogSettings = MoDiscoBrowserPlugin.getPlugin().getDialogSettings();

		this.showEmptyLinks = getPreference(BrowserConfiguration.SHOW_EMPTY_LINKS_SETTING, true,
				dialogSettings);
		this.showDerivedLinks = getPreference(BrowserConfiguration.SHOW_DERIVED_LINKS_SETTING,
				true, dialogSettings);
		this.sortInstances = getPreference(BrowserConfiguration.SORT_INSTANCES_SETTING, false,
				dialogSettings);
		this.sortLinks = getPreference(BrowserConfiguration.SORT_LINKS_SETTING, false,
				dialogSettings);
		this.sortLinksByType = getPreference(BrowserConfiguration.SORT_LINKS_BY_TYPE_SETTING,
				false, dialogSettings);
		this.showFullQualifiedNames = getPreference(
				BrowserConfiguration.SHOW_FULL_QUALIFIED_NAMES_SETTING, false, dialogSettings);
		this.showMultiplicity = getPreference(BrowserConfiguration.SHOW_MULTIPLICITY_SETTING,
				false, dialogSettings);
		this.showOppositeLinks = getPreference(BrowserConfiguration.SHOW_OPPOSITE_LINKS_SETTING,
				false, dialogSettings);
		this.showContainer = getPreference(BrowserConfiguration.SHOW_CONTAINER_SETTING, true,
				dialogSettings);
		this.showMetaclassesFullQualifiedNames = getPreference(
				BrowserConfiguration.SHOW_METACLASSES_FULL_QUALIFIED_NAMES_SETTING, false,
				dialogSettings);
		this.showEmptyMetaclasses = getPreference(BrowserConfiguration.SHOW_EMPTY_METACLASSES,
				false, dialogSettings);
		this.groupByPackage = getPreference(BrowserConfiguration.GROUP_BY_PACKAGE, false,
				dialogSettings);
		this.displayInstancesOfSubclasses = getPreference(
				BrowserConfiguration.DISPLAY_INSTANCES_OF_SUBCLASSES, false, dialogSettings);
		this.showDerivationTree = getPreference(BrowserConfiguration.SHOW_DERIVATION_TREE, false,
				dialogSettings);
		this.showAttributes = getPreference(BrowserConfiguration.SHOW_ATTRIBUTES, true,
				dialogSettings);
		this.showEmptyAttributes = getPreference(BrowserConfiguration.SHOW_EMPTY_ATTRIBUTES, true,
				dialogSettings);
		this.showOrdering = getPreference(BrowserConfiguration.SHOW_ORDERING, false, dialogSettings);
		this.fontSizeDelta = getPreference(BrowserConfiguration.FONT_SIZE_DELTA, 0, dialogSettings);
		this.showTypeOfLinks = getPreference(BrowserConfiguration.SHOW_TYPE_OF_LINKS, false,
				dialogSettings);
		this.showElementIDs = getPreference(BrowserConfiguration.SHOW_ELEMENT_IDS, false,
				dialogSettings);

		final String metaclassesSortMode = dialogSettings
				.get(BrowserConfiguration.METACLASSES_SORT_MODE);
		if (metaclassesSortMode != null) {
			this.fMetaclassesSortMode = BrowserConfiguration.MetaclassesSortMode
					.valueOf(metaclassesSortMode);
		}
		this.modificationCount++;
	}

	private boolean getPreference(final String settingKey, final boolean defaultValue,
			final IDialogSettings dialogSettings) {
		final String value = dialogSettings.get(settingKey);
		if (value == null) {
			return defaultValue;
		}
		return Boolean.valueOf(value);
	}

	private int getPreference(final String settingKey, final int defaultValue,
			final IDialogSettings dialogSettings) {
		final String value = dialogSettings.get(settingKey);
		if (value == null) {
			return defaultValue;
		}
		try {
			return Integer.parseInt(value);
		} catch (final NumberFormatException e) {
			return defaultValue;
		}
	}

	/** Save user settings */
	public void save() {
		final IDialogSettings dialogSettings = MoDiscoBrowserPlugin.getPlugin().getDialogSettings();
		dialogSettings.put(BrowserConfiguration.SHOW_EMPTY_LINKS_SETTING, this.showEmptyLinks);
		dialogSettings.put(BrowserConfiguration.SHOW_DERIVED_LINKS_SETTING, this.showDerivedLinks);
		dialogSettings.put(BrowserConfiguration.SORT_INSTANCES_SETTING, this.sortInstances);
		dialogSettings.put(BrowserConfiguration.SORT_LINKS_SETTING, this.sortLinks);
		dialogSettings.put(BrowserConfiguration.SORT_LINKS_BY_TYPE_SETTING, this.sortLinksByType);
		dialogSettings.put(BrowserConfiguration.SHOW_FULL_QUALIFIED_NAMES_SETTING,
				this.showFullQualifiedNames);
		dialogSettings.put(BrowserConfiguration.SHOW_MULTIPLICITY_SETTING, this.showMultiplicity);
		dialogSettings
				.put(BrowserConfiguration.SHOW_OPPOSITE_LINKS_SETTING, this.showOppositeLinks);
		dialogSettings.put(BrowserConfiguration.SHOW_CONTAINER_SETTING, this.showContainer);
		dialogSettings.put(BrowserConfiguration.METACLASSES_SORT_MODE, this.fMetaclassesSortMode
				.name());
		dialogSettings.put(BrowserConfiguration.SHOW_METACLASSES_FULL_QUALIFIED_NAMES_SETTING,
				this.showMetaclassesFullQualifiedNames);
		dialogSettings.put(BrowserConfiguration.SHOW_EMPTY_METACLASSES, this.showEmptyMetaclasses);
		dialogSettings.put(BrowserConfiguration.GROUP_BY_PACKAGE, this.groupByPackage);
		dialogSettings.put(BrowserConfiguration.DISPLAY_INSTANCES_OF_SUBCLASSES,
				this.displayInstancesOfSubclasses);
		dialogSettings.put(BrowserConfiguration.SHOW_DERIVATION_TREE, this.showDerivationTree);
		dialogSettings.put(BrowserConfiguration.SHOW_ATTRIBUTES, this.showAttributes);
		dialogSettings.put(BrowserConfiguration.SHOW_EMPTY_ATTRIBUTES, this.showEmptyAttributes);
		dialogSettings.put(BrowserConfiguration.SHOW_ORDERING, this.showOrdering);
		dialogSettings.put(BrowserConfiguration.FONT_SIZE_DELTA, this.fontSizeDelta);
		dialogSettings.put(BrowserConfiguration.SHOW_TYPE_OF_LINKS, this.showTypeOfLinks);
		dialogSettings.put(BrowserConfiguration.SHOW_ELEMENT_IDS, this.showElementIDs);
	}

	/** Clear the editor field */
	public void clearEditor() {
		this.editor = null;
	}

	/** @return the metamodel name or <code>null</code> if not found */
	public String getMetamodelName() {
		final String nsURI = getMetamodelURI();
		if (nsURI != null) {
			final String[] parts = nsURI.split("/"); //$NON-NLS-1$
			if (parts.length > 0) {
				return parts[parts.length - 1];
			}
		}
		return null;
	}

	/**
	 * @return the NsURI of the metamodel corresponding to an element in the
	 *         first resource or <code>null</code> if not found
	 */
	public String getMetamodelURI() {
		final EList<Resource> resources = this.resourceSet.getResources();
		if (resources.size() > 0) {
			final Resource firstResource = resources.get(0);
			final EList<EObject> contents = firstResource.getContents();
			if (contents.size() > 0) {
				final EObject firstElement = contents.get(0);
				final EClass eClass = firstElement.eClass();
				if (eClass != null) {
					final EPackage ePackage = eClass.getEPackage();
					if (ePackage != null) {
						return ePackage.getNsURI();
					}
				}
			}
		}

		return null;
	}

	public void addPropertyChangeListener(final PropertyChangeListener listener) {
		this.listeners.add(listener);
	}

	public void removePropertyChangeListener(final PropertyChangeListener listener) {
		this.listeners.remove(listener);
	}

	private void firePropertyChanged(final String propertyName, final int oldValue,
			final int newValue) {
		for (final PropertyChangeListener listener : this.listeners) {
			final PropertyChangeEvent event = new PropertyChangeEvent(this, propertyName, oldValue,
					newValue);
			listener.propertyChange(event);
		}
	}

	/**
	 * @return the counter for modifications to the browser configuration. Use
	 *         it to recompute caches when the counter has changed
	 */
	public int getModCount() {
		return this.modificationCount;
	}

	private final FacetContext facetContext = new FacetContext();
	private Set<Resource> browsedResources;

	public FacetContext getFacetContext() {
		return this.facetContext;
	}

	public void loadFacetSet(final FacetSet facetSet) {
		this.facetContext.addFacetSet(facetSet);
		final List<EClass> metaclasses = new ArrayList<EClass>();
		for (final EClassifier eClassifier : facetSet.getEClassifiers()) {
			if (eClassifier instanceof EClass) {
				metaclasses.add((EClass) eClassifier);
			}
		}
		this.instancesForMetaclasses.addMetaclasses(metaclasses);
		this.modificationCount++;
	}

	public void loadFacet(final Facet facet) {
		this.facetContext.addFacet(facet);
		this.instancesForMetaclasses.addMetaclass(facet);
		this.modificationCount++;
	}

	public Set<Facet> getLoadedFacets() {
		return this.facetContext.getAppliedFacets();
	}

	public void cleanFacets(final boolean reloadModel) {
		this.instancesForMetaclasses.removeMetaClasses(new ArrayList<EClass>(this.facetContext
				.getAppliedFacets()));
		this.facetContext.clean();
		if (reloadModel) {
			this.editor.reloadModel();
		}
		this.modificationCount++;
	}

	public void setSelectedQueriesManager(final SelectedQueriesManager selectedQueriesManager) {
		this.selectedQueriesManager = selectedQueriesManager;
		this.modificationCount++;
		this.selectedQueriesManager.addChangeListener(new ChangeListener() {
			public void changed() {
				BrowserConfiguration.this.modificationCount++;
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						BrowserConfiguration.this.editor.refreshTreeDelayed();
					}
				});

			}
		});
	}

	public SelectedQueriesManager getSelectedQueriesManager() {
		return this.selectedQueriesManager;
	}

	/**
	 * Set the list of resources browsed in the editor, so that the list of
	 * metaclasses can be computed once and for all.
	 */
	public void setBrowsedResources(final Set<Resource> browsedResources) {
		this.browsedResources = browsedResources;
		this.allClasses = EMFUtil.findAllClasses(browsedResources);
		this.modificationCount++;
	}

	public Set<Resource> getBrowsedResources() {
		return Collections.unmodifiableSet(this.browsedResources);
	}

	public void setReferencedResourcesLoadingDepth(final int referencedResourcesLoadingDepth) {
		this.referencedResourcesLoadingDepth = referencedResourcesLoadingDepth;
		this.modificationCount++;
	}

	/**
	 * The depth of referenced resources the user wants to load (
	 * <code>-1</code> means infinite, and <code>-2</code> means not yet asked)
	 */
	public int getReferencedResourcesLoadingDepth() {
		return this.referencedResourcesLoadingDepth;
	}
}
