/*******************************************************************************
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.ui.views;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.common.ui.MoDiscoCommonUIPlugin;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;

/**
 * An Eclipse view that displays a list of available MoDisco facet sets.
 * Clicking on a facet set opens it in the facet set editor.
 */
public abstract class AbstractCatalogView extends ViewPart {

	private static final Map<String, Integer> COLUMN_WIDTHS = new HashMap<String, Integer>();

	private static final int JOB_SCHEDULE_DELAY = 500;
	private Job refreshJob = null;

	private TreeViewer treeViewer;

	public TreeViewer getViewer() {
		return this.treeViewer;
	}

	@Override
	public void createPartControl(final Composite parent) {

		parent.setLayout(new FillLayout());

		this.treeViewer = new TreeViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE
				| SWT.FULL_SELECTION);
		this.treeViewer.getTree().setLinesVisible(true);
		this.treeViewer.getTree().setHeaderVisible(true);

		this.treeViewer.addOpenListener(new IOpenListener() {
			public void open(final OpenEvent event) {
				IStructuredSelection selection = (IStructuredSelection) AbstractCatalogView.this
						.getViewer().getSelection();
				openElement(selection.getFirstElement());
			}
		});

		createColumns();

		this.treeViewer.setContentProvider(getContentProvider());

		getSite().setSelectionProvider(this.treeViewer);
		refresh(false);
	}

	protected abstract void createColumns();

	protected abstract IContentProvider getContentProvider();

	protected abstract void openElement(Object element);

	protected void createColumn(final String columnText, final String uniqueID,
			final int defaultWidth, final ColumnLabelProvider columnLabelProvider) {
		final TreeViewerColumn viewerColumn = new TreeViewerColumn(this.treeViewer, SWT.NONE);
		viewerColumn.getColumn().setText(columnText);

		Integer width = AbstractCatalogView.COLUMN_WIDTHS.get(uniqueID);
		if (width != null) {
			viewerColumn.getColumn().setWidth(width);
		} else {
			viewerColumn.getColumn().setWidth(defaultWidth);
		}
		viewerColumn.getColumn().addControlListener(new ControlAdapter() {
			@Override
			public void controlResized(final ControlEvent e) {
				AbstractCatalogView.COLUMN_WIDTHS.put(uniqueID, viewerColumn.getColumn().getWidth());
			}
		});

		viewerColumn.setLabelProvider(columnLabelProvider);

		addSorter(viewerColumn, new ViewerSorter() {
			@Override
			public int compare(final Viewer viewer, final Object e1, final Object e2) {
				String text1 = columnLabelProvider.getText(e1);
				String text2 = columnLabelProvider.getText(e2);
				if (text1 == null) {
					return -1;
				}
				int compare = text1.compareTo(text2);
				if (viewer instanceof TreeViewer) {
					TreeViewer structuredViewer = (TreeViewer) viewer;
					if (structuredViewer.getTree().getSortDirection() == SWT.DOWN) {
						compare = compare * -1;
					}
				}
				return compare;
			}
		});
	}

	private void addSorter(final TreeViewerColumn column, final ViewerSorter viewerSorter) {
		column.getColumn().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				TreeViewer treeView = AbstractCatalogView.this.getViewer();
				treeView.setSorter(viewerSorter);
				treeView.getTree().setSortColumn(column.getColumn());
				if (treeView.getTree().getSortDirection() == SWT.UP) {
					treeView.getTree().setSortDirection(SWT.DOWN);
				} else if (treeView.getTree().getSortDirection() == SWT.DOWN) {
					treeView.getTree().setSortDirection(SWT.UP);
				} else {
					treeView.getTree().setSortDirection(SWT.UP);
				}
				treeView.refresh();
			}
		});

	}

	@Override
	public void setFocus() {
		this.treeViewer.getControl().setFocus();
	}

	/** Optionally delayed refresh */
	protected void refresh(final boolean delayed) {
		if (this.refreshJob == null) {
			this.refreshJob = new Job(getRefreshMessage()) {
				@Override
				protected IStatus run(final IProgressMonitor monitor) {
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							doRefresh();
						}
					});
					return Status.OK_STATUS;
				}
			};
		}
		if (delayed) {
			// delayed until it stops changing
			this.refreshJob.cancel();
			this.refreshJob.setPriority(Job.DECORATE);
			this.refreshJob.schedule(AbstractCatalogView.JOB_SCHEDULE_DELAY);
		} else {
			// force synchronous execution
			this.refreshJob.schedule();
			try {
				this.refreshJob.join();
			} catch (InterruptedException e) {
				MoDiscoLogger.logError(e, MoDiscoCommonUIPlugin.getDefault());
			}
		}
	}

	protected abstract String getRefreshMessage();

	protected abstract void doRefresh();

	@Override
	public void saveState(final IMemento memento) {
		super.saveState(memento);
		try {
			Set<Entry<String, Integer>> entrySet = AbstractCatalogView.COLUMN_WIDTHS.entrySet();
			for (Entry<String, Integer> entry : entrySet) {
				memento.putInteger(entry.getKey(), entry.getValue());
			}
		} catch (Exception e) {
			MoDiscoLogger
					.logError(e, "Error saving view state", MoDiscoCommonUIPlugin.getDefault()); //$NON-NLS-1$
		}
	}

	@Override
	public void init(final IViewSite site, final IMemento memento) throws PartInitException {
		super.init(site, memento);
		if (memento != null) {
			String[] attributeKeys = memento.getAttributeKeys();
			for (String key : attributeKeys) {
				AbstractCatalogView.COLUMN_WIDTHS.put(key, memento.getInteger(key));
			}
		}
	}

}
