/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.infra.discoverymanager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;

/**
 * @author GBarbier
 *
 */
public abstract class AbstractDiscovererImpl implements Discoverer {

	private final List<DiscoveryParameter> discovererParameters = new ArrayList<DiscoveryParameter>();
	/**
	 * To be able to add parameters to the list, we need a basic getter
	 * for sub-classes.
	 * However, we have to flag this method as final,
	 * to call it safely from constructors
	 * @return the modifiable list of discovery parameters
	 */
	final protected List<DiscoveryParameter> getBasicDiscovererParameters() {
		return this.discovererParameters;
	}
	
	/**
	 * @return the unmodifiable list of discovery parameters
	 */
	public List<DiscoveryParameter> getDiscovererParameters() {
		return Collections.unmodifiableList(this.discovererParameters);
	}

	private final DiscoveryParameterImpl targetModel;
	public DiscoveryParameter getTargetModelParameter() {
		return this.targetModel;
	}

	private final DiscoveryParameterImpl targetUri;
	public DiscoveryParameter getTargetUriParameter() {
		return this.targetUri;
	}
	
	public AbstractDiscovererImpl() {
		this.targetModel = new DiscoveryParameterImpl("TargetModel", DiscoveryParameterDirectionKind.out, Resource.class, false);
		this.getBasicDiscovererParameters().add(this.targetModel);
		this.targetUri = new DiscoveryParameterImpl("TargetUri", DiscoveryParameterDirectionKind.in, URI.class, false);
		this.getBasicDiscovererParameters().add(this.targetUri);
	}

	/*
	 * Provides a basic implementation of this deprecated method.
	 * It is to avoid re-implement it for each new discovery action.
	 * 
	 * (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.infra.discoverymanager.Discoverer#discoverElement(java.lang.Object, org.eclipse.emf.common.util.URI)
	 */
	public Resource discoverElement(Object source, URI target) {
		Resource result = null;
		Map<DiscoveryParameter, Object> parameters = new HashMap<DiscoveryParameter, Object>();
		parameters.put(this.getTargetUriParameter(), target);
		this.discoverElement(source, parameters);
		if (parameters.containsKey(this.getTargetModelParameter())) {
			result = (Resource) parameters.get(this.getTargetModelParameter());
		}
		return result;
	}
	
	
}
