/**
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Romain DERVAUX (Mia-Software) - initial API and implementation
 *    Fabien GIQUEL (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.java.actions.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.gmt.modisco.java.actions.DiscoveryParametersBean;
import org.eclipse.gmt.modisco.java.io.library.LibraryReaderOptions;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.PlatformUI;

/**
 * The page of the Java Wizard.
 */
public class JavaWizardFirstPage extends WizardPage {

	private Button btnMemberAnalyseJavaFiles;

	private Button btnFullAnalyseJavaFiles;

	/**
	 * the tree to selected the dependencies to discover
	 */
	private CheckboxTreeViewer dependenciesTree;

	/**
	 * The selected java project
	 */
	private IJavaProject javaProject;

	/**
	 * The button which allows to discover attached sources of a library instead
	 * of the java model created by the jdt.
	 */
	private Button btnUseSource;

	/**
	 * The button for incremental mode
	 */
	private Button btnIncrementalMode;

	/**
	 * the label which shows the number of selected projects/libraries.
	 */
	private Label textNumberElementsSelected;

	private Collection<Object> discoverableElements;

	private DiscoveryParametersBean discoveryParameters;
	
	/**
	 * the entry field where the user indicate the types and packages to exclude during analysis
	 */
	private Text entryFilter;

	public JavaWizardFirstPage(final IJavaProject javaProject,
			final String pageName,
			final Collection<Object> discoverableElements,
			final DiscoveryParametersBean discoveryParameters) {
		super(pageName);
		setTitle(pageName);
		setDescription(Messages.JavaWizardFirstPage_settingsLabel);
		this.javaProject = javaProject;
		this.discoverableElements = discoverableElements;
		this.discoveryParameters = discoveryParameters;
	}

	/**
	 * The tree content provider for the Dependencies Tree.
	 */
	class MyTreeContentProvider implements ITreeContentProvider {

		public Object[] getChildren(final Object element) {
			Set<IPackageFragmentRoot> libraries = new LinkedHashSet<IPackageFragmentRoot>();
			if (element instanceof IJavaProject) {
				for (Object tmp : JavaWizardFirstPage.this.discoverableElements) {
					if (tmp instanceof IPackageFragmentRoot) {
						IPackageFragmentRoot lib = (IPackageFragmentRoot) tmp;
						if (lib.getJavaProject().equals(element)) {
							libraries.add(lib);
						}
					}
				}
			}
			return libraries.toArray();
		}

		public Object getParent(final Object element) {
			if (element instanceof IPackageFragmentRoot) {
				return ((IPackageFragmentRoot) element).getJavaProject();
			}
			return null;
		}

		public boolean hasChildren(final Object element) {
			if (element instanceof IJavaProject) {
				return true;
			}
			return false;
		}

		public Object[] getElements(final Object inputElement) {
			Set<IJavaProject> projects = new LinkedHashSet<IJavaProject>();
			projects.add(JavaWizardFirstPage.this.javaProject);
			for (Object element : JavaWizardFirstPage.this.discoverableElements) {
				if (element instanceof IJavaProject) {
					projects.add((IJavaProject) element);
				}
			}
			return projects.toArray();
		}

		public void dispose() {
		}

		public void inputChanged(final Viewer viewer, final Object oldInput,
				final Object newInput) {
		}
	}

	public void createControl(final Composite parent) {

		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout formLayout = new GridLayout();
		formLayout.marginHeight = 10;
		formLayout.marginWidth = 10;
		composite.setLayout(formLayout);

		createDependenciesPanel(composite);

		createIncrementalPanel(composite);

		createJavaAnalysisLevelPanel(composite);

		createJavaFilterPanel(composite);

		// refreshDependenciesPanel();
		refreshInformationLabel();
		// refreshAnalysisLevelPanel();
		validatePage();

		setControl(composite);
	}

	private void createDependenciesPanel(final Composite composite) {
		// //////////////////////////////
		// the Dependencies Panel
		// //////////////////////////////
		Group dependencies = new Group(composite, SWT.NONE);
		dependencies.setText(Messages.JavaWizardFirstPage_dependencies);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 5;
		layout.marginWidth = 5;
		dependencies.setLayout(layout);

		GridData gridData = new GridData();
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalAlignment = SWT.FILL;
		gridData.verticalAlignment = SWT.FILL;
		dependencies.setLayoutData(gridData);

		Label l = new Label(dependencies, SWT.NONE);
		l.setText(Messages.JavaWizardFirstPage_selectProjects);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		l.setLayoutData(gridData);

		// //////////////////
		// Dependencies Tree
		// /////////////////
		this.dependenciesTree = new CheckboxTreeViewer(dependencies, SWT.BORDER
				| SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE | SWT.FULL_SELECTION);

		this.dependenciesTree.setContentProvider(new MyTreeContentProvider());
		this.dependenciesTree.setLabelProvider(new JavaElementLabelProvider());

		this.dependenciesTree.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(final CheckStateChangedEvent event) {

				Object element = event.getElement();

				if (JavaWizardFirstPage.this.discoveryParameters
						.getElementsToDiscover().contains(element)) {
					JavaWizardFirstPage.this.discoveryParameters
							.removeElementToDiscover(element);
				} else {
					JavaWizardFirstPage.this.discoveryParameters
							.addElementToDiscover(element);
				}

				JavaWizardFirstPage.this.dependenciesTree.update(event
						.getElement(), null);
				refreshBtnUseSources();
				refreshInformationLabel();
				validatePage();
			}
		});

		this.dependenciesTree.getTree().setLinesVisible(false);
		this.dependenciesTree.getTree().setHeaderVisible(true);

		this.dependenciesTree.setInput(new Object());

		// check the elements which are already knowned to be discovered
		for (TreeItem treeitem : this.dependenciesTree.getTree().getItems()) {
			recursiveCheckItem(treeitem);
		}

		TreeViewerColumn column1 = new TreeViewerColumn(this.dependenciesTree,
				SWT.NONE);
		column1.getColumn().setWidth(400);
		column1.getColumn().setText(Messages.JavaWizardFirstPage_name);
		column1.setLabelProvider(new JDTDelegateCellLabelProvider());

		TreeViewerColumn column2 = new TreeViewerColumn(this.dependenciesTree,
				SWT.NONE);
		column2.getColumn().setWidth(70);
		column2.getColumn().setText(Messages.JavaWizardFirstPage_useSources);
		column2.getColumn().setAlignment(SWT.CENTER);
		column2.setLabelProvider(new CellLabelProvider() {
			@Override
			public void update(final ViewerCell cell) {

				if (cell.getElement() instanceof IPackageFragmentRoot
						&& JavaWizardFirstPage.this.discoveryParameters
								.getElementsToDiscover().contains(
										cell.getElement())) {

					Map<String, Object> options = JavaWizardFirstPage.this.discoveryParameters
							.getDiscoveryOptions(cell.getElement());
					boolean useSources = Boolean.TRUE.equals(options
							.get(LibraryReaderOptions.USE_SOURCES.toString()));
					if (useSources) {
						cell.setText(Messages.JavaWizardFirstPage_useSourceOn);
						return;
					}
				}
				cell.setText(""); //$NON-NLS-1$
			}
		});

		this.dependenciesTree
				.addSelectionChangedListener(new ISelectionChangedListener() {
					public void selectionChanged(final SelectionChangedEvent e) {
						refreshBtnUseSources();
					}
				});

		gridData = new GridData();
		gridData.grabExcessVerticalSpace = true;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = SWT.FILL;
		gridData.verticalAlignment = SWT.FILL;
		gridData.horizontalSpan = 2;
		this.dependenciesTree.getTree().setLayoutData(gridData);

		// ////
		this.btnUseSource = new Button(dependencies, SWT.CHECK);
		this.btnUseSource.setText(Messages.JavaWizardFirstPage_useSources);
		this.btnUseSource.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				if (JavaWizardFirstPage.this.dependenciesTree.getTree()
						.getSelection().length == 1) {
					Object elt = JavaWizardFirstPage.this.dependenciesTree
							.getTree().getSelection()[0].getData();
					if (elt != null && elt instanceof IPackageFragmentRoot) {

						Map<String, Object> options = JavaWizardFirstPage.this.discoveryParameters
								.getDiscoveryOptions(elt);
						boolean useSource = Boolean.TRUE.equals(options
								.get(LibraryReaderOptions.USE_SOURCES
										.toString()));

						options.put(
								LibraryReaderOptions.USE_SOURCES.toString(),
								new Boolean(!useSource));

						JavaWizardFirstPage.this.dependenciesTree.update(elt,
								null);
					}
				}
				refreshBtnUseSources();
			}
		});
		this.btnUseSource.setSelection(false);
		this.btnUseSource.setEnabled(false);

		gridData = new GridData();
		gridData.horizontalSpan = 2;
		this.btnUseSource.setLayoutData(gridData);

		// /////////////
		Composite cLabels = new Composite(dependencies, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = SWT.FILL;
		gridData.grabExcessHorizontalSpace = true;
		cLabels.setLayoutData(gridData);

		RowLayout rowLayout = new RowLayout();
		rowLayout.wrap = false;
		cLabels.setLayout(rowLayout);

		// //
		Label imageInformationNumberElementsSelected = new Label(cLabels,
				SWT.NONE);
		imageInformationNumberElementsSelected.setImage(PlatformUI
				.getWorkbench().getSharedImages().getImage(
						org.eclipse.ui.ISharedImages.IMG_OBJS_INFO_TSK));

		this.textNumberElementsSelected = new Label(cLabels, SWT.NONE);
	}

	private void createIncrementalPanel(final Composite composite) {
		GridLayout layout;
		GridData gridData;
		// //////////////////////////////
		// the Incremental mode
		// //////////////////////////////
		Group mode = new Group(composite, SWT.NONE);
		mode.setText(Messages.JavaWizardFirstPage_mode);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 5;
		layout.marginWidth = 5;
		mode.setLayout(layout);

		gridData = new GridData();
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = SWT.FILL;
		gridData.verticalAlignment = SWT.FILL;
		mode.setLayoutData(gridData);

		// //// Incremental mode
		this.btnIncrementalMode = new Button(mode, SWT.CHECK);
		this.btnIncrementalMode
				.setText(Messages.JavaWizardFirstPage_useIncremental);
		this.btnIncrementalMode.setSelection(this.discoveryParameters
				.isIncrementalMode());
		this.btnIncrementalMode.setEnabled(true);

		gridData = new GridData();
		gridData.horizontalSpan = 2;
		this.btnIncrementalMode.setLayoutData(gridData);

		this.btnIncrementalMode.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				JavaWizardFirstPage.this.discoveryParameters
						.setIncrementalMode(JavaWizardFirstPage.this.btnIncrementalMode
								.getSelection());
			}

		});
	}

	private void createJavaAnalysisLevelPanel(final Composite composite) {
		GridData gridData;
		// //////////////////////////////
		// the Java Analysis Level panel
		// //////////////////////////////
		Group analysisLevel = new Group(composite, SWT.NONE);
		analysisLevel.setText(Messages.JavaWizardFirstPage_analysisLevel);

		GridLayout myLayout = new GridLayout();
		myLayout.numColumns = 1;
		analysisLevel.setLayout(myLayout);

		gridData = new GridData();
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = SWT.FILL;
		gridData.verticalAlignment = SWT.FILL;
		analysisLevel.setLayoutData(gridData);

		SelectionListener listener = new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				JavaWizardFirstPage.this.discoveryParameters
						.setFullJavaLevel(JavaWizardFirstPage.this.btnFullAnalyseJavaFiles
								.getSelection());
			}
		};

		this.btnFullAnalyseJavaFiles = new Button(analysisLevel, SWT.RADIO);
		this.btnFullAnalyseJavaFiles.setText(Messages.JavaWizardFirstPage_fullAnalysis);
		this.btnFullAnalyseJavaFiles.setLayoutData(new GridData());
		this.btnFullAnalyseJavaFiles
				.setSelection(JavaWizardFirstPage.this.discoveryParameters
						.isFullJavaLevel());
		this.btnFullAnalyseJavaFiles.addSelectionListener(listener);

		this.btnMemberAnalyseJavaFiles = new Button(analysisLevel, SWT.RADIO);
		this.btnMemberAnalyseJavaFiles
				.setText(Messages.JavaWizardFirstPage_limitedAnalysis);
		this.btnMemberAnalyseJavaFiles.setLayoutData(new GridData());
		this.btnMemberAnalyseJavaFiles.addSelectionListener(listener);
		this.btnMemberAnalyseJavaFiles
				.setSelection(!JavaWizardFirstPage.this.discoveryParameters
				.isFullJavaLevel());
	}

	private void recursiveCheckItem(final TreeItem treeitem) {
		if (this.discoveryParameters.getElementsToDiscover().contains(
				treeitem.getData())) {
			treeitem.setChecked(true);
			// expanded the libraries of the selected java project
			if (treeitem.getData() == this.javaProject) {
				treeitem.setExpanded(true);
			}
			this.dependenciesTree.refresh(treeitem.getData());
		}
		for (TreeItem child : treeitem.getItems()) {
			recursiveCheckItem(child);
		}
	}

	private void createJavaFilterPanel(final Composite composite) {
		GridData gridData;
		// //////////////////////////////
		// the Java Filter panel
		// //////////////////////////////
		Group filter = new Group(composite, SWT.NONE);
		filter.setText(Messages.JavaWizardFirstPage_filter);

		GridLayout myLayout = new GridLayout();
		myLayout.numColumns = 1;
		filter.setLayout(myLayout);

		gridData = new GridData();
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = SWT.FILL;
		gridData.verticalAlignment = SWT.FILL;
		filter.setLayoutData(gridData);

		//the label
		Label label = new Label(filter, SWT.NONE);
		label.setText(Messages.JavaWizardFirstPage_excludedElements);
		label.setLayoutData(new GridData());
		
		
		ModifyListener listener = new ModifyListener() {
			public void modifyText(ModifyEvent arg0) {
				validatePage();
			}
		};
		
		//the entry field to type the packages and types to exclude from analysis
		gridData = new GridData();
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = SWT.FILL;
		
		this.entryFilter = new Text(filter, SWT.SINGLE | SWT.BORDER);
		this.entryFilter
				.setToolTipText(Messages.JavaWizardFirstPage_filterToolTip);
		this.entryFilter.setLayoutData(gridData);
		this.entryFilter.addModifyListener(listener);
		
		if (this.discoveryParameters.getExcludedElements() != null
				&& this.discoveryParameters.getExcludedElements().size() > 0) {
			String lastExcludedElements = new String();
			for (String excludedElt : this.discoveryParameters
					.getExcludedElements()) {
				lastExcludedElements += excludedElt + ";"; //$NON-NLS-1$
			}
			this.entryFilter.setText(lastExcludedElements);
		}
	}
	
	/**
	 * change the state of the "use attached sources"
	 */
	void refreshBtnUseSources() {
		IStructuredSelection selection = (IStructuredSelection) this.dependenciesTree
				.getSelection();
		Object element = selection.getFirstElement();
		if (element instanceof IPackageFragmentRoot) {

			IPackageFragmentRoot lib = (IPackageFragmentRoot) element;

			if (this.discoveryParameters.getElementsToDiscover().contains(lib)) {

				Map<String, Object> options = this.discoveryParameters
						.getDiscoveryOptions(lib);
				boolean useSources = Boolean.TRUE.equals(options
						.get(LibraryReaderOptions.USE_SOURCES.toString()));

				if (useSources) {
					this.btnUseSource.setEnabled(true);
					this.btnUseSource.setSelection(true);
					return;
				} else {

					boolean hasSourceAttached = false;
					try {
						hasSourceAttached = (lib.getSourceAttachmentPath() != null);
					} catch (JavaModelException e) {
						// nothing
					}

					this.btnUseSource.setSelection(false);
					this.btnUseSource.setEnabled(hasSourceAttached);
					return;
				}
			}
		}
		this.btnUseSource.setEnabled(false);
		this.btnUseSource.setSelection(false);
	}

	/**
	 * validate the page
	 */
	void validatePage() {
		// check that the selected java project is checked in the tree
		for (TreeItem treeitem : this.dependenciesTree.getTree().getItems()) {
			if (treeitem.getData() == this.javaProject
					&& !treeitem.getChecked()) {
				setErrorMessage(Messages.JavaWizardFirstPage_firstProjectSelection);
				setPageComplete(false);
				return;
			}
		}
		
		//check the filter elements
		List<String> excludedElements = new ArrayList<String>();
		String text = this.entryFilter.getText().trim();
		String elements[] = text.split(";"); //$NON-NLS-1$
		for (int i = 0; i < elements.length; i++) {
			String elt = elements[i].trim();
			if(elt.length() == 0) {
				continue;	
			}
			if(elt.contains(" ") || elt.startsWith(".") || elt.endsWith(".")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				setErrorMessage(Messages.JavaWizardFirstPage_incorrectFiltered);
				setPageComplete(false);
				return;
			}
			if(elt.endsWith(".*")) { //$NON-NLS-1$
				elt = elt.substring(0, elt.length() - 2);
			}
			//element is well formed -> add it to the list
			excludedElements.add(elt);
		}
		this.discoveryParameters.setExcludedElements(excludedElements);
		
		// no problems encountered
		setErrorMessage(null);
		setMessage(null);
		setPageComplete(true);
	}

	/**
	 * Refresh the label on the number of elements checked
	 */
	void refreshInformationLabel() {

		int nbProjectsSelected = 0;
		int nbLibrariesSelected = 0;

		// we count the checked elements in the tree
		// if the user wants to discover dependencies
		for (Object element : this.dependenciesTree.getCheckedElements()) {
			if (element instanceof IJavaProject) {
				nbProjectsSelected++;
			} else if (element instanceof IPackageFragmentRoot) {
				nbLibrariesSelected++;
			}
		}

		StringBuilder sb = new StringBuilder();
		sb.append(nbProjectsSelected);
		if (nbProjectsSelected > 1) {
			sb.append(Messages.JavaWizardFirstPage_projects);
		} else {
			sb.append(Messages.JavaWizardFirstPage_project);
		}
			
		sb.append(Messages.JavaWizardFirstPage_and);
		sb.append(nbLibrariesSelected);
		if (nbLibrariesSelected > 1) {
			sb.append(Messages.JavaWizardFirstPage_libraries);
		} else {
			sb.append(Messages.JavaWizardFirstPage_library);
		}
		sb.append(Messages.JavaWizardFirstPage_selected);

		this.textNumberElementsSelected.setText(sb.toString());
		this.textNumberElementsSelected.pack(true);
	}

	/**
	 * A column label provider which extends the jdt label provider.
	 */
	class JDTDelegateCellLabelProvider extends ColumnLabelProvider {

		private JavaElementLabelProvider delegate = new JavaElementLabelProvider();

		@Override
		public void addListener(final ILabelProviderListener listener) {
			this.delegate.addListener(listener);
		}

		@Override
		public void dispose() {
			this.delegate.dispose();
		}

		@Override
		public boolean equals(final Object obj) {
			return this.delegate.equals(obj);
		}

		@Override
		public Image getImage(final Object element) {
			return this.delegate.getImage(element);
		}

		@Override
		public String getText(final Object element) {
			return this.delegate.getText(element);
		}

		@Override
		public int hashCode() {
			return this.delegate.hashCode();
		}

		@Override
		public boolean isLabelProperty(final Object element,
				final String property) {
			return this.delegate.isLabelProperty(element, property);
		}

		@Override
		public void removeListener(final ILabelProviderListener listener) {
			this.delegate.removeListener(listener);
		}

		@Override
		public String toString() {
			return this.delegate.toString();
		}

	}
}
