/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sbastien Minguet (Mia-Software) - initial API and implementation
 *    Frdric Madiot (Mia-Software) - initial API and implementation
 *    Fabien Giquel (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *    Erwan Breton (Sodifrance) - initial API and implementation
 *    Romain Dervaux (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.java.io.java;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmt.modisco.java.ASTNode;
import org.eclipse.gmt.modisco.java.Comment;
import org.eclipse.gmt.modisco.java.Package;
import org.eclipse.gmt.modisco.java.io.java.binding.PendingElement;

/**
 * The aim of this class is to link comments to the nodes retlated to JDT does
 * not attach comments (line and block comments) to model nodes.
 * 
 */
public final class CommentsManager {

	private CommentsManager() {

	}

	public static void resolveCommentPositions(final JDTVisitor visitor) {
		org.eclipse.jdt.core.dom.CompilationUnit cuJdtNode = visitor
				.getCuNode();
		org.eclipse.gmt.modisco.java.CompilationUnit moDiscoCuNode = (org.eclipse.gmt.modisco.java.CompilationUnit) visitor
				.getBijectiveMap().getValue(cuJdtNode);
		List<Comment> commentsList = new ArrayList<Comment>(visitor
				.getCommentsBinding().getValues());

		List<Comment> unLocatedComments = CommentsManager.jdtLocationSearch(
				visitor, cuJdtNode, moDiscoCuNode, commentsList);

		// processing remaining comments
		for (Comment comment : unLocatedComments) {
			boolean locationFound = CommentsManager.alternateLocationSearch(
					comment, visitor);

			/*
			 * We assure that all comments are processed, in linking remaining
			 * unlocated comments to root type.
			 */
			if (!locationFound) {
				if (visitor.getRootTypeOrEnum() != null) {
					visitor.getRootTypeOrEnum().getComments().add(comment);
				} else { // misc case of java file whithout type declaration
					EcoreUtil.delete(comment);
				}
			}
		}
	}

	/**
	 * Linking comments to Nodes using jdt location informations
	 * 
	 * @param visitor
	 * @param cuJdtNode
	 * @param moDiscoCuNode
	 * @param commentsList
	 * @return
	 */
	private static List<Comment> jdtLocationSearch(final JDTVisitor visitor,
			org.eclipse.jdt.core.dom.CompilationUnit cuJdtNode,
			org.eclipse.gmt.modisco.java.CompilationUnit moDiscoCuNode,
			List<Comment> commentsList) {
		List<Comment> localCommentList = new ArrayList<Comment>(commentsList);
		/**
		 * We must give an order to nodes which own comments : jdt ast nodes
		 * gives only starting and ending indexes. e.g., for a root class, start
		 * index will be 2 and end will be 11, for a method definition, start
		 * index will be 4 and end will be 6. Ordering will consider first the
		 * node whose starting index is the highest.
		 */
		SortedMap<Integer, org.eclipse.jdt.core.dom.ASTNode> nodesMap = new TreeMap<Integer, org.eclipse.jdt.core.dom.ASTNode>(
				new Comparator<Integer>() {
					public int compare(final Integer o1, final Integer o2) {
						return -o1.compareTo(o2);
					}
				});
		for (org.eclipse.jdt.core.dom.ASTNode node : visitor.getBijectiveMap()
				.getKeys()) {
			int index = cuJdtNode.firstLeadingCommentIndex(node);
			if (index != -1) {
				nodesMap.put(index, node);
			}
		}
		for (Integer indexMap : nodesMap.keySet()) {
			int index = indexMap;
			org.eclipse.jdt.core.dom.ASTNode jdtNode = nodesMap.get(indexMap);
			int lastIndex = cuJdtNode.lastTrailingCommentIndex(jdtNode);
			if (lastIndex == -1) { // occurs if only
				// one comment
				// associated
				lastIndex = indexMap + 1;
			}
			ASTNode element = visitor.getBijectiveMap().getValue(jdtNode);
			if (element instanceof PendingElement) {
				if (((PendingElement) element).getClientNode() != null) {
					element = ((PendingElement) element).getClientNode();
				} else { // should never happen
					element = visitor.getBijectiveMap().getValue(
							jdtNode.getParent());
				}
			}

			while (index != -1 && index < cuJdtNode.getCommentList().size()) {
				org.eclipse.jdt.core.dom.ASTNode jdtComment = (org.eclipse.jdt.core.dom.ASTNode) cuJdtNode
						.getCommentList().get(index);
				Comment comment = visitor.getCommentsBinding().get(jdtComment);
				if ((comment != null) && (localCommentList.contains(comment))) {
					if (index < lastIndex) {
						index++;
						if (!(element instanceof Package)) {
							element.getComments().add(comment);
							// CommentsManager.attachComment(comment,
							// jdtComment,
							// element, visitor);
							comment.setPrefixOfParent(true);
						} else {
							// misc case : one Package for many CompilationUnit
							moDiscoCuNode.getComments().add(comment);
							// CommentsManager.attachComment(comment,
							// jdtComment,
							// moDiscoCuNode, visitor);
							comment.setPrefixOfParent(true);
						}
						// remove from temporary list
						localCommentList.remove(comment);
					} else {
						index = -1;
					}
				} else {
					index = -1;
				}
			}
		}

		// return unlocated comment
		return localCommentList;
	}

	/**
	 * @param comment
	 * @param originalFileContent
	 * @return content of comment
	 */
	public static String extractCommentContent(
			final org.eclipse.jdt.core.dom.Comment comment,
			final String originalFileContent) {
		String result = originalFileContent.substring(comment
				.getStartPosition(), comment.getStartPosition()
				+ comment.getLength());
		return result;
	}

	/**
	 * Some misc comments are not linked to any node in jdt AST : - comments in
	 * a block, with at least one blank line before next node and one blank line
	 * after last node - comments in a block, at the end and with at least one
	 * blank line after last node. Use another algorithm to link them.
	 * 
	 * @param comment
	 * @param visitor
	 */
	private static boolean alternateLocationSearch(final Comment comment,
			final JDTVisitor visitor) {
		ASTNode bestParent = null;
		org.eclipse.jdt.core.dom.ASTNode jdtComment = visitor
				.getCommentsBinding().getKey(comment);

		// We search the nearest element in default parent subtree
		int bestFollowingNodeStart = Integer.MAX_VALUE;
		int bestFollowingNodeEnd = Integer.MIN_VALUE;
		int bestEnclosingNodeStart = Integer.MIN_VALUE;
		int bestEnclosingNodeEnd = Integer.MAX_VALUE;
		for (org.eclipse.jdt.core.dom.ASTNode jdtNode : visitor
				.getBijectiveMap().getKeys()) {
			if (CommentsManager.mayOwnComment(jdtNode)) {
				int sp = jdtNode.getStartPosition();
				int ep = jdtNode.getStartPosition() + jdtNode.getLength();
				// Does this node follow the comment more closely ?
				if ((sp >= jdtComment.getStartPosition()
						+ jdtComment.getLength())
						&& (sp < bestFollowingNodeStart || (sp == bestFollowingNodeStart && ep > bestFollowingNodeEnd))
						&& (sp < bestEnclosingNodeEnd)) {
					bestFollowingNodeStart = sp;
					bestFollowingNodeEnd = ep;
					bestParent = visitor.getBijectiveMap().get(jdtNode);
					comment.setEnclosedByParent(false);
					comment.setPrefixOfParent(true);
				}
				// Does this node enclose the comment more closely ?
				if ((sp <= jdtComment.getStartPosition())
						&& (ep > jdtComment.getStartPosition())
						&& (ep < bestFollowingNodeStart)
						&& (ep <= bestEnclosingNodeEnd)
						&& (sp >= bestEnclosingNodeStart)) {
					bestEnclosingNodeEnd = ep;
					bestEnclosingNodeStart = sp;
					bestParent = visitor.getBijectiveMap().get(jdtNode);
					comment.setEnclosedByParent(true);
					comment.setPrefixOfParent(false);
				}
			}
		}

		if (bestParent != null) {
			// insert comment in parent comment list at the right position
			CommentsManager.attachComment(comment, jdtComment, bestParent,
					visitor);
			return true;
		} else {
			return false;
		}

	}

	/*
	 * Links a comment to a receiver, in making sure of the comments ordering
	 */
	private static void attachComment(final Comment comment,
			org.eclipse.jdt.core.dom.ASTNode jdtComment, ASTNode receiver,
			final JDTVisitor visitor) {
		if (receiver.getComments().contains(comment))
			return;
		int insertIndex = 0;
		for (; insertIndex < receiver.getComments().size(); insertIndex++) {
			org.eclipse.jdt.core.dom.ASTNode jdtOtherComment = visitor
					.getCommentsBinding().getKey(
							receiver.getComments().get(insertIndex));
			if (jdtOtherComment != null
					&& jdtOtherComment.getStartPosition() > jdtComment
							.getStartPosition()) {
				break;
			}
		}
		receiver.getComments().add(insertIndex, comment);
	}

	/*
	 * A comment element (Comment, Tag, ...) can not own others comments (there
	 * is always an appropriate parent java statement)
	 */
	private static boolean mayOwnComment(
			org.eclipse.jdt.core.dom.ASTNode element) {
		return !(element instanceof org.eclipse.jdt.core.dom.Comment)
				&& !(element instanceof org.eclipse.jdt.core.dom.TagElement)
				&& !(element instanceof org.eclipse.jdt.core.dom.TextElement)
				&& !(element instanceof org.eclipse.jdt.core.dom.MemberRef)
				&& !(element instanceof org.eclipse.jdt.core.dom.MethodRef);
	}

}