 /*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Sbastien Minguet (Mia-Software) - initial API and implementation
 *    Frdric Madiot (Mia-Software) - initial API and implementation
 *    Fabien Giquel (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *    Erwan Breton (Sodifrance) - initial API and implementation
 *    Romain Dervaux (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.java.io.java;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.gmt.modisco.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.java.IModelReader;
import org.eclipse.gmt.modisco.java.JavaActivator;
import org.eclipse.gmt.modisco.java.Model;
import org.eclipse.gmt.modisco.java.emf.JavaFactory;
import org.eclipse.gmt.modisco.java.io.java.binding.BindingManager;
import org.eclipse.gmt.modisco.java.io.library.LibraryReader;
import org.eclipse.jdt.core.IClassFile;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeRoot;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.CompilationUnit;

/**
 * A {@code JavaReader} reads the contents of .java files and build the corresponding Java model.
 * <p>
 * As source, a {@code JavaReader} accepts {@link IJavaProject Java projects},
 * single .java {@link ICompilationUnit files}, and .class {@link IClassFile files} with source attached. 
 * </p>
 * <p>
 * It uses JDT model contruction. It is dedicated to J2SE5 (JDK 5 ~ JLS 3).
 * </p>
 * Using a more recent JLS release may require a new Mia metamodel definition,
 * since some new meta-objects may be required.
 * <p>
 * This reader does not work outside of an eclipse workspace context :
 * javaProject should reference a java project of an alive Eclipse Workspace !
 * (cf https://bugs.eclipse.org/bugs/show_bug.cgi?id=87852)
 * </p>
 * 
 * @see LibraryReader
 * @see ASTParser
 */
public class JavaReader implements IModelReader {

	
	protected final boolean ISINCREMENTALDISCOVERING;
	protected final boolean IS_FULL_LEVEL_ANALYSIS;
	protected JavaFactory factory;
	protected Model resultModel;
	protected BindingManager globalBindings;
	protected Map<String, Object> options;
	private final Collection<String> excludedElements;

	/**
	 * Constructs a {@code JavaReader} with no options and no incremental discovering.
	 * @param factory the EMF factory
	 */
	public JavaReader(final JavaFactory factory) {
		this(factory, new HashMap<String, Object>(), false, true);
	}
	
	/**
	 * Constructs a {@code JavaReader} with no options.
	 * Incremental discovering is determined by {@code isIncrementalDiscovering}.
	 * @param factory the EMF factory
	 * @param isIncrementalDiscovering if discovering is incremental
	 */
	public JavaReader(final JavaFactory factory,
			final boolean isIncrementalDiscovering) {
		this(factory, new HashMap<String, Object>(), isIncrementalDiscovering, true);
	}
	
	/**
	 * Constructs a {@code JavaReader}.
	 * @param factory the EMF factory
	 * @param options the options for this reader
	 * @param isIncrementalDiscovering if discovering is incremental
	 * @param isFullLevelAnalysis if analysis level is full or limited (class skeleton)
	 */
	public JavaReader(final JavaFactory factory,
			final Map<String, Object> options,
			final boolean isIncrementalDiscovering,
			final boolean isFullLevelAnalysis) {
		this(factory, options, isIncrementalDiscovering, isFullLevelAnalysis, new ArrayList<String>());
	}
	
	/**
	 * Constructs a {@code JavaReader}.
	 * @param factory the EMF factory
	 * @param options the options for this reader
	 * @param isIncrementalDiscovering if discovering is incremental
	 * @param isFullLevelAnalysis if analysis level is full or limited (class skeleton)
	 * @param excludedElements the elements (types & packages) to exclude during analysis
	 */
	public JavaReader(final JavaFactory factory,
			final Map<String, Object> options,
			final boolean isIncrementalDiscovering,
			final boolean isFullLevelAnalysis,
			Collection<String> excludedElements) {
		this.factory = factory;
		this.ISINCREMENTALDISCOVERING = isIncrementalDiscovering;
		this.IS_FULL_LEVEL_ANALYSIS = isFullLevelAnalysis;
		this.options = options;
		this.excludedElements = excludedElements;
	}
	
	
	
	public void readModel(final Object source, final Model resultModel,
			final IProgressMonitor monitor) {
		readModel(source, resultModel, getBindingManager(), monitor);
	}
	
	public void readModel(final Object source, final Model resultModel,
			final BindingManager bindingManager, final IProgressMonitor monitor) {
		
		if(source == null) {
			return;
		}
		
		this.resultModel = resultModel;
		this.globalBindings = bindingManager;
		if (this.ISINCREMENTALDISCOVERING) {
			this.globalBindings.enableIncrementalDiscovering(this.resultModel);
		} else {
			this.globalBindings.disableIncrementalDiscovering();
		}
		JDTVisitorUtils.initializePrimitiveTypes(this.factory, resultModel, this.globalBindings);
		
		try {
			if(source instanceof IJavaProject) {
				IJavaProject javaProject = (IJavaProject)source;
				
				if (resultModel.getName() == null || resultModel.getName().length() == 0) {
					resultModel.setName(javaProject.getElementName());
				}
				IPackageFragment[] packageFolder = javaProject.getPackageFragments();
				// loop on CompilationUnit-s
				for (IPackageFragment parent : packageFolder) {
					//test if package has compilations units and has not been excluded
					if (parent.getCompilationUnits().length > 0 && !this.isElementExcluded(parent.getElementName())){
						//report some feedback
						monitor.subTask(Messages.JavaReader_discoveringTask + parent.getElementName());
						//parse package
						parsePackage(javaProject, resultModel, parent, monitor);
					}
					if(monitor.isCanceled()){
						return;
					}
				}
			} else if (source instanceof ITypeRoot) {
				parseTypeRoot((ITypeRoot) source);
				
			} else {
				throw new IllegalArgumentException("Java reader can not handle source object : " + source.toString()); //$NON-NLS-1$
			}
		} catch (Exception e) {
			MoDiscoLogger.logError(e, JavaActivator.getDefault());
		}
	}
	
	/**
	 * Indicate if an element (type or package) has to be excluded during analysis.
	 * @param name the qualified name of the element (for example : org.eclipse or java.io.File)
	 * @return {@code true} if the element represented by {@code name} has to be excluded, {@code false} otherwise
	 */
	protected boolean isElementExcluded(String name) {
		for (String exc : this.excludedElements) {
			if(name.toLowerCase().startsWith(exc.toLowerCase())) {
				return true;
			}
		}
		return false;
	}
	
	protected void parseTypeRoot(final ITypeRoot source) {
		
		
		org.eclipse.jdt.core.dom.CompilationUnit parsedCompilationUnit = parseCompilationUnit(source);
		String fileContent = null;
		String filePath = null;
		try {
			if(source instanceof ICompilationUnit) {
				IFile theIFile = ResourcesPlugin.getWorkspace().getRoot()
						.getFile(parsedCompilationUnit.getJavaElement().getPath());
				//getContent(IFile) is faster than ICompilationUnit.getSource()
				fileContent = getContent(theIFile).toString();
				IProject project = source.getJavaProject().getProject();
				filePath = getRelativePath(project, parsedCompilationUnit);
			} else {
				//IJavaElement.CLASS_FILE
				fileContent = LibraryReader.getFileContent((IClassFile) source);
				filePath = LibraryReader.getPath((IClassFile) source);
			}
			visitCompilationUnit(this.resultModel, parsedCompilationUnit, filePath, fileContent);
			
		} catch (Exception e) {
			MoDiscoLogger.logError(e, JavaActivator.getDefault());
		}
	}
	
	protected void parsePackage(final IJavaProject javaProject,
			final Model resultModel, final IPackageFragment parent,
			final IProgressMonitor monitor) throws JavaModelException {
		ICompilationUnit[] children = parent.getCompilationUnits();
		for (ICompilationUnit cu : children) {
			//iterate on each type of each cu and check if one is excluded
			boolean isExcluded = false;
			for(IType t : cu.getTypes()) {
				if(this.isElementExcluded(t.getFullyQualifiedName())) {
					isExcluded = true;
					break;
				}
			}
			if(!isExcluded) {
				parseTypeRoot(cu);
			}
			if(monitor.isCanceled()) {
				return;
			}
		}
		if (this.ISINCREMENTALDISCOVERING) {
			reset();
		}
	}
	
	protected void visitCompilationUnit(
			final Model resultModel,
			final org.eclipse.jdt.core.dom.CompilationUnit parsedCompilationUnit,
			final String filePath, final String fileContent) {
		
		JDTVisitor jdtVisitor = new JDTVisitor(this.factory, resultModel,
				this.globalBindings, filePath, fileContent, 
				this.globalBindings.isIncrementalDiscovering(),
				this.IS_FULL_LEVEL_ANALYSIS);
		parsedCompilationUnit.accept(jdtVisitor);
	}

	protected org.eclipse.jdt.core.dom.CompilationUnit parseCompilationUnit(
			final ITypeRoot source) {
		// Code parsing : here is indicated the version of jdk (~JLS) to
		// consider, see Class comments
		ASTParser parser = ASTParser.newParser(AST.JLS3);
		parser.setResolveBindings(true);
		parser.setSource(source);
		org.eclipse.jdt.core.dom.CompilationUnit parsedCompilationUnit = (CompilationUnit) parser.createAST(null);
		return parsedCompilationUnit;
	}

	protected void resolveMethodRedefinition(final Model resultModel) {
		MethodRedefinitionManager.resolveMethodRedefinitions(resultModel, this.factory);
	}

	protected void finalResolveBindings(final Model resultModel) {
		this.globalBindings.resolveBindings(resultModel);
	}

	public static String getRelativePath(final IProject aProject,
			final org.eclipse.jdt.core.dom.CompilationUnit parsedCompilationUnit) {
		IPath projectpath = aProject.getFullPath();
		IPath filepath = parsedCompilationUnit.getJavaElement().getPath();
		//	we want path relative to project directory
		if (projectpath.isPrefixOf(filepath)) {
			filepath = filepath.removeFirstSegments(projectpath.segmentCount());
		}
		String filePathString = filepath.toOSString();
		if (!filePathString.startsWith(java.io.File.separator)) {
			filePathString = java.io.File.separator + filePathString;
		}
		return filePathString;
	}


	public static StringBuilder getContent(final IFile anIFile)
			throws CoreException, IOException {
		InputStream is = anIFile.getContents();
		StringBuilder cuText = new StringBuilder();
		Reader r = new InputStreamReader(is);
		char[] chars = new char[100];
		int read;
		while ((read = r.read(chars)) != -1) {
			if (read == 100) {
				cuText.append(chars);
			} else {
				cuText.append(chars, 0, read);
			}
		}
		is.close();
		return cuText;
	}
	
	protected void reset() {
		this.globalBindings.resolveBindings(this.resultModel);
		this.globalBindings = getBindingManager();
	}

	protected BindingManager getBindingManager() {
		BindingManager bindingManager = new BindingManager(this.factory);
		if (this.ISINCREMENTALDISCOVERING) {
			bindingManager.enableIncrementalDiscovering(this.resultModel);
		}
		return bindingManager;
	}

	public void terminate(final IProgressMonitor monitor) {
		monitor.subTask(Messages.JavaReader_bindingsTask);
		finalResolveBindings(this.resultModel);
		
		monitor.subTask(Messages.JavaReader_redefinitionsTask);
		resolveMethodRedefinition(this.resultModel);
	}


}
