/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.discoverer.kdm.source.actions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.logging.AtlLogHandler;
import org.eclipse.gmt.modisco.common.core.utils.Tools;
import org.eclipse.gmt.modisco.discoverer.kdm.source.KDMSourceDiscoverer;
import org.eclipse.gmt.modisco.infra.discoverymanager.Discoverer;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameter;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameterDirectionKind;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameterImpl;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

/**
 * @author Gabriel Barbier
 * 
 */
public class DiscoverSourceModelFromContainer implements Discoverer {

	private final String targetExtension = "kdm"; //$NON-NLS-1$
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.gmt.modisco.infra.discoverymanager.DiscovererInterface#applyTo
	 * (java.lang.Object)
	 */
	public boolean isApplicableTo(final Object source) {
		boolean result = false;
		if (source instanceof IJavaProject) {
			result = true;
		} else if (source instanceof IContainer) {
			result = true;
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.gmt.modisco.infra.discoverymanager.DiscovererInterface#
	 * discoverElement(java.lang.Object, org.eclipse.emf.common.util.URI)
	 */
	public Resource discoverElement(final Object source, final URI target) {
		Resource resource = null;
		final Logger logger = Logger.getLogger(this.getClass().getName());
		// retrieval of Java project
		IContainer tempProject = null;
		if (source instanceof IJavaProject) {
			tempProject = ((IJavaProject) source).getProject();
		} else if (source instanceof IContainer) {
			tempProject = (IContainer) source;			
		}
		final IContainer project = tempProject;

		// retrieval of target path
		IPath tempPath = project.getProject().getLocation()
				.addTrailingSeparator().append(project.getName()+"_Source") //$NON-NLS-1$
				.addFileExtension(this.targetExtension);
		if (target != null) {
			if (target.isRelative()) {
				tempPath = ResourcesPlugin.getWorkspace().getRoot()
						.getFullPath().append(target.path());
			} else {
				tempPath = new Path(target.toFileString());
			}
		}

		final IPath path = tempPath;

		final AtlLogHandler logHandler = new AtlLogHandler(path.toString()
				+ ".log"); //$NON-NLS-1$
		logger.addHandler(logHandler);
		try {
			final KDMSourceDiscoverer disco = new KDMSourceDiscoverer();
			resource = disco.getKDMModelFromContainer(project);
			if (target != null) {
				Tools.saveModel(resource, target);
			} else {
				Tools.saveModel(resource, path.toString());
			}
		} catch (Exception e) {
			logger.log(Level.WARNING, "Error in discovery of container", e); //$NON-NLS-1$
		} finally {
			logger.removeHandler(logHandler);
			logHandler.close();
			try {
				project.refreshLocal(1, null);
			} catch (Exception e1) {
				// give up
			}
		}
		// access UI resources so that we can obtain the current eclipse
		// workbench activePage
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				try {
					IWorkspaceRoot root = ResourcesPlugin.getWorkspace()
							.getRoot();
					IFile ifile = root.getFileForLocation(path);
					if ((ifile != null) && ifile.exists()) {
						IWorkbenchPage page = PlatformUI.getWorkbench()
								.getActiveWorkbenchWindow().getActivePage();
						// programmatically opens the j2se5 model using the
						// default editor if one is present
						IDE.openEditor(page, ifile);
					}
				} catch (PartInitException e) {
					e.printStackTrace();
				}
			}
		});

		return resource;
	}

	public String toString() {
		return Messages.DiscoverSourceModelFromContainer_0;
	}

	public DiscoverSourceModelFromContainer() {
		this.targetModel = new DiscoveryParameterImpl("TargetModel", DiscoveryParameterDirectionKind.out, Resource.class, false); //$NON-NLS-1$
		this.discovererParameters.add(this.targetModel);
		this.targetUri = new DiscoveryParameterImpl("TargetUri", DiscoveryParameterDirectionKind.in, URI.class, false); //$NON-NLS-1$
		this.discovererParameters.add(this.targetUri);
	}

	private final DiscoveryParameterImpl targetModel;
	private final DiscoveryParameterImpl targetUri;
	private final List<DiscoveryParameter> discovererParameters = new ArrayList<DiscoveryParameter>();
	
	public void discoverElement(final Object source,
			final Map<DiscoveryParameter, Object> parameters) {
		URI targetUri = null;
		if (parameters.containsKey(this.targetUri)) {
			Object value = parameters.get(this.targetUri);
			if (value instanceof URI) {
				targetUri = (URI) value;
			}
		}
		Resource result = this.discoverElement(source, targetUri);
		parameters.put(this.targetModel, result);
	}

	public List<DiscoveryParameter> getDiscovererParameters() {
		return Collections.unmodifiableList(this.discovererParameters);
	}
}
