/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.converter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.zest.core.widgets.Graph;
import org.eclipse.zest.core.widgets.GraphConnection;
import org.eclipse.zest.core.widgets.GraphNode;
import org.eclipse.zest.layouts.LayoutStyles;
import org.eclipse.zest.layouts.algorithms.HorizontalTreeLayoutAlgorithm;


/**
 * @author Gabriel Barbier
 *
 */
public class MethodCallsZestGraphReader {

	private final List<Operation> parents = new ArrayList<Operation>();
	
	/**
	 * @param operation
	 */
	public final void initializeZest(Operation operation) {
		// Create the shell
		Display d = new Display();
		Shell shell = new Shell(d);
		shell.setText(operation.getName() + "  |  z e s t  |  m e t h o d s  c a l l s  v i e w");
		shell.setLayout(new FillLayout());
		shell.setSize(400, 400);

		this.initializeMethodCallsGraph(operation, shell);
		
		shell.open();
		while (!shell.isDisposed()) {
			while (!d.readAndDispatch()) {
				d.sleep();
			}
		}
		
	}
	
	public final void initializeZest(final IFile model) {
		String packageName = model.getProjectRelativePath().removeFileExtension().lastSegment().toLowerCase();
		// Create the shell
		Display d = new Display();
		Shell shell = new Shell(d);
		shell.setText(packageName + "  |  z e s t  |  m e t h o d s  c a l l s  v i e w");
		shell.setLayout(new FillLayout());
		shell.setSize(400, 400);

		/*
		 * We have to load the model (to retrieve root elements)
		 * Then we have to build the graph.
		 * 
		 * Finally, we will be able to reuse existing code to visualize
		 * the inheritance graph.
		 */
		ResourceSet resourceSet = new ResourceSetImpl();
		Resource resource = 
			resourceSet.getResource(
					URI.createPlatformResourceURI(model.getFullPath().toString(), false), true);
		
		
		this.initializeMethodCallsGraph(resource, shell);
		
		resource.unload();
		
		shell.open();
		while (!shell.isDisposed()) {
			while (!d.readAndDispatch()) {
				d.sleep();
			}
		}
	}
	
	public final void initializeZest(final Resource resource) {
		String packageName = "";
		if (resource.getURI() != null) {
			packageName = resource.getURI().trimFileExtension().lastSegment().toLowerCase();
		}
		// Create the shell
		Display d = new Display();
		Shell shell = new Shell(d);
		shell.setText(packageName + "  |  z e s t  |  m e t h o d s  c a l l s  v i e w");
		shell.setLayout(new FillLayout());
		shell.setSize(400, 400);

		/*
		 * We have to load the model (to retrieve root elements)
		 * Then we have to build the graph.
		 * 
		 * Finally, we will be able to reuse existing code to visualize
		 * the inheritance graph.
		 */
		
		this.initializeMethodCallsGraph(resource, shell);
		
		resource.unload();
		
		shell.open();
		while (!shell.isDisposed()) {
			while (!d.readAndDispatch()) {
				d.sleep();
			}
		}	
	}
	/*
	 * From a uml model, we have to represent method calls as a graph.
	 * First step is to find all root methods
	 * (methods that have no dependencies pointing on them)
	 * Second step is to create links by following dependencies relationships.
	 * Third step is to manage cycles ?
	 * Forth step is to be able to get back to the code from a selected node
	 * Fifth step is to represents methods from real types
	 * (bis: have constraints to restrain possibilities)
	 * 
	 */
	private final  Graph initializeMethodCallsGraph(final Resource resource, final Shell shell) {
		Graph result = new Graph(shell, SWT.NONE);
		GraphNode parentNode = new GraphNode(result, SWT.NONE, "Method Calls Graph");
		result.setLayoutAlgorithm(new HorizontalTreeLayoutAlgorithm(LayoutStyles.NO_LAYOUT_NODE_RESIZING), true);

		List<Operation> allOperations = this.getAllOperations(resource);
		
		for (Operation rootElement : this.getRootOperationsForMethodCallsGraph(allOperations)) {
			this.generateMethodCallsNode(result, parentNode, rootElement);
		}
		
		return result;
	}
	
	private final Graph initializeMethodCallsGraph(final Operation rootElement, final Shell shell) {
		
		Graph result = new Graph(shell, SWT.NONE);
		GraphNode parentNode = new GraphNode(result, SWT.NONE, "Method Calls Graph");
		result.setLayoutAlgorithm(new HorizontalTreeLayoutAlgorithm(LayoutStyles.NO_LAYOUT_NODE_RESIZING), true);

		this.generateMethodCallsNode(result, parentNode, rootElement);
		
		return result;
	}
	
	private final void generateMethodCallsNode(final Graph result, final GraphNode parentNode, final Operation element) {
		GraphNode activeNode = null;
		String name = element.getNamespace().getName() + " :: " + element.getName();
		if (this.parents.contains(element)) {
			// This element has been already managed (recursion)
			activeNode = new GraphNode(result, SWT.NONE, "/recursion/ " + name);
			new GraphConnection(result, SWT.NONE, parentNode, activeNode);
		} else {
			// manage recursion
			this.parents.add(element);
			
			List<Operation> calledMethods = this.getCalledMethods(element);
			if (calledMethods.isEmpty()) {
				activeNode = new GraphNode(result, SWT.NONE, name);
			} else {
				activeNode = new GraphNode(result, SWT.NONE, name + " (" + calledMethods.size() + ")");
			}
			
			new GraphConnection(result, SWT.NONE, parentNode, activeNode);
			for (Operation child : calledMethods) {
				this.generateMethodCallsNode(result, activeNode, child);
			}
			// manage recursion
			this.parents.remove(element);
		}
	}
	
	private final List<Operation> getAllOperations(Resource resource) {
		List<Operation> result = new ArrayList<Operation>();
		TreeIterator<EObject> iterator = resource.getAllContents();
		while (iterator.hasNext()) {
			EObject object = iterator.next();
			if (object instanceof Operation) {
				Operation operation = (Operation) object;
				if (operation.getName().equalsIgnoreCase("dummy") == false) {
					result.add(operation);
				}
			}
		}
		return result;
	}
	
	private final List<Operation> getRootOperationsForMethodCallsGraph(List<Operation> elements) {
		List<Operation> result = new ArrayList<Operation>();
		List<Dependency> methodCalls = new ArrayList<Dependency>();
		for (Operation element : elements) {
			methodCalls.addAll(element.getClientDependencies());
		}
		for (Operation element : elements) {
			boolean root = true;
			for (Dependency methodCall : methodCalls) {
				if (methodCall.getSuppliers().contains(element)) {
					root = false;
				}
			}
			if (root) {
				result.add(element);
			}
		}
		return result;
	}
	/*
	 * We have the collection of dependencies.
	 * Yet, we have also to sort this collection by its order in call sequence
	 */
	private final List<Operation> getCalledMethods(Operation parent) {
		List<Operation> result = new ArrayList<Operation>();
		List<Dependency> methodCalls = parent.getClientDependencies();
		Collections.sort(methodCalls, new Comparator<Dependency>() {
			private final Integer extractCallRank(Dependency dependency) {
				Integer result = 0;
				String name = dependency.getName();
				if (name != null) {
					String[] parts = name.split(" ");
					String number = parts[parts.length - 1];
					
					result = Integer.valueOf(number);
				}
				return result;
			}
			public int compare(Dependency o1, Dependency o2) {
				return this.extractCallRank(o1).compareTo(this.extractCallRank(o2));
			}});
		
 		for (Dependency methodCall : methodCalls) {
			for (NamedElement callee : methodCall.getSuppliers()) {
				if (callee instanceof Operation) {
					result.add((Operation) callee);
				}
			}
		}
		return result;
	}
}
