/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.editor;

import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.Panel;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.JComponent;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.converter.PrefuseGraphContainerForUml;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;

import prefuse.data.Graph;
import prefuse.data.Node;
import prefuse.data.Table;
import prefuse.data.Tree;


/**
 * @author Gabriel Barbier
 *
 */
public class MethodCallsPrefuseGraphEditor extends EditorPart {

	public static final String EditorID = "org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.prefuseeditor.EditorID";
	
	private PrefuseGraphInput editorInput;
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public void doSave(IProgressMonitor monitor) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	@Override
	public void doSaveAs() {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {
		if (input instanceof PrefuseGraphInput) {
			this.editorInput = (PrefuseGraphInput) input;
			setSite(site);
			setInput(this.editorInput);
			setPartName("Prefuse Graph Viewer Part Name");
		} else {
			throw new PartInitException("Input should be of type PrefuseGraphInput");
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isDirty()
	 */
	@Override
	public boolean isDirty() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@SuppressWarnings("serial")
	@Override
	public void createPartControl(Composite parent) {
		if (this.editorInput != null) {
			Composite composite = new Composite(parent, SWT.EMBEDDED | SWT.NO_BACKGROUND);
		    Frame frame = SWT_AWT.new_Frame(composite);
		    Graph graph = null;
		    if (this.editorInput.getInputFile() != null) {
				final IFile model = this.editorInput.getInputFile();
				String packageName = model.getProjectRelativePath().removeFileExtension().lastSegment().toLowerCase();
				this.setPartName(packageName);
				/*
				 * We have to load the model (to retrieve root elements)
				 * Then we have to build the graph.
				 * 
				 */
				ResourceSet resourceSet = new ResourceSetImpl();
				Resource resource = 
					resourceSet.getResource(
							URI.createPlatformResourceURI(model.getFullPath().toString(), false), true);
								
				graph = this.initializeMethodCallsGraph(resource);
				
				resource.unload();
			} else if (this.editorInput.getInputResource() != null) {
				Resource resource = this.editorInput.getInputResource();
				String packageName = "";
				if (resource.getURI() != null) {
					packageName = resource.getURI().trimFileExtension().lastSegment().toLowerCase();
				}
				this.setPartName(packageName);
				graph = this.initializeMethodCallsGraph(resource);
			} else if (this.editorInput.getInputOperation() != null) {
				Operation operation = this.editorInput.getInputOperation();
				String name = operation.getName();
				this.setPartName(name);
				graph = this.initializeMethodCallsGraph(operation);
			}
		    JComponent treeview;
	        treeview = PrefuseGraphContainerForUml.getInstance().initializeTreeViewContainer(graph, this.nameAttribute, null);
	        
	        
	        //frame.setContentPane(treeview);
	        Panel panel = new Panel(new BorderLayout()) {
	        	public void update(java.awt.Graphics g) {
	        		paint(g);
	        	}
	        };
	        frame.add(panel);
	        panel.add(treeview);
	        panel.addMouseListener(new MouseAdapter() {

				/* (non-Javadoc)
				 * @see java.awt.event.MouseAdapter#mouseClicked(java.awt.event.MouseEvent)
				 */
				@Override
				public void mouseClicked(MouseEvent e) {
					System.out.println(e.getButton());
					System.out.println(e.getClickCount());
					System.out.println(e.getSource());
					System.out.println(e.getComponent());
				}} );
	        //frame.pack();
	        //frame.setVisible(true);
		}

	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		
	}

	private final String nameAttribute = "name";
	private final List<Operation> parents = new ArrayList<Operation>();
	
//	/**
//	 * @param operation
//	 */
//	public final void initializePrefuse(Operation operation) {
//		Graph graph = this.initializeMethodCallsGraph(operation);
//		
//		String label = "name";
//        
//        JComponent treeview;
//        treeview = Princ.demo(graph, label);
//        JFrame frame = new JFrame(operation.getName() + "  |  p r e f u s e  |  m e t h o d s  c a l l s  v i e w");
//        frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
//        frame.setContentPane(treeview);
//        frame.pack();
//        frame.setVisible(true);
//		
//	}
	
//	public final void initializePrefuse(final IFile model) {
//		/*
//		 * We have to load the model (to retrieve root elements)
//		 * Then we have to build the graph.
//		 * 
//		 * Finally, we will be able to reuse existing code to visualize
//		 * the inheritance graph.
//		 */
//		ResourceSet resourceSet = new ResourceSetImpl();
//		Resource resource = 
//			resourceSet.getResource(
//					URI.createPlatformResourceURI(model.getFullPath().toString(), false), true);
//		
//		
//		Graph graph = this.initializeMethodCallsGraph(resource);
//		
//		resource.unload();
//		
//		/*
//		 * Tricky code:
//		 * to get the url of xml file, we will use naming convention ...
//		 * name of xml file should be "<ePackage.name>_inheritance_graph.xml"
//		 * 
//		 * Hopefully, ecore file name are generally equals to <ePackage.name>
//		 */
//		String packageName = model.getProjectRelativePath().removeFileExtension().lastSegment().toLowerCase();
//		
//		String label = "name";
//        
//        JComponent treeview;
//        treeview = Princ.demo(graph, label);
//        JFrame frame = new JFrame(packageName + "  |  p r e f u s e  |  m e t h o d s  c a l l s  v i e w");
//        frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
//        frame.setContentPane(treeview);
//        frame.pack();
//        frame.setVisible(true);
//		
//	}
	
//	public final void initializePrefuse(final Resource resource) {
//				
//		Graph graph = this.initializeMethodCallsGraph(resource);
//		String packageName = "";
//		if (resource.getURI() != null) {
//			packageName = resource.getURI().trimFileExtension().lastSegment().toLowerCase();
//		}
//		String label = "name";
//        
//        JComponent treeview;
//        treeview = Princ.demo(graph, label);
//        JFrame frame = new JFrame(packageName + "  |  p r e f u s e  |  m e t h o d s  c a l l s  v i e w");
//        frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
//        frame.setContentPane(treeview);
//        frame.pack();
//        frame.setVisible(true);
//		
//	}
	/*
	 * From a uml model, we have to represent method calls as a graph.
	 * First step is to find all root methods
	 * (methods that have no dependencies pointing on them)
	 * Second step is to create links by following dependencies relationships.
	 * Third step is to manage cycles ?
	 * Forth step is to be able to get back to the code from a selected node
	 * Fifth step is to represents methods from real types
	 * (bis: have constraints to restrain possibilities)
	 * 
	 */
	private final  Tree initializeMethodCallsGraph(Resource resource) {
		Tree result = new Tree();
		Table nodes = result.getNodeTable();
		Node activeNode = null;
		
		// add attribute name for each node
		nodes.addColumn(this.nameAttribute, String.class);
		
		// add root element
		activeNode = result.addRoot();
		activeNode.set(this.nameAttribute, "Method Calls Graph");
		nodes.addColumn(PrefuseGraphContainerForUml.umlOperation, Operation.class);
		nodes.addColumn(PrefuseGraphContainerForUml.javaProject, IJavaProject.class);
		
		List<Operation> allOperations = this.getAllOperations(resource);
		
		Node parentNode = activeNode;
		for (Operation rootElement : this.getRootOperationsForMethodCallsGraph(allOperations)) {
			this.generateMethodCallsNode(result, parentNode, rootElement);
		}
		return result;
	}
	
	private final Tree initializeMethodCallsGraph(Operation rootElement) {
		Tree result = new Tree();
		
		Table nodes = result.getNodeTable();
		Node activeNode = null;
		
		// add attribute name for each node
		nodes.addColumn(this.nameAttribute, String.class);
		nodes.addColumn(PrefuseGraphContainerForUml.umlOperation, Operation.class);
		nodes.addColumn(PrefuseGraphContainerForUml.javaProject, IJavaProject.class);
		
		// add root element
		activeNode = result.addRoot();
		activeNode.set(this.nameAttribute, "Method Calls Graph");
		
		
		Node parentNode = activeNode;
		this.generateMethodCallsNode(result, parentNode, rootElement);
		return result;
	}
	
	private final void generateMethodCallsNode(final Tree result, final Node parentNode, final Operation element) {
		Node activeNode = result.addChild(parentNode);
		activeNode.set(PrefuseGraphContainerForUml.umlOperation, element);
		activeNode.set(PrefuseGraphContainerForUml.javaProject, this.editorInput.getJavaProject());
		
		String name = element.getNamespace().getName() + " :: " + element.getName();
		if (this.parents.contains(element)) {
			// This element has been already managed (recursion)
			activeNode.set(this.nameAttribute, "/recursion/ " + name);
		} else {
			// manage recursion
			this.parents.add(element);
			
			List<Operation> calledMethods = this.getCalledMethods(element);
			if (calledMethods.isEmpty()) {
				activeNode.set(this.nameAttribute, name);
			} else {
				activeNode.set(this.nameAttribute, name + " (" + calledMethods.size() + ")");
			}
			for (Operation child : calledMethods) {
				this.generateMethodCallsNode(result, activeNode, child);
			}
			// manage recursion
			this.parents.remove(element);
		}
	}
	
	private final List<Operation> getAllOperations(Resource resource) {
		List<Operation> result = new ArrayList<Operation>();
		TreeIterator<EObject> iterator = resource.getAllContents();
		while (iterator.hasNext()) {
			EObject object = iterator.next();
			if (object instanceof Operation) {
				Operation operation = (Operation) object;
				if (operation.getName().equalsIgnoreCase("dummy") == false) {
					result.add(operation);
				}
			}
		}
		return result;
	}
	
	private final List<Operation> getRootOperationsForMethodCallsGraph(List<Operation> elements) {
		List<Operation> result = new ArrayList<Operation>();
		List<Dependency> methodCalls = new ArrayList<Dependency>();
		for (Operation element : elements) {
			methodCalls.addAll(element.getClientDependencies());
		}
		for (Operation element : elements) {
			boolean root = true;
			for (Dependency methodCall : methodCalls) {
				if (methodCall.getSuppliers().contains(element)) {
					root = false;
				}
			}
			if (root) {
				result.add(element);
			}
		}
		return result;
	}
	/*
	 * We have the collection of dependencies.
	 * Yet, we have also to sort this collection by its order in call sequence
	 */
	private final List<Operation> getCalledMethods(Operation parent) {
		List<Operation> result = new ArrayList<Operation>();
		List<Dependency> methodCalls = parent.getClientDependencies();
		Collections.sort(methodCalls, new Comparator<Dependency>() {
			private final Integer extractCallRank(Dependency dependency) {
				Integer result = 0;
				String name = dependency.getName();
				if (name != null) {
					String[] parts = name.split(" ");
					String number = parts[parts.length - 1];
					
					result = Integer.valueOf(number);
				}
				return result;
			}
			public int compare(Dependency o1, Dependency o2) {
				return this.extractCallRank(o1).compareTo(this.extractCallRank(o2));
			}});
		
 		for (Dependency methodCall : methodCalls) {
			for (NamedElement callee : methodCall.getSuppliers()) {
				if (callee instanceof Operation) {
					result.add((Operation) callee);
				}
			}
		}
		return result;
	}
}
