/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.editor;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.Label;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.zest.core.widgets.Graph;
import org.eclipse.zest.core.widgets.GraphConnection;
import org.eclipse.zest.core.widgets.GraphNode;
import org.eclipse.zest.layouts.LayoutStyles;
import org.eclipse.zest.layouts.algorithms.HorizontalTreeLayoutAlgorithm;

/**
 * @author Gabriel Barbier
 *
 */
public class MethodCallsZestGraphEditor extends EditorPart {

	public static final String EditorID = "org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.zesteditor.EditorID";
	
	private ZestGraphInput editorInput;
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public void doSave(IProgressMonitor monitor) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	@Override
	public void doSaveAs() {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {
		if (input instanceof ZestGraphInput) {
			this.editorInput = (ZestGraphInput) input;
			setSite(site);
			setInput(this.editorInput);
			setPartName("Zest Graph Viewer Part Name");
		} else {
			throw new PartInitException("Input should be of type ZestGraphInput");
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isDirty()
	 */
	@Override
	public boolean isDirty() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		if (this.editorInput != null) {
			if (this.editorInput.getInputFile() != null) {
				final IFile model = this.editorInput.getInputFile();
				String packageName = model.getProjectRelativePath().removeFileExtension().lastSegment().toLowerCase();
				this.setPartName(packageName);
				/*
				 * We have to load the model (to retrieve root elements)
				 * Then we have to build the graph.
				 * 
				 */
				ResourceSet resourceSet = new ResourceSetImpl();
				Resource resource = 
					resourceSet.getResource(
							URI.createPlatformResourceURI(model.getFullPath().toString(), false), true);
								
				this.initializeMethodCallsGraph(resource, parent);				
				resource.unload();
			} else if (this.editorInput.getInputResource() != null) {
				Resource resource = this.editorInput.getInputResource();
				String packageName = "";
				if (resource.getURI() != null) {
					packageName = resource.getURI().trimFileExtension().lastSegment().toLowerCase();
				}
				this.setPartName(packageName);
				this.initializeMethodCallsGraph(resource, parent);
			} else if (this.editorInput.getInputOperation() != null) {
				Operation operation = this.editorInput.getInputOperation();
				String name = operation.getName();
				this.setPartName(name);
				this.initializeMethodCallsGraph(operation, parent);
			}
		}

	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		
	}

	private final List<Operation> parents = new ArrayList<Operation>();
	
	/*
	 * From a uml model, we have to represent method calls as a graph.
	 * First step is to find all root methods
	 * (methods that have no dependencies pointing on them)
	 * Second step is to create links by following dependencies relationships.
	 * Third step is to manage cycles ?
	 * Forth step is to be able to get back to the code from a selected node
	 * Fifth step is to represents methods from real types
	 * (bis: have constraints to restrain possibilities)
	 * 
	 */
	private final  Graph initializeMethodCallsGraph(final Resource resource, final Composite shell) {
		Graph result = new Graph(shell, SWT.NONE);
		result.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				MethodCallsZestGraphEditor.this.zestWidgetSelected(e);
			}
		});

		GraphNode parentNode = new GraphNode(result, SWT.NONE, "Method Calls Graph");
		result.setLayoutAlgorithm(new HorizontalTreeLayoutAlgorithm(LayoutStyles.NO_LAYOUT_NODE_RESIZING), true);

		List<Operation> allOperations = this.getAllOperations(resource);
		
		for (Operation rootElement : this.getRootOperationsForMethodCallsGraph(allOperations)) {
			this.generateMethodCallsNode(result, parentNode, rootElement);
		}
		
		return result;
	}
	
	private final Graph initializeMethodCallsGraph(final Operation rootElement, final Composite shell) {
		
		Graph result = new Graph(shell, SWT.NONE);
		result.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				MethodCallsZestGraphEditor.this.zestWidgetSelected(e);
			}
		});
		
		GraphNode parentNode = new GraphNode(result, SWT.NONE, "Method Calls Graph");
		result.setLayoutAlgorithm(new HorizontalTreeLayoutAlgorithm(LayoutStyles.NO_LAYOUT_NODE_RESIZING), true);

		this.generateMethodCallsNode(result, parentNode, rootElement);
		
		return result;
	}
	
	@SuppressWarnings("unchecked")
	private final void zestWidgetSelected(SelectionEvent e) {
		System.out.println(((Graph) e.widget).getSelection());
		Graph current = (Graph) e.widget;
		List<GraphNode> graphNodes = current.getSelection();
		for (GraphNode node : graphNodes) {
			System.out.println(node.getText());
			System.out.println(((Label) node.getTooltip()).getText());
		}
	}
	
	private final void generateMethodCallsNode(final Graph result, final GraphNode parentNode, final Operation element) {
		GraphNode activeNode = null;
		String name = element.getNamespace().getName() + " :: " + element.getName();
		if (this.parents.contains(element)) {
			// This element has been already managed (recursion)
			activeNode = new GraphNode(result, SWT.NONE, "/recursion/ " + name);
			IFigure tooltip = new Label(this.getQualifiedName(element));
			activeNode.setTooltip(tooltip);
			new GraphConnection(result, SWT.NONE, parentNode, activeNode);
		} else {
			// manage recursion
			this.parents.add(element);
			
			List<Operation> calledMethods = this.getCalledMethods(element);
			if (calledMethods.isEmpty()) {
				activeNode = new GraphNode(result, SWT.NONE, name);
			} else {
				activeNode = new GraphNode(result, SWT.NONE, name + " (" + calledMethods.size() + ")");
			}
			IFigure tooltip = new Label(this.getQualifiedName(element));
			activeNode.setTooltip(tooltip);
			new GraphConnection(result, SWT.NONE, parentNode, activeNode);
			for (Operation child : calledMethods) {
				this.generateMethodCallsNode(result, activeNode, child);
			}
			// manage recursion
			this.parents.remove(element);
		}
	}
	
	private final String getQualifiedName(Operation operation) {
		StringBuilder result = new StringBuilder();
		// reverse order ...
		for (int index = operation.allNamespaces().size() - 1; index >= 0; index--) {
			NamedElement element = operation.allNamespaces().get(index);
			if ((element instanceof Model) == false) {
				result.append(element.getName());
				result.append(".");
			}
		}
		result.append(operation.getName());
		return result.toString();
	}
	private final List<Operation> getAllOperations(Resource resource) {
		List<Operation> result = new ArrayList<Operation>();
		TreeIterator<EObject> iterator = resource.getAllContents();
		while (iterator.hasNext()) {
			EObject object = iterator.next();
			if (object instanceof Operation) {
				Operation operation = (Operation) object;
				if (operation.getName().equalsIgnoreCase("dummy") == false) {
					result.add(operation);
				}
			}
		}
		return result;
	}
	
	private final List<Operation> getRootOperationsForMethodCallsGraph(List<Operation> elements) {
		List<Operation> result = new ArrayList<Operation>();
		List<Dependency> methodCalls = new ArrayList<Dependency>();
		for (Operation element : elements) {
			methodCalls.addAll(element.getClientDependencies());
		}
		for (Operation element : elements) {
			boolean root = true;
			for (Dependency methodCall : methodCalls) {
				if (methodCall.getSuppliers().contains(element)) {
					root = false;
				}
			}
			if (root) {
				result.add(element);
			}
		}
		return result;
	}
	/*
	 * We have the collection of dependencies.
	 * Yet, we have also to sort this collection by its order in call sequence
	 */
	private final List<Operation> getCalledMethods(Operation parent) {
		List<Operation> result = new ArrayList<Operation>();
		List<Dependency> methodCalls = parent.getClientDependencies();
		Collections.sort(methodCalls, new Comparator<Dependency>() {
			private final Integer extractCallRank(Dependency dependency) {
				Integer result = 0;
				String name = dependency.getName();
				if (name != null) {
					String[] parts = name.split(" ");
					String number = parts[parts.length - 1];
					
					result = Integer.valueOf(number);
				}
				return result;
			}
			public int compare(Dependency o1, Dependency o2) {
				return this.extractCallRank(o1).compareTo(this.extractCallRank(o2));
			}});
		
 		for (Dependency methodCall : methodCalls) {
			for (NamedElement callee : methodCall.getSuppliers()) {
				if (callee instanceof Operation) {
					result.add((Operation) callee);
				}
			}
		}
		return result;
	}
}
