/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.editor;

import java.awt.Frame;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JComponent;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.gmt.modisco.java.AbstractMethodDeclaration;
import org.eclipse.gmt.modisco.java.AbstractMethodInvocation;
import org.eclipse.gmt.modisco.java.Block;
import org.eclipse.gmt.modisco.java.BodyDeclaration;
import org.eclipse.gmt.modisco.java.ClassDeclaration;
import org.eclipse.gmt.modisco.java.TypeAccess;
import org.eclipse.gmt.modisco.java.TypeDeclaration;
import org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.converter.PrefuseGraphContainerForJava;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import prefuse.data.Graph;
import prefuse.data.Node;
import prefuse.data.Table;
import prefuse.data.Tree;


/**
 * @author Gabriel Barbier
 *
 */
public class NewMethodCallsPrefuseGraphEditor extends EditorPart {

	private static class MethodInvocationComparator implements Comparator<AbstractMethodInvocation>  {

		public int compare(AbstractMethodInvocation invocation1,
				AbstractMethodInvocation invocation2) {
			int result = -1;
			/*
			 * Initially, I used location in file
			 * to sort invocations, but needed information
			 * is no longer retained in java model ...
			 * 
			 * Perhaps could I use index of parent element
			 * which is directly contained in block statement
			 * of the declaring method ?
			 */
			Block rootBlock1 = this.getRootBlock(invocation1);
			Block rootBlock2 = this.getRootBlock(invocation2);
			if ((rootBlock1 == null) || (rootBlock2 == null)) {
				result = 0;
			} else {
				int index1 = this.computeIndex(invocation1, rootBlock1);
				int index2 = this.computeIndex(invocation2, rootBlock2);
				/*
				 * specific case:
				 * both invocations have the same index
				 * and are contained in the same block.
				 * 
				 * So we have to retrieve first common parent block
				 * and use it to compute index ...
				 */
				if ((index1 == index2) && (rootBlock1 == rootBlock2)) {
					Block commonBlock = this.getFirstCommonParentBlock(invocation1, invocation2, rootBlock1);
					index1 = this.computeIndex(invocation1, commonBlock);
					index2 = this.computeIndex(invocation2, commonBlock);
				}

				result = Integer.valueOf(index1).compareTo(Integer.valueOf(index2));
			}
			return result;
		}
		
		private final int computeIndex(EObject element, Block rootBlock) {
			int result = -2;
			if (element.eContainer() == rootBlock) {
				result = rootBlock.getStatements().indexOf(element);
			} else {
				result = this.computeIndex(element.eContainer(), rootBlock);
			}
			return result;
		}
		
		private final Block getRootBlock(EObject element) {
			Block result = null;
			if (element != null) {
				if (element instanceof AbstractMethodDeclaration) {
					result = ((AbstractMethodDeclaration) element).getBody();
				} else {
					result = this.getRootBlock(element.eContainer());
				}
			}
			return result;
		}
		/*
		 * To retrieve the first common parent block of two elements
		 * lets start from an example:
		 * block A contains another block B (with other statements)
		 * block B contains two blocks C and D which contains respectively element1 and element2
		 * 
		 * Here the algorithm:
		 * From element1, I get the nearest parent block
		 * Then I will iterate on all parent blocks of element2 and test identity
		 * If not found, I have to iterate on parent block of element1
		 * And test again on all parent blocks of element2
		 * ...
		 */
		private final Block getFirstCommonParentBlock(EObject sourceElement, EObject element2, Block stopper) {
			Block result = this.getParentBlock(sourceElement);
			if (result != stopper) {
				if (this.isParentBlock(element2, result, stopper) == false) {
					result = this.getFirstCommonParentBlock(result, element2, stopper);
				} // else, result is the first common parent block
			}
			return result;
		}
		private final boolean isParentBlock(EObject element, Block target, Block stopper) {
			boolean result = false;
			Block parent = this.getParentBlock(element);
			if (parent != stopper) {
				if (parent == target) {
					result = true;
				} else {
					result = this.isParentBlock(parent, target, stopper);
				}
			}
			return result;
		}
		private final Block getParentBlock(EObject element) {
			Block result = null;
			if (element != null) {
				if (element.eContainer() instanceof Block) {
					result = (Block) element.eContainer();
				} else {
					result = this.getParentBlock(element.eContainer());
				}
			}
			return result;
		}
	}
	public static final String EditorID = "org.eclipse.gmt.modisco.usecase.modelfilter.methodcalls.newprefuseeditor.EditorID";
	
	private NewPrefuseGraphInput editorInput;
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public void doSave(IProgressMonitor monitor) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	@Override
	public void doSaveAs() {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {
		if (input instanceof NewPrefuseGraphInput) {
			this.editorInput = (NewPrefuseGraphInput) input;
			setSite(site);
			setInput(this.editorInput);
			setPartName("Prefuse Graph Viewer Part Name");
		} else {
			throw new PartInitException("Input should be of type PrefuseGraphInput");
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isDirty()
	 */
	@Override
	public boolean isDirty() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		if (this.editorInput != null) {
			Composite composite = new Composite(parent, SWT.EMBEDDED | SWT.NO_BACKGROUND);
		    Frame frame = SWT_AWT.new_Frame(composite);
		    Graph graph = null;
		    if (this.editorInput.getJavaxmiFile() != null) {
				final IFile model = this.editorInput.getJavaxmiFile();
				String packageName = model.getProjectRelativePath().removeFileExtension().lastSegment().toLowerCase();
				this.setPartName(packageName);
				/*
				 * We have to load the model (to retrieve root elements)
				 * Then we have to build the graph.
				 * 
				 */
				ResourceSet resourceSet = new ResourceSetImpl();
				Resource resource = 
					resourceSet.getResource(
							URI.createPlatformResourceURI(model.getFullPath().toString(), false), true);
								
				graph = this.initializeMethodCallsGraph(resource);
				
				resource.unload();
			} else if (this.editorInput.getJavaxmiResource() != null) {
				Resource resource = this.editorInput.getJavaxmiResource();
				String packageName = "";
				if (resource.getURI() != null) {
					packageName = resource.getURI().trimFileExtension().lastSegment().toLowerCase();
				}
				this.setPartName(packageName);
				graph = this.initializeMethodCallsGraph(resource);
			} else if (this.editorInput.getInputOperation() != null) {
				AbstractMethodDeclaration operation = this.editorInput.getInputOperation();
				String name = operation.getName();
				this.setPartName(name);
				graph = this.initializeMethodCallsGraph(operation);
			}
		    JComponent treeview;
	        treeview = PrefuseGraphContainerForJava.getInstance().initializeTreeViewContainer(graph, this.nameAttribute, null);
	        frame.add(treeview);

		}

	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		
	}

	private final String nameAttribute = "name";
	private final List<AbstractMethodDeclaration> parents = new ArrayList<AbstractMethodDeclaration>();
	
	/*
	 * From a uml model, we have to represent method calls as a graph.
	 * First step is to find all root methods
	 * (methods that have no dependencies pointing on them)
	 * Second step is to create links by following dependencies relationships.
	 * Third step is to manage cycles ?
	 * Forth step is to be able to get back to the code from a selected node
	 * Fifth step is to represents methods from real types
	 * (bis: have constraints to restrain possibilities)
	 * 
	 */
	private final  Tree initializeMethodCallsGraph(Resource resource) {
		Tree result = new Tree();
		Node parentNode = this.intializeMethodCallsGraph(result, resource);
		
		List<AbstractMethodDeclaration> allOperations = this.getAllOperations(resource);
		
		for (AbstractMethodDeclaration rootElement : this.getRootOperationsForMethodCallsGraph(allOperations)) {
			this.generateMethodCallsNode(result, parentNode, rootElement);
		}
		return result;
	}
	
	private final Tree initializeMethodCallsGraph(AbstractMethodDeclaration rootElement) {
		Tree result = new Tree();
		Node parentNode = this.intializeMethodCallsGraph(result, rootElement.eResource());
		this.generateMethodCallsNode(result, parentNode, rootElement);
		return result;
	}
	
	private final Node intializeMethodCallsGraph(final Tree result, final Resource resource) {
		/*
		 * compute the graph of method calls
		 */
		this.computeMethodCallsHierarchy(resource);
		/*
		 * initialise prefuse graph informations
		 */
		Table nodes = result.getNodeTable();
		Node activeNode = null;
		
		// add attribute name for each node
		nodes.addColumn(this.nameAttribute, String.class);
		nodes.addColumn(PrefuseGraphContainerForJava.javaOperation, AbstractMethodDeclaration.class);
		nodes.addColumn(PrefuseGraphContainerForJava.jdtProject, IJavaProject.class);
		
		// add root element
		activeNode = result.addRoot();
		activeNode.set(this.nameAttribute, "Method Calls Graph");
		
		return activeNode;
	}
	
	private final void generateMethodCallsNode(final Tree result, final Node parentNode, final AbstractMethodDeclaration element) {
		Node activeNode = result.addChild(parentNode);
		activeNode.set(PrefuseGraphContainerForJava.javaOperation, element);
		activeNode.set(PrefuseGraphContainerForJava.jdtProject, this.editorInput.getJavaProject());
		
		String parentName = "";
		if (element.getAbstractTypeDeclaration() != null) {
			parentName = element.getAbstractTypeDeclaration().getName();
		}
		String name = parentName + " :: " + element.getName();
		if (this.parents.contains(element)) {
			// This element has been already managed (recursion)
			activeNode.set(this.nameAttribute, "/recursion/ " + name);
		} else {
			// manage recursion
			this.parents.add(element);
			
			List<AbstractMethodDeclaration> calledMethods = this.getCalledMethods(element);
			if (calledMethods.isEmpty()) {
				activeNode.set(this.nameAttribute, name);
				/*
				 * Check that parent type is an interface ?
				 * if yes, we would like to continue the graph
				 * with implementations class/methods
				 * 
				 * A better check could be to test body of the method:
				 * no body = abstract method
				 * All cases are handled, no matter if it is an interface
				 * or an abstract class !
				 */
				if (element.getBody() == null) {
					TypeDeclaration parentType = this.getTypeDeclaration(element);
					for (TypeDeclaration subtype : this.getAllSubTypes(parentType)) {
						/*
						 * we have to retrieve corresponding method
						 * same name, same number of parameters
						 * same name of parameters ? -> may change
						 * same type of parameters ? may be a subtype
						 */
						for (BodyDeclaration body : subtype.getBodyDeclarations()) {
							if (body instanceof AbstractMethodDeclaration) {
								AbstractMethodDeclaration subMethod = (AbstractMethodDeclaration) body;
								if (element.getName().equals(body.getName())) {
									if (element.getParameters().size() == subMethod.getParameters().size()) {
										this.generateMethodCallsNode(result, activeNode, subMethod);
									}
								}
							}
						}
					}
				}
			} else {
				activeNode.set(this.nameAttribute, name + " (" + calledMethods.size() + ")");
			}
			for (AbstractMethodDeclaration child : calledMethods) {
				this.generateMethodCallsNode(result, activeNode, child);
			}
			// manage recursion
			this.parents.remove(element);
		}
	}
	
	private final Set<TypeDeclaration> getAllSubTypes(TypeDeclaration contextClass) {
		Set<TypeDeclaration> result = new HashSet<TypeDeclaration>();
		if (contextClass != null) {
			TreeIterator<EObject> content = contextClass.eResource().getAllContents();
			while (content.hasNext()) {
				EObject eObject = content.next();
				if (eObject instanceof TypeDeclaration) {
					TypeDeclaration currentClassDeclaration = (TypeDeclaration) eObject;
					if (isSuperTypeOf(contextClass, currentClassDeclaration)) {
						result.add(currentClassDeclaration);
						result.addAll(this.getAllSubTypes(currentClassDeclaration));
					}
				}
			}
		}
		return result;
	}
	/**
	 * @param contextClass
	 * @return
	 */
	private final boolean isSuperTypeOf(TypeDeclaration self,
			TypeDeclaration typeDeclaration) {
		if (typeDeclaration.getSuperInterfaces().contains(self)) {
			return true;
		}
		for (TypeAccess superTypeAccess : typeDeclaration.getSuperInterfaces()) {
			if (superTypeAccess.getType() instanceof TypeDeclaration) {
				TypeDeclaration superType = (TypeDeclaration) superTypeAccess
						.getType();
				if (superType == self || isSuperTypeOf(self, superType)) {
					return true;
				}
			}
		}
		if (typeDeclaration instanceof ClassDeclaration) {
			ClassDeclaration classDeclaration = (ClassDeclaration) typeDeclaration;
			if (classDeclaration.getSuperClass() != null && classDeclaration.getSuperClass().getType() == self) {
				return true;
			}
			if (classDeclaration.getSuperClass()!=null && classDeclaration.getSuperClass().getType() instanceof TypeDeclaration) {
				TypeDeclaration superType = (TypeDeclaration) classDeclaration
						.getSuperClass().getType();
				if (isSuperTypeOf(self, superType)) {
					return true;
				}
			}
		}
		return false;
	}
	
	private final List<AbstractMethodDeclaration> getAllOperations(Resource resource) {
		List<AbstractMethodDeclaration> result = new ArrayList<AbstractMethodDeclaration>();
		TreeIterator<EObject> iterator = resource.getAllContents();
		while (iterator.hasNext()) {
			EObject object = iterator.next();
			if (object instanceof AbstractMethodDeclaration) {
				AbstractMethodDeclaration operation = (AbstractMethodDeclaration) object;
				if (operation.getName().equalsIgnoreCase("dummy") == false) {
					result.add(operation);
				}
			}
		}
		return result;
	}
	private List<AbstractMethodInvocation> allInvocations;
	private final List<AbstractMethodInvocation> getAllInvocations(Resource resource) {
		if (this.allInvocations == null) {
			this.allInvocations = new ArrayList<AbstractMethodInvocation>();
			TreeIterator<EObject> iterator = resource.getAllContents();
			while (iterator.hasNext()) {
				EObject object = iterator.next();
				if (object instanceof AbstractMethodInvocation) {
					AbstractMethodInvocation invocation = (AbstractMethodInvocation) object;
					this.allInvocations.add(invocation);
				}
			}
		}
		return this.allInvocations;
	}
	private final List<AbstractMethodDeclaration> getRootOperationsForMethodCallsGraph(List<AbstractMethodDeclaration> elements) {
		List<AbstractMethodDeclaration> result = new ArrayList<AbstractMethodDeclaration>();
		for (AbstractMethodDeclaration element : elements) {
			boolean root = element.getUsages().isEmpty();
			if (root) {
				result.add(element);
			}
		}
		return result;
	}
	/*
	 * We have the collection of dependencies.
	 * Yet, we have also to sort this collection by its order in call sequence
	 * 
	 * If we want to get all the called methods from this method declaration,
	 * we have to select all usages which are contained by this method declaration.
	 * It means all instances of AbstractMethodInvocation
	 *  
	 */
	private final List<AbstractMethodDeclaration> getCalledMethods(AbstractMethodDeclaration parent) {
		List<AbstractMethodDeclaration> result = new ArrayList<AbstractMethodDeclaration>();
		
		if (this.mapInvocations.containsKey(parent)) {
			result = this.mapInvocations.get(parent);
		}
		
		return result;
	}
	
	private final Map<AbstractMethodDeclaration, List<AbstractMethodDeclaration>> mapInvocations = new HashMap<AbstractMethodDeclaration, List<AbstractMethodDeclaration>>();
	private final void computeMethodCallsHierarchy(Resource resource) {
		/*
		 * Idea is to build a map with needed informations
		 * key = MethodDeclaration
		 * List of values are called methods
		 * So perhaps do we only have to retrieve instances of MethodDeclaration
		 * instead of getting method declarations and invocations.
		 * 
		 * Second step is to add links to real types used when
		 * a called method has been defined in an interface.
		 * First try: on all known implementation classes
		 */
		List<AbstractMethodInvocation> invocations = this.getAllInvocations(resource);
		Collections.sort(invocations, new MethodInvocationComparator());
		
		for (AbstractMethodInvocation invocation : invocations) {
				AbstractMethodDeclaration invoker = this.getInvoker(invocation);
				List<AbstractMethodDeclaration> invokeds = this.mapInvocations.get(invoker);
				if (invokeds == null) {
					invokeds = new ArrayList<AbstractMethodDeclaration>();
					this.mapInvocations.put(invoker, invokeds);
				}
				invokeds.add(invocation.getMethod());
		}
	}
	
	
	private final AbstractMethodDeclaration getInvoker(EObject element) {
		AbstractMethodDeclaration result = null;
		if (element != null) {
			if (element instanceof AbstractMethodDeclaration) {
				result = (AbstractMethodDeclaration) element;
			} else {
				result = this.getInvoker(element.eContainer());
			}
		}
		return result;
	}
	
	private final TypeDeclaration getTypeDeclaration(EObject element) {
		TypeDeclaration result = null;
		if (element != null) {
			if (element instanceof TypeDeclaration) {
				result = (TypeDeclaration) element;
			} else {
				result = this.getTypeDeclaration(element.eContainer());
			}
		}
		return result;
	}
}
