/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.usecases.modelfilter.actions;

import java.util.Map;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.infra.discoverymanager.AbstractDiscovererImpl;
import org.eclipse.gmt.modisco.infra.discoverymanager.Discoverer;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameter;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameterDirectionKind;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameterImpl;
import org.eclipse.gmt.modisco.usecases.modelfilter.converter.JavaConverter;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

/**
 * @author Gabriel Barbier
 * 
 */
public class DiscoverUmlModelWithDependenciesFromJavaFile extends AbstractDiscovererImpl implements Discoverer {

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.gmt.modisco.infra.discoverymanager.DiscovererInterface#applyTo
	 * (java.lang.Object)
	 */
	public boolean isApplicableTo(final Object source) {
		boolean result = false;
		if (source instanceof ICompilationUnit) {
			result = true;
		}
		return result;
	}

	public String toString() {
		return "Discover UML model with extended dependencies from Java class"; //$NON-NLS-1$
	}

	private final DiscoveryParameterImpl recursionLevel;
	public DiscoveryParameter getRecursionLevelParameter() {
		return this.recursionLevel;
	}
	
	public DiscoverUmlModelWithDependenciesFromJavaFile() {
		super();
		this.recursionLevel = new DiscoveryParameterImpl("Recursion level", DiscoveryParameterDirectionKind.in, Integer.class, false); //$NON-NLS-1$
		this.getBasicDiscovererParameters().add(this.recursionLevel);
	}

	
	public void discoverElement(final Object source,
			final Map<DiscoveryParameter, Object> parameters) {
		// retrieval of Java file
		final ICompilationUnit javaFile = (ICompilationUnit) source;
		
		final String elementName = javaFile.getPath().removeFileExtension().lastSegment();
		final IJavaProject javaProject = javaFile.getJavaProject();
		
		if (parameters.containsKey(this.getRecursionLevelParameter()) == false) {
			final IInputValidator validator = new IInputValidator() {
				public String isValid(String newText) {
					String result = null; // no error status = null
					// we have to convert text into integer value
					try {
						int value = Integer.parseInt(newText);
						if (value < 0) {
							result = Messages.DiscoverUmlModelWithDependenciesFromJavaFile_positiveInteger;
						}
					} catch (NumberFormatException nfe) {
						result = Messages.DiscoverUmlModelWithDependenciesFromJavaFile_recursionLevelAsAnInteger;
					}
					return result;
				}			
			};

			// access UI resources so that we can obtain the current eclipse
			// workbench activePage
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					InputDialog dialog = new InputDialog(new Shell(), Messages.DiscoverUmlModelWithDependenciesFromJavaFile_recursionLevelTitle,
							Messages.DiscoverUmlModelWithDependenciesFromJavaFile_recursionLevelMessage, "1", validator); //$NON-NLS-2$

					if (dialog.open() == Window.OK) {
						int level = Integer.parseInt(dialog.getValue());
						parameters.put(DiscoverUmlModelWithDependenciesFromJavaFile.this.getRecursionLevelParameter(), level);
					} else {
						// default value ?
						parameters.put(DiscoverUmlModelWithDependenciesFromJavaFile.this.getRecursionLevelParameter(), 1);
					}
				}
			});
			
		}
		final int level = (Integer) parameters.get(this.getRecursionLevelParameter());
		JavaConverter converter = new JavaConverter(javaProject, javaFile);
		Resource result = converter.convertJavaProjectToUml2(elementName, level);
		parameters.put(this.getTargetModelParameter(), result);
	}

}
