/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.usecases.modelfilter.converter;

import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import java.util.logging.Logger;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.operations.OperationHistoryFactory;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.workspace.WorkspaceEditingDomainFactory;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.diagram.core.services.ViewService;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.gmt.modisco.common.core.logging.AtlLogHandler;
import org.eclipse.gmt.modisco.common.core.utils.ModelUtils;
import org.eclipse.gmt.modisco.common.core.utils.Tools;
import org.eclipse.gmt.modisco.java.actions.DiscoverJavaModelFromJavaProject;
import org.eclipse.gmt.modisco.kdm.uml2converter.KDMtoUML2Converter;
import org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface;
import org.eclipse.gmt.modisco.usecases.modelfilter.modelfilterproperties.ModelFilterPropertiesFactory;
import org.eclipse.gmt.modisco.usecases.modelfilter.modelfilterproperties.SelectedElement;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.m2m.atl.common.ATLLogger;
import org.eclipse.uml2.diagram.clazz.edit.parts.PackageEditPart;
import org.eclipse.uml2.diagram.clazz.part.Messages;
import org.eclipse.uml2.diagram.clazz.part.UMLDiagramEditorPlugin;
import org.eclipse.uml2.uml.Model;

/**
 * @author Gabriel Barbier
 * 
 */
public class JavaConverter {

	private final ModelFilterPropertiesFactory factory = ModelFilterPropertiesFactory.eINSTANCE;
	
	private final Logger logger = Logger.getLogger(ATLLogger.LOGGER_ID);
	private final IJavaProject javaProject;
	private final ICompilationUnit javaFile;
	private final IPath path;
	private final IPath j2se5ModelPath;
	private final URL kdmTransformation;
	private final URL umlTransformation;

	public JavaConverter(IJavaProject javaProjectParameter,
			ICompilationUnit javaFileParameter) {
		this.javaProject = javaProjectParameter;
		this.javaFile = javaFileParameter;

		this.path = this.javaFile.getPath().removeFileExtension();
		
		this.j2se5ModelPath = this.javaProject.getProject().getLocation().addTrailingSeparator().append(this.javaProject.getElementName()).addFileExtension("j2se5");
		// initialisation des transformations
		this.kdmTransformation = this.getClass().getResource(
				"transformations/J2se5FiltersToKdm.asm");
		this.umlTransformation = this.getClass().getResource(
				"transformations/FiltersAtUml.asm");
	}

	public Resource convertJavaProjectToUml2(String elementName, int dependenciesLevel) {
		Resource result = null;
		
		// initialisation of properties model
		SelectedElement selectedElement = this.factory.createSelectedElement();
		selectedElement.setElementName(elementName);
		selectedElement.setDependenciesLevel(dependenciesLevel);

		final URI propertiesUri = URI.createPlatformResourceURI(
				this.path.addFileExtension("modelfilterproperties").toString(), false);
		try {
			Tools.save(selectedElement, propertiesUri);
		} catch (IOException e2) {
			e2.printStackTrace();
		}

		// transformations chain
		AtlLogHandler logHandler = new AtlLogHandler(
				JavaConverter.this.javaProject.getProject().getLocation().addTrailingSeparator().toString() + 
				JavaConverter.this.path.removeFirstSegments(1).addFileExtension("log").toString());
		JavaConverter.this.logger.addHandler(logHandler);
		try {
			/* first step */
			URI kdmModelUri = URI.createPlatformResourceURI(JavaConverter.this.path
					.addFileExtension("kdm").toString(), false);
			
			/*
			 * We will make a specific treatment to improve performances:
			 * global reverse of project should be done only once.
			 * So, if it already exists, we will use the kdm model
			 * located on the root of the project, other else we have
			 * to create it. 
			 * 
			 */
			// for compatibility reason, we have to check that java model
			// has been correctly loaded ...
			
			URI j2se5ModelUri = URI.createFileURI(JavaConverter.this.j2se5ModelPath.toString());
			if (JavaConverter.this.j2se5ModelPath.toFile().exists()) {
				try {
					ResourceSet resourceSet = new ResourceSetImpl();
					resourceSet.getResource(j2se5ModelUri, true);
				} catch (Exception ioe) {
					JavaConverter.this.j2se5ModelPath.toFile().delete();
				}
			}
			if (JavaConverter.this.j2se5ModelPath.toFile().exists() == false) {
				DiscoverJavaModelFromJavaProject javaDiscoverer = new DiscoverJavaModelFromJavaProject();
				javaDiscoverer.discoverElement(this.javaProject, j2se5ModelUri);
			}

			/* second step */
			//					Resource kdmModel = null;
			WesternGecoConverter wgConverter = new WesternGecoConverter();
			//					kdmModel = 
			wgConverter.getKDMModelFromJ2SE5ModelWithCustomTransformation(
					j2se5ModelUri, propertiesUri,
					JavaConverter.this.kdmTransformation,
					kdmModelUri);
			//					Tools.saveModel(kdmModel, kdmModelUri);

			/* third step */
			KdmToUmlConverterInterface converter = new KDMtoUML2Converter();
			URI outputURI = URI.createPlatformResourceURI(JavaConverter.this.path
					.addFileExtension("uml").toString(), false);
			converter.getUML2ModelFromKDMModelWithCustomTransformation(
					kdmModelUri, false,
					JavaConverter.this.umlTransformation, outputURI);
			//					converter.getUML2ModelFromKDMModel(kdmModel, false, outputURI);
			/* forth step, management of bidirectional links */
			wgConverter.manageBidirectionalAssociations(outputURI);

			// load result resources ...
			result = ModelUtils.load(outputURI);
			
			/* fifth step, creation of diagram. */
			URI diagramURI = URI.createPlatformResourceURI(this.path
					.addFileExtension("umlclass").toString(), false);

			TransactionalEditingDomain editingDomain = WorkspaceEditingDomainFactory.INSTANCE.createEditingDomain();
			ResourceSet resourceSet = editingDomain.getResourceSet();
			final Resource modelResource = resourceSet.getResource(outputURI, true);
			EObject rootObject = modelResource.getContents().get(0);
			for (EObject eObject : rootObject.eContents()) {
				if (eObject instanceof Model) {
					if (((Model) eObject).getName().equals(this.javaProject.getElementName())) {
						rootObject = eObject;
					}
				}
			}
			final EObject diagramRoot = rootObject;

			final Resource diagramResource = editingDomain.getResourceSet().createResource(diagramURI);
			final String diagramName = diagramURI.lastSegment();

			final Diagram diagram = ViewService.createDiagram(diagramRoot, PackageEditPart.MODEL_ID, UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
			if (diagram != null) {
				AbstractTransactionalCommand command = new AbstractTransactionalCommand(editingDomain, Messages.UMLDiagramEditorUtil_CreateDiagramCommandLabel, Collections.EMPTY_LIST) {

					protected CommandResult doExecuteWithResult(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
						diagramResource.getContents().add(diagram);
						diagram.setName(diagramName);
						diagram.setElement(diagramRoot);
						return CommandResult.newOKCommandResult();
					}
				};
				IStatus status = OperationHistoryFactory.getOperationHistory().execute(command, new SubProgressMonitor(new NullProgressMonitor(), 1), null);
				if (status.isOK()) {
					Tools.saveModel(diagramResource, diagramURI);
				}
			}


		} catch (Exception e) {
			e.printStackTrace();
		}

		JavaConverter.this.logger.removeHandler(logHandler);
		logHandler.close();
		try {
			JavaConverter.this.javaFile.getParent().getCorrespondingResource().refreshLocal(1, new NullProgressMonitor());
			JavaConverter.this.javaProject.getCorrespondingResource()
			.refreshLocal(1, new NullProgressMonitor());
		} catch (Exception e1) {
			// give up
		}

		return result;		
	}
}
