/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.usecases.modelfilter.converter;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.gmt.modisco.common.core.utils.ModelUtils;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameter;
import org.eclipse.gmt.modisco.java.actions.DefaultDiscoverer;
import org.eclipse.gmt.modisco.java.actions.DiscoverJavaModelFromJavaProject;
import org.eclipse.gmt.modisco.java.actions.DiscoverKdmModelFromJavaProject;
import org.eclipse.gmt.modisco.kdm.uml2converter.KDMtoUML2Converter;
import org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.m2m.atl.core.ATLCoreException;

/**
 * @author Gabriel Barbier
 * 
 */
public class MethodCallsJavaConverter {

	private final IJavaProject javaProject;
	private final IPath path;
	private URL umlTransformation;

	private URL getUmlTransformation() {
		if (this.umlTransformation == null) {
			this.umlTransformation = this.getClass().getResource(
			"transformations/KDMtoUMLwithMethodCall.asm");
		}
		return this.umlTransformation;
	}
	
	public MethodCallsJavaConverter(IJavaProject javaProjectParameter)
			throws JavaModelException {
		this.javaProject = javaProjectParameter;

		this.path = this.javaProject.getCorrespondingResource().getLocation()
				.addTrailingSeparator().append(
						this.javaProject.getElementName());
		
	}
	public Resource convertJavaProjectToJavaModel() {
		Resource result = null;

		final IPath javaModelPath = this.path.addFileExtension("javaxmi");
		
		/* first step */
		URI javaModelUri = URI.createFileURI(javaModelPath.toString());
		
		/*
		 * We will make a specific treatment to improve performances:
		 * global reverse of project should be done only once.
		 * So, if it already exists, we will use the java model
		 * located on the root of the project, other else we have
		 * to create it. 
		 * 
		 */
		// for compatibility reason, we have to check that java model
		// has been correctly loaded ...
		if (javaModelPath.toFile().exists()) {
			try {
				ResourceSet resourceSet = new ResourceSetImpl();
				result = resourceSet.getResource(javaModelUri, true);
			} catch (Exception ioe) {
				javaModelPath.toFile().delete();
			}
		}
		if (javaModelPath.toFile().exists() == false) {
			DiscoverJavaModelFromJavaProject javaDiscoverer = new DiscoverJavaModelFromJavaProject();
			Map<DiscoveryParameter, Object> parameters = new HashMap<DiscoveryParameter, Object>();
			parameters.put(DefaultDiscoverer.PARAMETER_SILENT_MODE, true);
			javaDiscoverer.discoverElement(this.javaProject, parameters);
			result = (Resource) parameters.get(DefaultDiscoverer.PARAMETER_TARGET_RESOURCE);
			
		}
		return result;
	}
	
	public Resource convertJavaProjectToUml2() throws JavaModelException,
			ATLCoreException, IOException {
		Resource result = null;

		final IPath kdmModelPath = this.path.addFileExtension("kdm");
		final IPath umlModelPath = this.javaProject.getCorrespondingResource()
				.getLocation().addTrailingSeparator().append(
						this.javaProject.getElementName() + "MethodCalls")
				.addFileExtension("uml");

		// transformations chain

		/* first step */
		URI kdmModelUri = URI.createFileURI(kdmModelPath.toString());
		
		/*
		 * We will make a specific treatment to improve performances:
		 * global reverse of project should be done only once.
		 * So, if it already exists, we will use the kdm model
		 * located on the root of the project, other else we have
		 * to create it. 
		 * 
		 */
		// for compatibility reason, we have to check that java model
		// has been correctly loaded ...
		if (kdmModelPath.toFile().exists()) {
			try {
				ResourceSet resourceSet = new ResourceSetImpl();
				resourceSet.getResource(kdmModelUri, true);
			} catch (Exception ioe) {
				kdmModelPath.toFile().delete();
			}
		}
		boolean forceUmlTransformation = false;
		if (kdmModelPath.toFile().exists() == false) {
			forceUmlTransformation = true;
			DiscoverKdmModelFromJavaProject javaDiscoverer = new DiscoverKdmModelFromJavaProject();
			javaDiscoverer.discoverElement(this.javaProject, kdmModelUri);
		}

		/* second step */
		KdmToUmlConverterInterface converter = new KDMtoUML2Converter();
		URI outputURI = URI.createFileURI(URI.decode(umlModelPath.toString()));
		forceUmlTransformation = forceUmlTransformation
				|| (umlModelPath.toFile().exists() == false);
		if (forceUmlTransformation) {
			converter.getUML2ModelFromKDMModelWithCustomTransformation(
					kdmModelUri, false, this.getUmlTransformation(), outputURI);

			/* third step, management of bidirectional links */
			WesternGecoConverter wgConverter = new WesternGecoConverter();
			wgConverter.manageBidirectionalAssociations(outputURI);
		}

		try {
			this.javaProject.getCorrespondingResource().refreshLocal(1,
					new NullProgressMonitor());
		} catch (Exception e1) {
			// give up
		}

		// load result resources ...
		result = ModelUtils.load(outputURI);
		return result;
	}
}
