/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 */

package org.eclipse.gmt.modisco.usecases.modelfilter.converter;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.utils.UriUtils;
import org.eclipse.m2m.atl.common.ATLLaunchConstants;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.core.IExtractor;
import org.eclipse.m2m.atl.core.IInjector;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.ModelFactory;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.core.service.CoreService;
import org.eclipse.m2m.atl.core.ui.vm.asm.ASMModelWrapper;
import org.eclipse.m2m.atl.drivers.emf4atl.ASMEMFModel;
import org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel;

/**
 * @author Gabriel Barbier
 * 
 */
public class WesternGecoConverter {

	// resources location
	private final String kdmMMUri = "http://www.eclipse.org/MoDisco/kdm/action";
	private final String j2se5MMUri = "http://www.eclipse.org/MoDisco/J2SE5";
	private final String umlMMUri = "http://www.eclipse.org/uml2/2.1.0/UML";
	private final String csharpMMUri = "http://www.eclipse.org/MoDisco/CSharp";
	private final String propertiesMMUri = "http://www.eclipse.org/Modelplex/modelfilterproperties";

	public Resource getKDMModelFromJ2SE5ModelWithCustomTransformation(
			URI j2se5SourceModelUri, URI propertiesUri,
			final URL transformation, URI kdmTargetModelUri) throws IOException, ATLCoreException {
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String j2se5MetaModelName = "j2se5";
		modelHandlers.put(j2se5MetaModelName, "EMF");
		String kdmMetaModelName = "kdm";
		modelHandlers.put(kdmMetaModelName, "EMF");
    	String propertiesMetaModelName = "Filter";
    	modelHandlers.put(propertiesMetaModelName, "EMF");
    	
    	Map<String, Object> launcherOptions = new HashMap<String, Object>();
    	launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);
		
		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);

		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		// loading of meta model Java
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(j2se5MetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", j2se5MetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.j2se5MMUri); //$NON-NLS-1$
		IReferenceModel j2se5MM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(j2se5MM, this.j2se5MMUri);
		
		// lodaing of model Java
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(j2se5SourceModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(j2se5MM, modelOptions);
		injector.inject(input, UriUtils.toString(j2se5SourceModelUri));
		launcher.addInModel(input, inModelName, j2se5MetaModelName);

		// loading of meta model Properties
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(propertiesMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", propertiesMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.propertiesMMUri); //$NON-NLS-1$
		IReferenceModel propertiesMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(propertiesMM, this.propertiesMMUri);
		
		// loading of model Properties
		modelOptions = new HashMap<String, Object>();
		inModelName = "properties";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(propertiesUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel properties = factory.newModel(propertiesMM, modelOptions);
		injector.inject(properties, UriUtils.toString(propertiesUri));
		launcher.addInModel(properties, inModelName, propertiesMetaModelName);

		// loading of meta model KDM
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.kdmMMUri); //$NON-NLS-1$
		IReferenceModel kdmMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(kdmMM, this.kdmMMUri);
		// loading of model KDM
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(kdmTargetModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(kdmMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, kdmMetaModelName);

		/*
		 * encapsulate the transformation in a new Thread
		 * to get an empty execution stack.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Convert J2SE5 model to KDM") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		extractor.extract(outputInstance, UriUtils.toString(kdmTargetModelUri));
		
		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		}
		return output;
	}

	public Resource getKDMModelFromCSharpModelWithCustomTransformation(
			URI csharpSourceModelUri, URI propertiesUri,
			final URL transformation, URI kdmTargetModelUri) throws IOException, ATLCoreException {
		
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String csharpMetaModelName = "csharp";
		modelHandlers.put(csharpMetaModelName, "EMF");
		String kdmMetaModelName = "kdm";
		modelHandlers.put(kdmMetaModelName, "EMF");
    	String propertiesMetaModelName = "Filter";
    	modelHandlers.put(propertiesMetaModelName, "EMF");
    	
    	Map<String, Object> launcherOptions = new HashMap<String, Object>();
    	launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);
		
		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);

		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		// loading of meta model Java
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(csharpMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", csharpMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.csharpMMUri); //$NON-NLS-1$
		IReferenceModel j2se5MM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(j2se5MM, this.csharpMMUri);
		
		// loading of model Java
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(csharpSourceModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(j2se5MM, modelOptions);
		injector.inject(input, UriUtils.toString(csharpSourceModelUri));
		launcher.addInModel(input, inModelName, csharpMetaModelName);

		// loading of meta model Properties
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(propertiesMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", propertiesMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.propertiesMMUri); //$NON-NLS-1$
		IReferenceModel propertiesMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(propertiesMM, this.propertiesMMUri);
		
		// loading of model Properties
		modelOptions = new HashMap<String, Object>();
		inModelName = "properties";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(propertiesUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel properties = factory.newModel(propertiesMM, modelOptions);
		injector.inject(properties, UriUtils.toString(propertiesUri));
		launcher.addInModel(properties, inModelName, propertiesMetaModelName);

		// loading of meta model KDM
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.kdmMMUri); //$NON-NLS-1$
		IReferenceModel kdmMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(kdmMM, this.kdmMMUri);
		// loading of model KDM
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(kdmTargetModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(kdmMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, kdmMetaModelName);

		/*
		 * encapsulate the transformation in a new Thread
		 * to get an empty execution stack.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Convert CSharp model to KDM") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		extractor.extract(outputInstance, UriUtils.toString(kdmTargetModelUri));
		
		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		}
		return output;
	}

	/**
	 * @param outputURI
	 * @throws ATLCoreException 
	 */
	public void manageBidirectionalAssociations(URI umlSourceUri) throws ATLCoreException {
		final URL transformation = this.getClass().getResource("transformations/UML2Copy.asm");
		final URL superimposedTransformation = this.getClass().getResource(
				"transformations/BidirectionalAssociation.asm");
		
		Map<String, String> modelHandlers = new HashMap<String, String>();
		String umlMetaModelName = "uml";
		modelHandlers.put(umlMetaModelName, "UML2");
		
    	Map<String, Object> launcherOptions = new HashMap<String, Object>();
    	launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);
		
		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);

		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());

		// loading of meta model UML
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(umlMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", umlMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", this.umlMMUri); //$NON-NLS-1$
		IReferenceModel umlMM = factory.newReferenceModel(referenceModelOptions);
		injector.inject(umlMM, this.umlMMUri);
		
		// loading of model UML as input
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "umlInput";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(umlSourceUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(umlMM, modelOptions);
		injector.inject(input, UriUtils.toString(umlSourceUri));
		launcher.addInModel(input, inModelName, umlMetaModelName);

		// loading of model UML as output
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT";
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(umlSourceUri)); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(umlMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, umlMetaModelName);

		/*
		 * encapsulate the transformation in a new Thread
		 * to get an empty execution stack.
		 */
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true");
		options.put("printExecutionTime", "true");

		Job transformationThread = new Job("Additional transformation for Uml") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream(), superimposedTransformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		
		extractor.extract(outputInstance, UriUtils.toString(umlSourceUri));
	}
}
