/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hugo Bruneliere (INRIA) - initial API and implementation
 *    Sbastien Minguet (Mia-Software) - initial API and implementation
 *    Frdric Madiot (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.modelplex.controller.impl;

import java.io.File;
import java.net.URL;
import java.util.logging.Logger;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.common.core.logging.AtlLogHandler;
import org.eclipse.gmt.modisco.java.actions.DiscoverJavaModelFromJavaProject;
import org.eclipse.gmt.modisco.java.actions.DiscoverKdmModelFromJavaProject;
import org.eclipse.gmt.modisco.java.actions.TranslateJavaModelToKdm;
import org.eclipse.gmt.modisco.kdm.uml2converter.KDMtoUML2Converter;
import org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface;
import org.eclipse.gmt.modisco.modelplex.controller.Converter;
import org.eclipse.gmt.modisco.modelplex.model.ModelServices;
import org.eclipse.gmt.modisco.modelplex.model.modelplex.AtlTransformation;
import org.eclipse.gmt.modisco.modelplex.model.modelplex.ModelplexDiscoverer;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.m2m.atl.common.ATLLogger;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

public class ConverterImpl implements Converter {

	private final Logger logger = Logger.getLogger(ATLLogger.LOGGER_ID);
	private final IJavaProject javaProject;
	private IPath path;
	private final ModelServices model;

	public ConverterImpl(IJavaProject javaProjectParameter,
			ModelServices modelParameter) {
		this.javaProject = javaProjectParameter;
		this.model = modelParameter;
		try {
			if (this.javaProject != null) {
				IResource resource = this.javaProject
						.getCorrespondingResource();
				if (resource != null)
					this.path = resource.getLocation().addTrailingSeparator()
							.append(this.javaProject.getElementName());
			}
		} catch (JavaModelException e) {
			e.printStackTrace();
		}
	}

	public void convertJavaProjectToUml2() {
		Job job = new Job("Discovering UML2 model from Java project") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				AtlLogHandler logHandler = new AtlLogHandler(
						ConverterImpl.this.path.addFileExtension("log")
								.toString());
				ConverterImpl.this.logger.addHandler(logHandler);
				try {
					ChainedTransformationLauncher launcher = new ChainedTransformationLauncher();
					/*
					 * tapes :
					 *  1. du modle spcifique au modle KDM, avec
					 * potentiellement une transformation utilisateur
					 *  2.
					 * <optionnel> des transformations sur le modle KDM -> mode
					 * refining ... paramtres input:kdm et output:kdm 
					 * 3. du
					 * modle KDM au modle UML, avec potentiellement une
					 * transformation utilisateur 
					 * 4. <optionnel> des
					 * transformations sur le modle UML -> mode refining ...
					 * paramtres umlOutput:uml, umlInput:uml et kdmInput:kdm
					 */
					ModelplexDiscoverer discoverer = ConverterImpl.this.model
							.getDiscoverer();
					/* premire tape */
					Resource kdmModel = null;
					URI kdmModelUri = URI.createFileURI(ConverterImpl.this.path
							.addFileExtension("kdm").toString());
					if (discoverer.getToKdm().isCustom()) {
						String transformationPath = discoverer.getToKdm()
								.getTransformationPath();
						URL transformation = new File(transformationPath)
								.toURI().toURL();
						DiscoverJavaModelFromJavaProject javaDiscoverer = new DiscoverJavaModelFromJavaProject();
						Resource j2se5Model = javaDiscoverer
								.discoverElement(ConverterImpl.this.javaProject, (URI) null);
						// resource shall have been saved 
						TranslateJavaModelToKdm kdmTranslater = new TranslateJavaModelToKdm();
						kdmModel = kdmTranslater
								.getKDMModelFromJavaModelWithCustomTransformation(
										j2se5Model.getURI(), transformation,
										kdmModelUri);
					} else {
						DiscoverKdmModelFromJavaProject javaDiscoverer = new DiscoverKdmModelFromJavaProject();
						kdmModel = javaDiscoverer.discoverElement(
								ConverterImpl.this.javaProject, kdmModelUri);
					}

					/* deuxime tape */
					for (AtlTransformation atlTransformation : discoverer
							.getToKdm().getChainedTransformations()) {
						System.out.println("Additional transformation for KDM : "
								+ atlTransformation);
						launcher.launchKdmToKdmTransformation(atlTransformation, kdmModelUri, kdmModelUri);
					}

					/* troisime tape */
					KdmToUmlConverterInterface converter = new KDMtoUML2Converter();
					URI outputURI = URI.createFileURI(ConverterImpl.this.path
							.addFileExtension("uml").toString());
					if (discoverer.getToUml().isCustom()) {
						String transformationPath = discoverer.getToUml()
								.getTransformationPath();
						URL transformation = new File(transformationPath)
								.toURI().toURL();
						converter
								.getUML2ModelFromKDMModelWithCustomTransformation(
										kdmModel.getURI(), false,
										transformation, outputURI);
					} else {
						converter.getUML2ModelFromKDMModel(kdmModel, false,
								outputURI);
					}

					/* quatrime tape */
					for (AtlTransformation atlTransformation : discoverer
							.getToUml().getChainedTransformations()) {
						System.out.println("Additional transformation for UML : "
								+ atlTransformation);
						launcher.launchUmlToUmlTransformation(atlTransformation, outputURI, kdmModelUri, outputURI);
					}
				} catch (Exception e) {
					result = Status.CANCEL_STATUS;
					e.printStackTrace();
				}

				ConverterImpl.this.logger.removeHandler(logHandler);
				logHandler.close();
				try {
					ConverterImpl.this.javaProject.getCorrespondingResource()
							.refreshLocal(1, null);
				} catch (Exception e1) {
					// give up
				}
				return result;
			}
		};
		job.addJobChangeListener(new JobChangeAdapter() {
			public void done(IJobChangeEvent event) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						try {
							IWorkspaceRoot root = ResourcesPlugin
									.getWorkspace().getRoot();
							IFile ifile = root
									.getFileForLocation(ConverterImpl.this.path
											.addFileExtension("uml"));
							if (ifile.exists()) {
								IWorkbenchPage page = PlatformUI.getWorkbench()
										.getActiveWorkbenchWindow()
										.getActivePage();
								IDE.openEditor(page, ifile);
							}
						} catch (PartInitException e) {
							e.printStackTrace();
						}
					}
				});
			}
		});
		job.schedule();
	}

}
