/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.workflow.ui.common;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.gmt.modisco.workflow.Activator;
import org.eclipse.gmt.modisco.workflow.Constants;
import org.eclipse.gmt.modisco.workflow.controller.ControllerWorkflowTransformation;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

/**
 * This class defines a composite constitute of 6 buttons (New, Edit, Remove, Load, Up, Down).
 * @author Nicolas Payneau
 */
public class AdditionalButtonsComposite extends Composite {
	private Button newButton = null;
	private Button editButton = null;
	private Button removeButton = null;
	private Button upButton = null;
	private Button downButton = null;
	private Button loadButton = null;
	private Table tableAssociee;
	private ControllerWorkflowTransformation controller;

	/**
	 * Class constructor
	 * @param parent The composite parent
	 * @param style The style of composite
	 * @param table The table of transformations
	 * @param controller The controller managing the display
	 */
	public AdditionalButtonsComposite(Composite parent, int style, Table table, ControllerWorkflowTransformation controller) {
		super(parent, style);
		this.tableAssociee = table;
		this.controller = controller;
		initialize();
	}

	/**
	 * Create the composite representing the additional buttons
	 */
	private void initialize() {
		GridData gridDataUp = new GridData();
		gridDataUp.horizontalAlignment = GridData.FILL;
		gridDataUp.verticalAlignment = GridData.CENTER;
		GridData gridDataDown = new GridData();
		gridDataDown.horizontalAlignment = GridData.FILL;
		gridDataDown.verticalAlignment = GridData.CENTER;
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.CENTER;
		this.newButton = new Button(this, SWT.NONE);
		this.newButton.setText(Messages.AdditionalButtonsComposite_0);
		this.newButton.setLayoutData(gridData);
		this.newButton.setImage(Activator.getImageDescriptor(Constants.pathIconNew).createImage());
		this.newButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			@Override
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				AdditionalButtonsComposite.this.newAdditionalMoDiscoWork();
			}
		});
		this.editButton = new Button(this, SWT.NONE);
		this.editButton.setText(Messages.AdditionalButtonsComposite_1);
		this.editButton.setLayoutData(gridData);
		this.editButton.setImage(Activator.getImageDescriptor(Constants.pathIconEdit).createImage());
		this.editButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			@Override
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				AdditionalButtonsComposite.this.editAdditionalMoDiscoWork();				
			}
		});
		this.removeButton = new Button(this, SWT.NONE);
		this.removeButton.setText(Messages.AdditionalButtonsComposite_2);
		this.removeButton.setLayoutData(gridData);
		this.removeButton.setImage(Activator.getImageDescriptor(Constants.pathIconRemove).createImage());
		this.removeButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			@Override
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				AdditionalButtonsComposite.this.removeAdditionalMoDiscoWork();
			}
		});
		this.loadButton = new Button(this, SWT.NONE);
		this.loadButton.setText(Messages.AdditionalButtonsComposite_3);
		this.loadButton.setLayoutData(gridData);
		this.loadButton.setImage(Activator.getImageDescriptor(Constants.pathIconLoad).createImage());
		this.loadButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			@Override
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				AdditionalButtonsComposite.this.loadAdditionalTransformation();
			}
		});
		this.upButton = new Button(this, SWT.PUSH);
		//\u0020\u0020 for add one space character
		this.upButton.setText(Messages.AdditionalButtonsComposite_4);
		gridDataUp.verticalIndent = 30;
		this.upButton.setLayoutData(gridDataUp);
		this.upButton.setImage(Activator.getImageDescriptor(Constants.pathIconUp).createImage());
		this.upButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			@Override
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				AdditionalButtonsComposite.this.upAdditionalMoDiscoWork();
			}
		});
		this.downButton = new Button(this, SWT.PUSH);
		this.downButton.setText(Messages.AdditionalButtonsComposite_5);
		this.downButton.setLayoutData(gridDataDown);
		this.downButton.setImage(Activator.getImageDescriptor(Constants.pathIconDown).createImage());
		this.downButton.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
			@Override
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				AdditionalButtonsComposite.this.downAdditionalMoDiscoWork();
			}
		});
		
		this.tableAssociee.addMouseListener(new MouseAdapter() {		
			@Override
			public void mouseDoubleClick(MouseEvent e) {
				AdditionalButtonsComposite.this.editAdditionalMoDiscoWork();			
			}
		});		
		GridData layoutData = new GridData();
		layoutData.verticalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		this.setSize(new Point(63, 202));
		this.setLayout(new GridLayout());
		this.setLayoutData(layoutData);
	}

	/**
	 * Add a new MoDiscoWork in the MoDiscoWorkflow
	 */
	protected final void newAdditionalMoDiscoWork() {
		this.controller.newElement();
	}
	
	/**
	 * Edit a present MoDiscoWork in the MoDiscoWorkflow
	 */
	protected final void editAdditionalMoDiscoWork() {
		if(this.tableAssociee.getSelectionIndex()!=-1){
			Constants result = this.controller.editElement(this.tableAssociee.getSelectionIndex());
			if(result.equals(Constants.LaunchConfigurationNotFound)){
				MessageDialog.openError(getShell(), Messages.AdditionalButtonsComposite_6, Messages.AdditionalButtonsComposite_7);
			}else if(result.equals(Constants.MoDiscoWorkflowSelected)){
				MessageDialog.openError(getShell(), Messages.AdditionalButtonsComposite_8, Messages.AdditionalButtonsComposite_9);
			}
		}
	}
	
	/**
	 * Remove a MoDiscoWork in the MoDiscoWorkflow
	 */
	protected final void removeAdditionalMoDiscoWork() {
		if(this.tableAssociee.getSelectionIndex()!=-1){
			RemoveDialog dialog = new RemoveDialog(getShell(),this.tableAssociee.getItem(this.tableAssociee.getSelectionIndex()).getText());
			dialog.open();
			if(dialog.getMode().equals(Constants.deleteFromLauncherMode)){
				this.controller.removeFromLauncher(getAllIndexFromString(this.tableAssociee.getItem(this.tableAssociee.getSelectionIndex()).getText()));
			}else if(dialog.getMode().equals(Constants.deleteFromWorkflowMode)){
				this.controller.removeFromWorkflow(this.tableAssociee.getSelectionIndex());
			}
		}

	}
	
	/**
	 * Allows to retrieve all items which have the same name.
	 * @parameter name Name of item
	 * @return Return a list of index item corresponding to name specified
	 */
	private List<Integer> getAllIndexFromString(String name) {
		List<Integer> indexList = new ArrayList<Integer>();
		for(TableItem item : this.tableAssociee.getItems()){
			if(item.getText().equals(name)){
				indexList.add(this.tableAssociee.indexOf(item));
			}
		}
		Collections.sort(indexList);
		Collections.reverse(indexList);
		return indexList;
	}

	/**
	 * Up the MoDiscoWork selected in the MoDiscoWorkflow
	 */
	protected final void upAdditionalMoDiscoWork() {
		if(this.tableAssociee.getSelectionIndex()>0&&this.tableAssociee.getSelectionIndex()!=-1){
			this.controller.upElement(this.tableAssociee.getSelectionIndex());
			this.controller.refresh(this.tableAssociee.getSelectionIndex()-1);
		}
	}	
	
	/**
	 * Down the MoDiscoWork selected in the MoDiscoWorkflow
	 */
	protected final void downAdditionalMoDiscoWork() {
		//+1 because the numbering begining at 0
		if((this.tableAssociee.getSelectionIndex()+1<this.tableAssociee.getItemCount())
				&&this.tableAssociee.getSelectionIndex()!=-1){
				this.controller.downElement(this.tableAssociee.getSelectionIndex());
				this.controller.refresh(this.tableAssociee.getSelectionIndex()+1);
		}
	}
	
	/**
	 * Load a MoDiscoWork in the MoDiscoWorkflow
	 */
	protected void loadAdditionalTransformation() {
		this.controller.loadElement();
	}
	
	/**
	 * Allows to create a table item 
	 * @param name Item name
	 * @param driverType Type of driver to retrieve its image
	 */
	public void createItemTable(String name, String driverType){
		TableItem item = new TableItem(this.tableAssociee,SWT.NULL);
		item.setText(name);
		item.setImage(Common.getInstance().retrieveImageDriver(driverType));	
	}
	
	/**
	 * Allows to refresh the table selection
	 */
	public void refreshTable() {
		this.tableAssociee.notifyListeners(SWT.Selection, null);
	}
	
	/**
	 * Allows to open a message dialog for the recursion
	 */
	public void dialogLoop() {
		MessageDialog.openError(getShell(), Messages.AdditionalButtonsComposite_10, Messages.AdditionalButtonsComposite_11);		
	}
	
	public Table getTable(){
		return this.tableAssociee;
	}
	
	public void setController(ControllerWorkflowTransformation controller){
		controller.setAdditionnalButton(this);
		this.controller = controller;
	}
	
	/**
	 * Allows to manage the rights of button
	 * @param allRight True if user as all the rights otherwise false
	 */
	public void enabledButtonToManageMoDiscoWorkflow(boolean allRight) {
		if(allRight){
			this.newButton.setEnabled(true);
			this.editButton.setEnabled(true);
			this.removeButton.setEnabled(true);
			this.upButton.setEnabled(true);
			this.downButton.setEnabled(true);
			this.loadButton.setEnabled(true);
		}else{
			this.newButton.setEnabled(false);
			this.editButton.setEnabled(false);
			this.upButton.setEnabled(false);
			this.downButton.setEnabled(false);
			this.loadButton.setEnabled(false);
		}
	}


	
}
