/*
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 */
package org.eclipse.gmt.mofscript.launch.ui;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.gmt.mofscript.launch.Activator;
import org.eclipse.gmt.mofscript.launch.MOFScriptLaunchConstants;
import org.eclipse.gmt.mofscript.launch.common.ControllerMOFScript;
import org.eclipse.mofscript.MOFScriptModel.MOFScriptParameter;
import org.eclipse.mofscript.MOFScriptModel.ParameterDirection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;

public class MOFScriptTab extends AbstractLaunchConfigurationTab{

	private ScrolledComposite scrollContainer;
	private Composite rootContainer;
	private Text containerMOFScriptTransforamtion;
	private Text containerOutputDirectory;
	private Label labelTrans;
	private Label labelOutput;
	private Button buttonBrowse1;
	private Button buttonBrowse;
	protected IWorkbench workbench;
	private String pathTransformation;
	private HashMap<Text,String> mapLabel;
	private HashMap<Text,String> mapParam;
	private ControllerMOFScript controller;
	private Composite compParameters;
	
	/*
	 * Class constructor
	 */
	public MOFScriptTab() {
		this.pathTransformation = "";
		this.controller = new ControllerMOFScript();
		this.controller.setTabMOFScript(this);
		this.mapLabel = new HashMap<Text,String>();
		this.mapParam = new HashMap<Text,String>();
	}
	
	/*
	 * Create user interface
	 * @parameter parent the parent of user interface
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		this.scrollContainer = new ScrolledComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL);
		this.scrollContainer.setExpandHorizontal(true);
		this.scrollContainer.setExpandVertical(true);
		
		this.rootContainer = new Composite(this.scrollContainer, SWT.NULL);
		GridLayout layout = new GridLayout();
		this.rootContainer.setLayout(layout);
		layout.numColumns = 3;
		layout.verticalSpacing = 9;
		this.scrollContainer.setContent(this.rootContainer);
		
		this.labelTrans = new Label(this.rootContainer, SWT.NULL);
		this.labelTrans.setText("MOFScript transformation");
		this.containerMOFScriptTransforamtion = new Text(this.rootContainer, SWT.BORDER | SWT.SINGLE);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		this.containerMOFScriptTransforamtion.setLayoutData(gd);
		this.mapLabel.put(this.containerMOFScriptTransforamtion,this.labelTrans.getText());
		this.containerMOFScriptTransforamtion.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				dialogChangedContainer();
			}
		});
		this.containerMOFScriptTransforamtion.setText(this.pathTransformation);
		this.buttonBrowse = new Button(this.rootContainer, SWT.PUSH);
		this.buttonBrowse.setText("Browse...");
		this.buttonBrowse.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				IFile file = handleBrowseWorkspace("Choose your output directory",MOFScriptTab.this.containerMOFScriptTransforamtion);
				if (file!=null){
					updateParameters(file);
				}
			}
		});
		this.labelOutput = new Label(this.rootContainer, SWT.NULL);
		this.labelOutput.setText("Output directory");
		this.containerOutputDirectory = new Text(this.rootContainer, SWT.BORDER | SWT.SINGLE);
		GridData gdOut = new GridData(GridData.FILL_HORIZONTAL);
		this.containerOutputDirectory.setLayoutData(gdOut);
		this.mapLabel.put(this.containerOutputDirectory,this.labelOutput.getText());
		this.containerOutputDirectory.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				dialogChangedContainer();
			}
		});
		this.buttonBrowse1 = new Button(this.rootContainer, SWT.PUSH);
		this.buttonBrowse1.setText("Browse...");
		this.buttonBrowse1.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleBrowseDirectory("Choose your output directory",MOFScriptTab.this.containerOutputDirectory);
			}
		});
		this.compParameters = new Composite(this.rootContainer,SWT.NULL);
		GridData gdParam = new GridData(GridData.FILL_HORIZONTAL);
		gdParam.horizontalSpan = 3;
		this.compParameters.setLayoutData(gdParam);
		GridLayout layoutParamter = new GridLayout();
		layoutParamter.numColumns = 3;
		layoutParamter.verticalSpacing = 9;
		this.compParameters.setLayout(layoutParamter);
		setControl(this.scrollContainer);
	}

	/*
	 * Allows to load the parameters (in,out,inout) of a transformation MOFScript
	 * @parameter message The message of group
	 * @parameter direction The direction of parameter (in, out, inout)
	 */
	private void loadParameter(String message, int direction){
		int indice = 1;
		boolean present = false;
		Group group = new Group(this.compParameters,SWT.NULL);
		group.setText(message);
		GridData gdGroup = new GridData(GridData.FILL_HORIZONTAL);
		gdGroup.horizontalSpan = 3;
		group.setLayoutData(gdGroup);
		GridLayout layoutParameter = new GridLayout();
		group.setLayout(layoutParameter);
		layoutParameter.numColumns = 3;
		layoutParameter.verticalSpacing = 9;
		if(this.controller.getMOFScriptTransformation()!=null){
			for(MOFScriptParameter parameter : this.controller.getMOFScriptTransformation().getParameters()){
				if(parameter.getDirection().getValue()== direction){
					Label labParam = new Label(group, SWT.NULL);
					labParam.setText("Model"+indice+" : "+parameter.getName());
					final Text containerParam = new Text(group, SWT.BORDER | SWT.SINGLE);
					containerParam.addModifyListener(new ModifyListener() {
						public void modifyText(ModifyEvent e) {
							dialogChangedContainer();
						}
					});
					GridData gd = new GridData(GridData.FILL_HORIZONTAL);
					containerParam.setLayoutData(gd);
					this.mapParam.put(containerParam, labParam.getText());
					Button buttonBrowse = new Button(group, SWT.PUSH);
					buttonBrowse.setText("Browse...");
					buttonBrowse.addSelectionListener(new SelectionAdapter() {
						public void widgetSelected(SelectionEvent e) {
							handleBrowseFile("Choose your model",containerParam);
						}
					});
					labParam.pack();
					containerParam.pack();
					buttonBrowse.pack();
					indice++;
					present = true;
				}
			}
			if(!present){
				group.setText("Any "+message);
			}
		}
	}
	
	/*
	 * Open a elementTreeSelectionDialog to choose a file and return this.
	 * @parameter message The message of the dialog
	 * @parameter Text The container associated with the selection of a file
	 * @return the file chooses
	 */
	private IFile handleBrowseWorkspace(String message, Text container){
		ElementTreeSelectionDialog elementTreeSelectionDialog = new ElementTreeSelectionDialog(
				getShell(), new WorkbenchLabelProvider(), new WorkbenchContentProvider());
		elementTreeSelectionDialog.setInput(ResourcesPlugin.getWorkspace().getRoot());
		elementTreeSelectionDialog.setMessage(message);
		elementTreeSelectionDialog.setAllowMultiple(false);
		elementTreeSelectionDialog.setDoubleClickSelects(true);
		elementTreeSelectionDialog.open();
		Object result = elementTreeSelectionDialog.getFirstResult();

		if ((result != null) && (result instanceof IFile)) {
			IFile currentFile = (IFile)result;
			container.setText(currentFile.getFullPath().toString());
			return currentFile;
		}
		return null;
	}
	
	/*
	 * Open a DirectoryDialog to choose a directory output
	 * @parameter message The message of the dialog
	 * @parameter Text The container associated with the selection of a directory
	 */
	private void handleBrowseDirectory(String message, Text container) {
		DirectoryDialog dialog = new DirectoryDialog(getShell());
		dialog.setText(message);
		dialog.open();
		if(!dialog.getFilterPath().equals(""))
			container.setText(dialog.getFilterPath());
	}
	
	/*
	 * Open a FileDialog to choose a file in the system
	 * @parameter message The message of the dialog
	 * @parameter Text The container associated with the selection of a file
	 */
	private void handleBrowseFile(String message, Text container) {
		FileDialog dialog = new FileDialog(getShell());
		dialog.setText(message);
		dialog.open();
		if(!dialog.getFileName().equals(""))
			container.setText(dialog.getFilterPath()+"\\"+dialog.getFileName());
	}
	
	/*
	 * Verify if all container is valid. If one of container is not valid then we update 
	 * the AbstractLaunchConfigurationTab error message
	 */
	private void dialogChangedContainer() {
		if(this.controller.getError()){
			updateStatus("Compilation Error. You can't continue");
			return;
		}
		for (Map.Entry<Text, String> path : this.mapLabel.entrySet()) {
			if (path.getKey().getText().length() == 0) {
				updateStatus(path.getValue()+" container must be specified");
				return;
			}
			if(path.getValue().equals(this.labelTrans.getText())){
				if(!validatePath(path.getKey().getText(),true)){
					updateStatus("URI invalid for "+path.getValue()+" container");
					return;
				}
				int dotLoc = path.getKey().getText().lastIndexOf('.');
				if (dotLoc != -1) {
					String ext = path.getKey().getText().substring(dotLoc + 1);
					if (ext.equalsIgnoreCase("m2t") == false) {
						updateStatus("File extension must be \"m2t\"");
						return;
					}
				}
			}else{
				if(!validatePath(path.getKey().getText(),false)){
					updateStatus("URI invalid for "+path.getValue()+" container");
					return;
				}
			}
			
		}
		for (Map.Entry<Text, String> path : this.mapParam.entrySet()) {
			if (path.getKey().getText().length() == 0) {
				updateStatus(path.getValue()+" container must be specified");
				return;
			}
			if(!validatePath(path.getKey().getText(),false)){
				updateStatus("URI invalid for "+path.getValue()+" container");
				return;
			}
		}
		updateStatus(null);
	}
	
	/*
	 * Verify if the resource is valid
	 * @parameter path The resource path
	 * @parameter workspace True if the resource is in the workspace otherwise false.
	 * return true if the resource is valid
	 */
	private boolean validatePath(String path, boolean workspace) {
		if(workspace){
			final IPath ipath = new Path(path.trim());
			IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(ipath);
			if (res == null) {
				return false;
			}else{
				if(res.getType()!=IResource.FILE){
					return false;
				}
			}
			return true;
		}else{
			File f = new File(path);
			if(f.exists()){
				return true;
			}
			return false;
		}
	}

	/*
	 * Update the AbstractLaunchConfigurationTab error message
	 * @parameter message The error message of dialog
	 */
	private void updateStatus(String message) {
		setErrorMessage(message);

		updateLaunchConfigurationDialog();
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.debug.ui.AbstractLaunchConfigurationTab#isValid(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean isValid(ILaunchConfiguration configuration) {
		boolean isValid = true;
		try{
			String mofscriptTransformation = configuration.getAttribute(MOFScriptLaunchConstants.mofScriptTransformation,"");
			String mofscriptOutput = configuration.getAttribute(MOFScriptLaunchConstants.mofScriptOutput, "");
			List<String> mofscriptParam = configuration.getAttribute(MOFScriptLaunchConstants.mofScriptParam,(List<String>)null);
			if(mofscriptTransformation.equals("")||mofscriptOutput.equals("")||mofscriptParam==null){
				isValid = false;				
			}else if(mofscriptParam.size()==0){
					isValid = false;
			}else if(!validatePath(mofscriptTransformation,true)||!validatePath(mofscriptOutput,false)){
				isValid = false;
			}else{
				for(String param : mofscriptParam){
					if(!validatePath(param,false)){
						isValid = false;
						break;
					}
				}
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return isValid;
	}
	
	/*
	 * Update the user interface for parameters of the transformation selected.
	 * @parameter file The file of the transformation
	 */
	private void updateParameters(IFile file){
		this.controller.initialization(file);
		disposeAllControlParameter();
		if(!this.controller.getError()&&file.getLocationURI()!=null){
			loadParameter(MOFScriptLaunchConstants.inParameter, ParameterDirection.IN);
			loadParameter(MOFScriptLaunchConstants.outParameter, ParameterDirection.OUT);
			loadParameter(MOFScriptLaunchConstants.inoutParameter, ParameterDirection.INOUT);
		}
		dialogChangedContainer();
		refresh();
	}
	
	/*
	 * Dispose all children of container which contains the parameters of the selected transformation
	 * Reboot the map of parameters
	 */
	private void disposeAllControlParameter(){
		for(Control control : this.compParameters.getChildren()){
			control.dispose();
		}
		this.mapParam.clear();
	}
	
	/*
	 * Refresh the container which contains the parameters of the selected transformation
	 */
	private void refresh(){
		for(Control ctrl : this.rootContainer.getChildren()){
			setControl(ctrl);
		}
		this.rootContainer.layout();
	}
	
	/*
	 * Getter of the path output
	 * @return path output of the transformation
	 */
	public String getOutputPath(){
		return this.containerOutputDirectory.getText();
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getName()
	 */
	public String getName() {
		return MOFScriptLaunchConstants.tabTitile;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	@SuppressWarnings("unchecked")
	public void initializeFrom(ILaunchConfiguration configuration) {
		try {
			this.containerMOFScriptTransforamtion.setText(configuration.getAttribute(MOFScriptLaunchConstants.mofScriptTransformation,""));
			this.containerOutputDirectory.setText(configuration.getAttribute(MOFScriptLaunchConstants.mofScriptOutput, ""));
			File f = new File(this.containerMOFScriptTransforamtion.getText());
			IWorkspace workspace= ResourcesPlugin.getWorkspace();
			IPath location= Path.fromOSString(f.getAbsolutePath());
			IFile file= workspace.getRoot().getFile(location); 
			
			List<String> listParam = configuration.getAttribute(MOFScriptLaunchConstants.mofScriptParam,(List<String>)null);
			//-1 because index of list beginning at 0
			updateParameters(file);
			if(listParam!=null){
				int ind =listParam.size()-1;
				if(ind!=-1){
					for(Text param : this.mapParam.keySet()){
						param.setText(listParam.get(ind));
						//-- because the map is reverse
						ind--;
					}
				}
			}
			dialogChangedContainer();
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		List<String> listParam = new ArrayList<String>();
		configuration.setAttribute(MOFScriptLaunchConstants.mofScriptTransformation,this.containerMOFScriptTransforamtion.getText());
		configuration.setAttribute(MOFScriptLaunchConstants.mofScriptOutput,this.containerOutputDirectory.getText());
		configuration.setAttribute(MOFScriptLaunchConstants.mofScriptParamNumber,this.mapParam.size());
		for(Text param : this.mapParam.keySet()){
			if(!param.getText().equals("")){
				listParam.add(param.getText());
			}
		}
		configuration.setAttribute(MOFScriptLaunchConstants.mofScriptParam,listParam);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#setDefaults(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {}

	/*
	 * Return a list of files corresponding to the parameters of the transformation
	 * @return a list of files corresponding to the parameters of the transformation
	 */
	public List<File> getFileParameter() {
		List<File> listFile = new ArrayList<File>();
		for (Text parameter : this.mapParam.keySet()) {
			File file = new File(parameter.getText());
			listFile.add(file);
		}
		return listFile;
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.debug.ui.AbstractLaunchConfigurationTab#getImage()
	 */
	@Override
	public Image getImage() {
		return Activator.getImageDescriptor("/icon/mofScript_logo.png").createImage();
	}

}
