package prefuse.util.display;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JSlider;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Swing widget which displays a preview image and helps select the
 * scale for an exported image.
 * 
 * @author <a href="http://jheer.org">jeffrey heer</a>
 */
public class ScaleSelector extends JComponent implements ChangeListener {

    /**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final static int MAX_SIZE = 135;
    
    private ImagePanel preview;
    private JLabel     value;
    private JLabel     size;
    private JSlider    slider;
    private Image      image;
    private int width, height;
    
    /**
     * Create a new ScaleSelector.
     */
    public ScaleSelector() {
        this.slider = new JSlider(1,10,1);
        this.value  = new JLabel("x1");
        this.size = new JLabel("   ");
        this.preview = new ImagePanel();
        
        this.value.setPreferredSize(new Dimension(25,10));
        this.size.setHorizontalAlignment(SwingConstants.CENTER);
        this.slider.setMajorTickSpacing(1);
        this.slider.setSnapToTicks(true);
        this.slider.addChangeListener(this);
        
        setLayout(new BorderLayout());
        
        Box b1 = new Box(BoxLayout.X_AXIS);
        b1.add(Box.createHorizontalStrut(5));
        b1.add(Box.createHorizontalGlue());
        b1.add(this.preview);
        b1.add(Box.createHorizontalGlue());
        b1.add(Box.createHorizontalStrut(5));
        add(b1, BorderLayout.CENTER);
        
        Box b2 = new Box(BoxLayout.X_AXIS);
        b2.add(this.slider);
        b2.add(Box.createHorizontalStrut(5));
        b2.add(this.value);
        
        Box b3 = new Box(BoxLayout.X_AXIS);
        b3.add(Box.createHorizontalStrut(5));
        b3.add(Box.createHorizontalGlue());
        b3.add(this.size);
        b3.add(Box.createHorizontalGlue());
        b3.add(Box.createHorizontalStrut(5));
        
        Box b4 = new Box(BoxLayout.Y_AXIS);
        b4.add(b2);
        b4.add(b3);
        add(b4, BorderLayout.SOUTH);
    }

    /**
     * Set the preview image.
     * @param img the preview image
     */
    public void setImage(Image img) {
        this.image = getScaledImage(img);
        stateChanged(null);
    }
    
    /**
     * Get a scaled version of the input image.
     * @param img the input image
     * @return a scaled version of the image
     */
    private Image getScaledImage(Image img) {
        int w = this.width = img.getWidth(null);
        int h = this.height = img.getHeight(null);
        double ar = ((double)w)/h;
        
        int nw = MAX_SIZE, nh = MAX_SIZE;
        if ( w > h ) {
            nh = (int)Math.round(nw/ar);
        } else {
            nw = (int)Math.round(nh*ar);
        }
        return img.getScaledInstance(nw,nh,Image.SCALE_SMOOTH);
    }
    
    /**
     * Monitor changes to the scale slider.
     * @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
     */
    public void stateChanged(ChangeEvent evt) {
        int scale = this.slider.getValue();
        this.value.setText("x"+String.valueOf(scale));
        this.size.setText("Image Size: "+(this.width*scale)+" x "+(this.height*scale)+" pixels");
        this.preview.repaint();
    }
    
    /**
     * Get the current image scale reported by the slider.
     * @return the image scale to use
     */
    public double getScale() {
        return this.slider.getValue();
    }
    
    /**
     * Swing component that draws an image scaled to the current
     * scale factor.
     */
    public class ImagePanel extends JComponent {
        /**
		 * 
		 */
		private static final long serialVersionUID = 1L;
		Dimension d = new Dimension(MAX_SIZE, MAX_SIZE);
        public ImagePanel() {
            this.setPreferredSize(this.d);
            this.setMinimumSize(this.d);
            this.setMaximumSize(this.d);
        }
        public void paintComponent(Graphics g) {
            double scale = 0.4+(0.06*getScale());
            int w = (int)Math.round(scale*ScaleSelector.this.image.getWidth(null));
            int h = (int)Math.round(scale*ScaleSelector.this.image.getHeight(null));
            Image img = (scale == 1.0 ? ScaleSelector.this.image : 
                ScaleSelector.this.image.getScaledInstance(w,h,Image.SCALE_DEFAULT));
            int x = (MAX_SIZE-w)/2;
            int y = (MAX_SIZE-h)/2;
            g.drawImage(img,x,y,null);
        }
    }
    
} // end of class ScaleSelector
