/*******************************************************************************
* Copyright  2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.core.library;

import java.io.File;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.StringTokenizer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.core.library.config.LibrarySpecification;
import org.eclipse.mtj.core.version.Version;
import org.osgi.framework.BundleContext;

/**
 * The main plugin class to be used in the desktop.
 */
public class MtjCoreLibraryPlugin extends Plugin {
	public static final String LIBRARY_TYPE_CONFIGURATION = "configuration"; //$NON-NLS-1$
	public static final String LIBRARY_TYPE_PROFILE = "profile"; //$NON-NLS-1$
	public static final String LIBRARY_TYPE_SERVICE_API = "api"; //$NON-NLS-1$	
	public static final String J2ME_LIBRARIES_ID = "libraries"; //$NON-NLS-1$
	
	// The configuration specifications provided by extension points
	private LibrarySpecification[] librarySpecs;
	
	//The shared instance.
	private static MtjCoreLibraryPlugin plugin;
	
	/**
	 * The constructor.
	 */
	public MtjCoreLibraryPlugin() {
		plugin = this;
	}

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		plugin = null;
	}

	/**
	 * Returns the shared instance.
	 */
	public static MtjCoreLibraryPlugin getDefault() {
		return plugin;
	}

	public static List<String> processFile(File path) throws Exception {
		List<String> entryList = new BasicEList();
		JarFile jarFile = new JarFile(path);
		Enumeration enumeration = jarFile.entries();	
		JarEntry jarEntry = null;
		while(enumeration.hasMoreElements()) {
			jarEntry = (JarEntry) enumeration.nextElement();
			if(jarEntry.getName().endsWith(".class"))  //$NON-NLS-1$
				entryList.add(jarEntry.getName());
		}
		return entryList;
	}

	public static boolean checkClassList(String packageString, String classListString, List<String> entryList) {		
		if( packageString == null || classListString == null )
			return false;
		if( packageString.equals("") || classListString.equals("") ) //$NON-NLS-1$ //$NON-NLS-2$
			return false;
			
		StringTokenizer st = new StringTokenizer(classListString,","); //$NON-NLS-1$
		boolean success = true;
		
		while(st.hasMoreTokens()) {
			String classString = st.nextToken();
			if(success)	
				if(!entryList.contains(packageString.replace('.','/')+"/"+classString+".class")) { //$NON-NLS-1$ //$NON-NLS-2$
					success = false;
					break;
				}
			}
		
		return success;
	}

	
	public static List<LibrarySpecification> getCorrLibrarySpecifications(File libraryFile) throws CoreException {
		LibrarySpecification[] libs = getLibrarySpecifications();
		List<LibrarySpecification> list = new BasicEList();
		
		try {
			List<String> entryList = processFile(libraryFile);
			for(LibrarySpecification lib: libs) {
				// rejecting "configuration" and "profile"
				if(!lib.getType().equals(LIBRARY_TYPE_SERVICE_API))
					continue;
				
				String allPackagesString = lib.getPackages();
				StringTokenizer st = new StringTokenizer(allPackagesString, ","); //$NON-NLS-1$
				boolean success = allPackagesString == null? false: !allPackagesString.equals(""); //$NON-NLS-1$
				
				// going through packages
				if(success) {
					while(st.hasMoreTokens()) {
						String packageString = st.nextToken();
						String packageFileList = lib.getPackageFiles(packageString);					
						success = checkClassList(packageString, packageFileList, entryList);
						if(!success)
							break;
					}
				}
				
				if(success) {
					// System.out.println("contains : "+packageString);
					list.add(lib);
					lib.setBinFile(libraryFile.getPath());
				}
			}		
		} catch (Exception e) {
			e.printStackTrace();
		}
		return list;
	}

	public static LibrarySpecification[] getLibrarySpecifications(File libraryFile, String libraryType) throws CoreException {
		LibrarySpecification[] libs = getLibrarySpecifications();
		ArrayList list = new ArrayList();
		
		try {
			List<String> entryList = processFile(libraryFile);
			for(LibrarySpecification lib: libs) {
				// rejecting "configuration" and "profile"
				if(!lib.getType().equals(libraryType))
					continue;
				
				String allPackagesString = lib.getPackages();
				StringTokenizer st = new StringTokenizer(allPackagesString, ","); //$NON-NLS-1$
				boolean success = allPackagesString == null? false: !allPackagesString.equals(""); //$NON-NLS-1$
								
				// going through packages
				if(success) {
					while(st.hasMoreTokens()) {
						String packageString = st.nextToken();
						String packageFileList = lib.getPackageFiles(packageString);
					
						success = checkClassList(packageString, packageFileList, entryList);
						if(!success)
							break;
					}
				}
				
				if(success) {
					// System.out.println("contains : "+packageString);
					list.add(lib);
					lib.setBinFile(libraryFile.getPath());
				}
			}		
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		LibrarySpecification[] ret = new LibrarySpecification[list.size()];
		for (int i = 0; i < list.size(); i++) {
			ret[i] = (LibrarySpecification)list.get(i);
		}
		                     
		return ret;
	}
	
	/**
	 * Return the configuration specifications provided by the plugin
	 * extension points.
	 * 
	 * @return the configuration specifications
	 * @throws CoreException 
	 */
	public static LibrarySpecification[] getLibrarySpecifications() 
		throws CoreException 
	{
		MtjCoreLibraryPlugin plugin = getDefault();
		if (plugin.librarySpecs == null) {
			IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(
					MtjCorePlugin.getId(), 
					J2ME_LIBRARIES_ID); 
			IConfigurationElement[] configElements = point.getConfigurationElements();
			plugin.librarySpecs = new LibrarySpecification[configElements.length];	
			
			for (int i = 0; i < configElements.length; i++) {
				plugin.librarySpecs[i] = createComponentSpecification(configElements[i]);
			}
		}		
		return plugin.librarySpecs;
	}
	
	/**
	 * Create a component specification for the specified configuration
	 * element
	 * @param element
	 * @return
	 * @throws CoreException 
	 */
	private static LibrarySpecification createComponentSpecification(IConfigurationElement element) 
		throws CoreException 
	{
		String id = element.getAttribute("id"); //$NON-NLS-1$
		String type = element.getAttribute("type"); //$NON-NLS-1$
		String name = element.getAttribute("name"); //$NON-NLS-1$
		String label = element.getAttribute("label"); //$NON-NLS-1$
		String versionString = element.getAttribute("version"); //$NON-NLS-1$
		Version version = new Version(versionString);
		String binFile = element.getAttribute("binFile"); //$NON-NLS-1$
		String classEntry = element.getAttribute("classEntry"); //$NON-NLS-1$
		String sourceFile = element.getAttribute("sourceFile"); //$NON-NLS-1$
		String javadocFile = element.getAttribute("javadocFile"); //$NON-NLS-1$
		String fileNameList = element.getAttribute("fileNameList"); //$NON-NLS-1$
		String packagesList = element.getAttribute("packages"); //$NON-NLS-1$

		LibrarySpecification specification = new LibrarySpecification();
		specification.setIdentifier(id);
		specification.setType(type);
		specification.setName(name);
		specification.setLabel(label);
		specification.setVersion(version);
		specification.setClassEntry(classEntry);
		specification.setBinFile(binFile);
		specification.setSourceFile(sourceFile);
		specification.setJavadocFile(javadocFile);
		specification.setFileNameList(fileNameList);
		specification.setPackages(packagesList);

		StringTokenizer packagest = new StringTokenizer(packagesList,","); //$NON-NLS-1$
		while(packagest.hasMoreTokens()) {
			String packageString = packagest.nextToken();
			String fileListString = element.getAttribute(packageString);
			specification.setPackage(packageString, fileListString);
		}
	
		return specification;
	}
}


