/*******************************************************************************
* Copyright (c) 2005,2006 Nokia Corporation and others
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
 * Additional Contributors -
 *  	Kevin Horowitz (IBM Corp) - Update javadoc
*******************************************************************************/
package org.eclipse.mtj.api.extension;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.signings.SigningCertificate;
import org.eclipse.mtj.exception.MtjException;

/**
 * The Signing Provider provides the interface to sign a project's deployment package.
 * <br>
 */
public interface SigningProvider extends BuildExtension {

	/**
	 * Returns whether the SingingProvider plug-in can provide 
	 * root certificate management features.
	 * @return true if root certificate management features are available, false otherwise
	 * @throws MtjException
	 */
	public boolean isRootCertToolAvailable() throws MtjException;
	
	/**
	 * Attempt to import keystore element (alias) into rootcert. Uses password to access keystore
	 * @param rootcert
	 * @param keystore
	 * @param alias
	 * @param password
	 * @throws MtjException
	 */
	public void importToRootCert(String rootcert, String keystore, String alias, String password)throws MtjException;

	/**
	 * Returns string array populated with the contents of the rootcert.
	 * @param rootcert
	 * @return
	 * @throws MtjException
	 */
	public String [] listrootcert(String rootcert) throws MtjException;

	/**
	 * Attempts to remove an entry from rootcert. References target entry by index (removeindex).
	 * @param rootcert
	 * @param removeindex
	 * @throws MtjException
	 */
	public void removeCertFromRoot(String rootcert, int removeindex) throws MtjException;
	
	/**
	 * Returns the deployment types that are supported.
	 * @return DeploymentType[]
	 */
	public DeploymentType[] getSupportedTypes() throws MtjException;

	/**
	 * Method signs the project with the certificates.
	 * 
	 * @param project
	 * @param targetFolder
	 * @param certificates
	 * @param monitor
	 * @return 
	 * @throws MtjException
	 */
	public boolean sign(IMtjProject project, IFolder targetFolder, SigningCertificate certificates, IProgressMonitor monitor) throws MtjException;
		
	/**
	 * Method unsigns the project.
	 * 
	 * @param mtjProject
	 * @param targetFolder
	 * @param monitor
	 * @return
	 * @throws MtjException
	 */
	public boolean unsign(IMtjProject mtjProject, IFolder targetFolder, IProgressMonitor monitor) throws MtjException;
	 
	/**
	 * Method determines and reports if deployment is signed.
	 *  
	 * @param mtjProject
	 * @param targetFolder
	 * @param monitor
	 * @return
	 * @throws MtjException
	 */
	public boolean isSigned(IMtjProject mtjProject, IFolder targetFolder, IProgressMonitor monitor) throws MtjException;
	 	 
	 /**
	 * Implementations that rely on an external security tool,
	 * are responsible for persistant storage of the tool location value. </br>
	 * This method should return the location of the tool. Should never return null.
	 * @param monitor
	 * @return
	 * @throws MtjException
	 */
	public String getToolLocation(IProgressMonitor monitor) throws MtjException;
		
	 /**
	 * Implementations that rely on an external security tool,
	 * are responsible for persistant storage of the tool location value.  This method should contain
	 * the code to store the location.
	 * @param loc
	 * @param monitor
	 * @throws MtjException
	 */
	public void storeToolLocation(String loc, IProgressMonitor monitor) throws MtjException;
		
}
